/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore;

import java.io.Serializable;

import jp.sourceforge.dvibrowser.dvicore.util.Canonicalizer;
import jp.sourceforge.dvibrowser.dvicore.util.SimpleCanonicalizer;


// immutable.

public final class DviFontName
implements Serializable
{
  private static final long serialVersionUID = 5473736840728986859L;
  private final int al, nl;
  private final String name;
  private final int hash;

  private DviFontName(int al, int nl, String name)
  {
    if (name == null)
      throw new NullPointerException
        ("name can't be null");
    if (al < 0 || 255 < al)
      throw new IllegalArgumentException
        ("illegal value of area length: " + al);
    if (nl < 0 || 255 < nl)
      throw new IllegalArgumentException
        ("illegal value of name length: " + nl);
    if (name.length() != (al + nl))
      throw new IllegalArgumentException
        ("name size mismatch:"
        + " name.length()=" + name.length()
        + " al=" + al
        + " nl=" + nl
        );

    this.al = al;
    this.nl = nl;
    this.name = name;
    this.hash = al + 33*(nl + 33*name.hashCode());
  }

  public int areaLength() { return al; }
  public int nameLength() { return nl; }
  public String name() { return name; }

  private static final Canonicalizer<DviFontName> canonicalizer
    = new SimpleCanonicalizer<DviFontName>();

  public static DviFontName getInstance(int al, int nl, String name)
  {
    return canonicalizer.canonicalize(
      new DviFontName(al, nl, name)
    );
  }
  public static DviFontName getInstance(int nl, String name)
  {
    return getInstance(0, nl, name);
  }
  public static DviFontName getInstance(String name)
  {
    return getInstance(0, name.length(), name);
  }

  private volatile String string = null;
  public String toString() {
    if (string == null) {
      string = getClass().getName()
        + "[al=" + al
        + ",nl=" + nl
        + ",name=" + name
        + "]";
    }
    return string;
  }

  public boolean equals(Object obj) {
    if (obj == this) return true;
    if (obj instanceof DviFontName) {
      DviFontName fn = (DviFontName) obj;
      return name.equals(fn.name) && al == fn.al && nl == fn.nl;
    }
    return false;
  }

  public int hashCode() {
    return hash;
  }
}
