/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is bbs2chreader.
 *
 * The Initial Developer of the Original Code is
 * flyson.
 * Portions created by the Initial Developer are Copyright (C) 2008
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *    flyson <flyson at users.sourceforge.jp>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */


function b2rStorageThread(){
	this._init();
}

b2rStorageThread.prototype = {
	_init: function(){
		var database = gStorageService.database;
		database.beginTransaction();
		try{
			if(!database.tableExists("thread_data")){
				let sql = <>
					CREATE TABLE thread_data(
					    board_id           TEXT NOT NULL,
					    url                TEXT NOT NULL,
					    dat_id             TEXT NOT NULL,
					    title              TEXT,
					    title_n            TEXT,
					    line_count         INTEGER DEFAULT 0,
					    read_position      INTEGER DEFAULT 0,
					    http_last_modified TEXT,
					    maru_getted        INTEGER DEFAULT 0,
					    stored             INTEGER DEFAULT 0,
					    post_name          TEXT,
					    post_mail          TEXT,
					    rate               INTEGER DEFAULT 0
					);
				</>.toString().replace(/\t/g, "");
				database.executeSimpleSQL(sql);
				database.executeSimpleSQL("CREATE INDEX IF NOT EXISTS thread_data_board_id_index ON thread_data(board_id);");
				database.executeSimpleSQL("CREATE INDEX IF NOT EXISTS thread_data_dat_id_index ON thread_data(dat_id);");
			}
		}catch(ex){
			Components.utils.reportError(ex);
		}finally{
			database.commitTransaction();
		}

		this._statements = new Array();
		this._statements["setThreadData_select"] = database.createStatement(
				"SELECT _rowid_ FROM thread_data WHERE board_id=?1 AND dat_id=?2;");
		this._statements["setThreadData_update"] = database.createStatement(
				"UPDATE thread_data SET url=?1, line_count=?2, http_last_modified=?3, maru_getted=?4 WHERE _rowid_=?5;");
		this._statements["setThreadData_insert"] = database.createStatement(
				"INSERT INTO thread_data(board_id, url, dat_id, title, title_n, line_count, http_last_modified, maru_getted) VALUES(?1,?2,?3,?4,?5,?6,?7,?8);");

		this._statements["getThreadData_select"] = database.createStatement(
				"SELECT board_id, url, dat_id, title, title_n, line_count, read_position, http_last_modified, "
				+ "maru_getted, stored, post_name, post_mail, rate FROM thread_data WHERE board_id=?1 AND dat_id=?2");

		this._statements["deleteThreadData_delete"] = database.createStatement(
				"DELETE FROM thread_data WHERE board_id=?1 AND dat_id=?2");
	},


	setThreadData: function(aThreadURL, aBoardURL, aDatID, aTitle, aLineCount, aLastModified, aMaruGetted){
		var b2rService = XPC.getService("@bbs2ch.sourceforge.jp/b2r-global-service;1", "b2rIGlobalService");
		var boardID = b2rService.threadUtils.getBoardID(aBoardURL);

		var database = gStorageService.database;
		database.beginTransaction();
		try{
			var statement;
			statement = this._statements["setThreadData_select"];
			statement.bindStringParameter(0, boardID);
			statement.bindStringParameter(1, aDatID);
			var threadID = 0;
			if(statement.executeStep()){
				threadID = statement.getInt64(0);
			}
			statement.reset();
			if(threadID){
				statement = this._statements["setThreadData_update"];
				statement.bindStringParameter(0, aThreadURL.spec);
				statement.bindInt32Parameter(1, aLineCount);
				statement.bindStringParameter(2, aLastModified);
				statement.bindInt32Parameter(3, aMaruGetted ? 1 : 0);
				statement.bindInt64Parameter(4, threadID);
				statement.execute();
			}else{
				statement = this._statements["setThreadData_insert"];
				statement.bindStringParameter(0, boardID);
				statement.bindStringParameter(1, aThreadURL.spec);
				statement.bindStringParameter(2, aDatID);
				statement.bindStringParameter(3, aTitle);
				statement.bindStringParameter(4, gStorageService._getNormarizedString(aTitle));
				statement.bindInt32Parameter(5, aLineCount);
				statement.bindStringParameter(6, aLastModified);
				statement.bindInt32Parameter(7, aMaruGetted ? 1 : 0);
				statement.execute();
			}
		}catch(ex){
			Components.utils.reportError(ex);
		}finally{
			database.commitTransaction();
		}
	},


	getThreadData: function(aBoardURL, aDatID){
		var b2rService = XPC.getService("@bbs2ch.sourceforge.jp/b2r-global-service;1", "b2rIGlobalService");
		var boardID = b2rService.threadUtils.getBoardID(aBoardURL);
		var statement = this._statements["getThreadData_select"];
		statement.bindStringParameter(0, boardID);
		statement.bindStringParameter(1, aDatID);

		var result = null;
		gStorageService.database.beginTransaction();
		try{
			if(statement.executeStep()){
				result = new b2rThreadData(
					statement.getString(0),  // boardID
					statement.getString(1),  // url
					statement.getString(2),  // datID
					statement.getString(3),  // title
					statement.getString(4),  // titleN
					statement.getInt32(5),   // lineCount
					statement.getInt32(6),   // readPosition
					statement.getString(7),  // httpLastModified
					statement.getInt32(8),   // maruGetted
					statement.getInt32(9),   // stored
					statement.getString(10), // postName
					statement.getString(11), // postMail
					statement.getInt32(12)   // rate
				);
			}
		}catch(ex){
			Components.utils.reportError(ex);
		}finally{
			statement.reset();
			gStorageService.database.commitTransaction();
		}

		return result;
	},

	deleteThreadData: function(aBoardURL, aDatID, aRemoveDatFile){
		var b2rService = XPC.getService("@bbs2ch.sourceforge.jp/b2r-global-service;1", "b2rIGlobalService");
		var boardID = b2rService.threadUtils.getBoardID(aBoardURL);
		var statement = this._statements["deleteThreadData_delete"];
		statement.bindStringParameter(0, boardID);
		statement.bindStringParameter(1, aDatID);
		gStorageService.database.beginTransaction();
		try{
			statement.execute();
		}catch(ex){
			Components.utils.reportError(ex);
		}finally{
			gStorageService.database.commitTransaction();
		}

		if(aRemoveDatFile){
				// OfBNg .dat t@C
			var datFile = b2rService.io.getLogFileAtURL(aBoardURL.resolve(aDatID + ".dat"));
			try{
				if(datFile.exists()) datFile.remove(false);
			}catch(ex){
				Components.utils.reportError(ex);
			}
		}

	}

};


function b2rThreadData(boardID, url, datID, title, titleN, lineCount, readPosition, httpLastModified,
				maruGetted, stored, postName, postMail, rate){

	this._boardID = boardID;
	this._url = url;
	this._datID = datID;
	this._title = title;
	this._titleN = titleN;
	this._lineCount = lineCount;
	this._readPosition = readPosition;
	this._httpLastModified = httpLastModified;
	this._maruGetted = Boolean(maruGetted==1);
	this._stored = Boolean(stored==1);
	this._postName = postName;
	this._postMail = postMail;
	this._rate = rate;
}

b2rThreadData.prototype = {

	get boardID() {
		return this._boardID;
	},
	get url() {
		if(this._url){
			var ioService = Components.classes["@mozilla.org/network/io-service;1"]
					.getService(Components.interfaces.nsIIOService);
			return ioService.newURI(this._url, null, null).QueryInterface(Components.interfaces.nsIURL);
		}
		return null;
	},
	get datID() {
		return this._datID;
	},
	get title() {
		return this._title;
	},
	get titleN() {
		return this._titleN;
	},
	get lineCount() {
		return this._lineCount;
	},
	get readPosition() {
		return this._readPosition;
	},
	get httpLastModified() {
		return this._httpLastModified;
	},
	get maruGetted() {
		return this._maruGetted;
	},
	get stored() {
		return this._stored;
	},
	get postName() {
		return this._postName;
	},
	get postMail() {
		return this._postMail;
	},
	get rate() {
		return this._rate;
	},

	// ********** ********* implements nsISupports ********** **********

	QueryInterface: function(aIID){
		if(aIID.equals(Ci.b2rIThreadData)) return this;
		if(aIID.equals(Ci.nsISupports)) return this;

		throw Components.results.NS_ERROR_NO_INTERFACE;
	}
};