package cerot.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;

/**
 * リソースの読み取りを行うユーティリティクラス。
 * @author cero-t
 * 
 */
public final class ResourceUtils {
	/** デフォルトのエンコード */
	private static final String DEFAULT_CHARSET = "UTF-8";

	/** バッファサイズ */
	private static final int BUFFER_SIZE = 4096;

	/**
	 * リソースを読み込み、文字列として返します。
	 * @param resourceName リソース名
	 * @return リソースの内容
	 * @throws IOException 入出力エラーが発生した場合
	 */
	public static String readResourceToString(String resourceName)
			throws IOException {
		return readResourceToString(resourceName, DEFAULT_CHARSET);
	}

	/**
	 * リソースを読み込み、文字列として返します。
	 * @param resourceName リソース名
	 * @param charset 文字セット名
	 * @return リソースの内容
	 * @throws IOException 入出力エラーが発生した場合
	 */
	public static String readResourceToString(String resourceName,
			String charset) throws IOException {
		InputStream is = ResourceUtils.class.getResourceAsStream(resourceName);
		InputStreamReader reader = null;
		try {
			reader = new InputStreamReader(is, DEFAULT_CHARSET);
			StringBuilder builder = new StringBuilder();
			char[] chars = new char[BUFFER_SIZE];
			int count = 0;
			while ((count = reader.read(chars)) != -1) {
				builder.append(chars, 0, count);
			}

			return new String(builder);
		} finally {
			if (reader != null) {
				reader.close();
			} else if (is != null) {
				is.close();
			}
		}
	}

	/**
	 * リソースを読み込み、文字列のリストとして返します。
	 * @param resourceName リソース名
	 * @return リソースの内容
	 * @throws IOException 入出力エラーが発生した場合
	 */
	public static List<String> readResourceToList(String resourceName)
			throws IOException {
		return readResourceToList(resourceName, DEFAULT_CHARSET);
	}

	/**
	 * リソースを読み込み、文字列のリストとして返します。
	 * @param resourceName リソース名
	 * @param charset 文字セット名
	 * @return リソースの内容
	 * @throws IOException 入出力エラーが発生した場合
	 */
	public static List<String> readResourceToList(String resourceName,
			String charset) throws IOException {
		List<String> result = new ArrayList<String>();
		InputStream is = ResourceUtils.class.getResourceAsStream(resourceName);
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new InputStreamReader(is,
					DEFAULT_CHARSET));
			String line = null;
			while ((line = reader.readLine()) != null) {
				result.add(line);
			}
		} finally {
			if (reader != null) {
				reader.close();
			} else if (is != null) {
				is.close();
			}
		}

		return result;
	}

	/**
	 * プライベートコンストラクタ。呼び出し禁止。
	 */
	private ResourceUtils() {
		// Do Nothing.
	}
}
