require 'common'

logging = { level = 10 }

-------------------------------------------
degree = 7
dt = 1.5e-5
--cfl = 0.5
--track_name = './track/'
-------------------------------------------

-- global simulation options
simulation_name='ateles_left'
sim_control = {
  time_control = {
    min = 0,
    max = { iter = 11010 },
    interval = { iter = 1 }
  }
}

-- table for preCICE
-- When using RK2, we also have the possibility to exchange data during the
-- intermdeiate timestep, therefore the option "use_RK2_inter" should be set
-- to true.
precice = {
  accessor = 'Ateles_left',
  configFile ='precice_config.xml',
  --use_RK2_inter = true,
}
-- Restart settings
restart = {
  --read = './restart/left/ateles_left_lastHeader.lua',
  write = './restart/left/',
  time_control = {
    min = 0,
    max = { iter = 11000 },
    interval = { iter = 1000 }
  },
}

-- For the spacetime-function kind=precice, there exist now several option for
-- the interpolation. When setting:
-- Make use of equidistant points:
-- 1. use_EQ_points = true
-- Set how many points should be created for write to precice, by default this
-- variable is set to 1, thus the number of points is equal to the scheme order.
-- The factor is always multiplied by the scheme order, which results in the
-- total number of points per face.
-- 2. factor_EQ_points = 1
-- Using Nearest-Projection with nonequidistant points:
-- 3. use_NP = true
-- Using equidistant points and Nearest-Projection:
-- 4. use_EQ_Points = true
--    use_NP = true
--    factor_EQ_points = <number>
-- REMEMBER: With the new implementation two meshIDs are needed for preCICE, thus
-- one for read and one for write.
-- 5. If nothing is set (every logical statement is false), just the
-- non-equidistant points are considered


characteristic = 0.0
relax_temperature = 0.0
relax_velocity = {0.0, 0.0}

-- Variable system definintion--
variable = {
  {
    name = 'characteristic',
    ncomponents = 1,
    vartype = "st_fun",
    st_fun = characteristic
  },
  {
    name = 'relax_velocity',
    ncomponents = 2,
    vartype = "st_fun",
    st_fun = relax_velocity
  },
  {
    name = 'relax_temperature',
    ncomponents = 1,
    vartype = "st_fun",
    st_fun = relax_temperature
  },
  {
    name = 'ref_density',
    ncomponents = 1,
    vartype ='st_fun',
    st_fun = gauss
  },
  {
    name = 'error',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'difference',
      input_varname = {'density', 'ref_density'}
    }
  },
  {
    name = 'coupling',
    ncomponents = 4,
    vartype = 'st_fun',
    st_fun = {
      predefined = 'precice',
      use_EQ_points= true,
      use_NP=true,
      factor_EQ_points = 3,
      precice_meshWrite = 'leftSurface_Write',
          write_varname = { 'Density_left',
                            'MomX_left',
                            'MomY_left',
                            'Energy_left'
      },
      precice_meshRead = 'leftSurface_Read',
          read_varname = { 'Density_right',
                           'MomX_right',
                           'MomY_right',
                           'Energy_right'
      },
    }
  },
  -- write to precice
  {
    name = 'Density_left',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'extract',
      input_varname = 'density',
      input_varindex = { 1 }
    }
  },
  {
    name = 'MomX_left',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'extract',
      input_varname = 'momentum',
      input_varindex = { 1 }
    }
  },
  {
    name = 'MomY_left',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'extract',
      input_varname = 'momentum',
      input_varindex = { 2 }
    }
  },
  {
    name = 'Energy_left',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'extract',
      input_varname = 'energy',
      input_varindex = { 1 }
    }
  },
  -- read from precice
  {
    name = 'Density_right',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'extract',
      input_varname = 'coupling',
      input_varindex = { 1 }
    }
  },
  {
    name = 'MomX_right',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'extract',
      input_varname = 'coupling',
      input_varindex = { 2 }
    }
  },
  {
    name = 'MomY_right',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'extract',
      input_varname = 'coupling',
      input_varindex = { 3 }
    }
  },
  {
    name = 'Energy_right',
    ncomponents = 1,
    vartype = 'operation',
    operation = {
      kind = 'extract',
      input_varname = 'coupling',
      input_varindex = { 4 }
    }
  }
}

-- Mesh definitions --
mesh = '../mesh_left/'


-- timing settings (i.e. output for performance measurements, this table is otional)
timing_file = 'timing_left.res'         -- the filename of the timing results

-- Equation definitions --
equation = {
  name = 'euler_2d',
  therm_cond = 2.555e-02,
  isen_coef = 1.4,
  r = 296.0,
  material = {
    characteristic = 'characteristic',
    relax_velocity = 'relax_velocity',
    relax_temperature = 'relax_temperature'
  }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
  -- the spatial discretization scheme
  spatial =  {
    name = 'modg_2d',
    modg_space = 'Q',
    m = degree,
  },
  --stabilization = {
  --  name = 'spectral_viscosity',
  --  alpha = 36,
  --  order = filter_order,
  --},
  -- the temporal discretization scheme
  temporal = {
    name = 'explicitSSPRungeKutta',
    steps = 2,
    control = {
      name = 'fixed',
      dt = dt
    },
  },
}

-- Initial condition
initial_condition = {
  density = ic_gauss,
  pressure = press,
  velocityX = velocityX,
  velocityY = 0.0,
}

 -- Boundary definitions
boundary_condition = {
  {
    label = 'west',
    kind = 'primitives',
    density = dens,
    velocityX = velocityX,
    velocityY = 0.0,
    pressure = press
  }
  --{
  --  label = 'west',
  --  kind = 'inflow',
  --  density = dens,
  --  velocityX = velocityX,
  --  velocityY = 0.0,
  --}
 ,
  {
    label = 'east',
    kind = 'conservatives',
    density = 'Density_right',
    momentumX = 'MomX_right',
    momentumY = 'MomY_right',
    energy = 'Energy_right',
  }
  ,
  {
    label = 'north',
    kind = 'primitives',
    density = dens,
    velocityX = velocityX,
    velocityY = 0.0,
    pressure = press,
  }
  ,
  {
    label = 'south',
    kind = 'primitives',
    density = dens,
    velocityX = velocityX,
    velocityY = 0.0,
    pressure = press,
  }
  ,
  {
    label = 'top',
    kind = 'primitives',
    density = dens,
    velocityX = velocityX,
    velocityY = 0.0,
    pressure = press,
  }
  ,
  {
    label = 'bottom',
    kind = 'primitives',
    density = dens,
    velocityX = velocityX,
    velocityY = 0.0,
    pressure = press,
  }
}
