-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Euler equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of Gaussian pulse in density. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- This testcase can be run in serial (only one execution unit) or in parallel (with multiple mpi ranks). 
-- To specify the number of ranks please modify nprocs variable. To calculate a grid convergence behavior please modify the 
-- level variable. An increment of one will half the radius of your elements.

-- global simulation options
simulation_name = 'ateles_comp' -- the name of the simualtion
sim_control = {
                time_control = {   
                  min = 0, 
                  max = 0.008, -- final simulation time
                  interval = {iter = 1} -- The interval is used to define the
                }
              }

-- Mesh definitions --
mesh = 'mesh_comp/'

---- Restart settings
estart = { 
--            -- file to restart from
--            read = './restart/maxwell/per_osci_maxwell_modg_lastHeader.lua',                        
--            -- folder to write restart data to
            write = './restart/',                                        
            -- temporal definition of restart write
            time = {   
                      useIterations = true,  
                      min = 0, 
                      max = sim_control.time_control.max, 
                      interval = 10,
                   },  
          }

-- timing settings (i.e. output for performance measurements, this table is otional)
timing = {
          folder = './',                  -- the folder for the timing results
          filename = 'timing_compl.res'         -- the filename of the timing results
         }

-- Equation definitions --
equation = {
    name   = 'euler',
    therm_cond = 2.555e-02,
    isen_coef = 1.4,
    r      = 296.0,
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg',            -- we use the modal discontinuous Galerkin scheme 
               modg_space = 'Q',        -- the polynomial space Q or P
               m = 7,                   -- the maximal polynomial degree for each spatial direction
               }, 
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitRungeKutta',  --'explicitEuler',
               steps = 4,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.8,     -- CourantFriedrichsLewy number
                         },
               },
}

function dens(x,y,z)
  return x
end

projection = {
              kind = 'l2p',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattoPoints = false  -- if lobatto points should be used, default = false
              factor = 2.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
           -- blocksize = 32,        -- for fpt, default -1
           -- fftMultiThread = false -- for fpt, logical, default false
             }

-- This is a very simple example to define constant boundary condtions.
-- Transport velocity of the pulse in x direction.
velocityX = 250
initial_condition ={  density = 1.225,
                      pressure = {
                                    predefined='gausspulse',
                                    center={-1.0, 0.0, 0.0},
                                    halfwidth=0.2,
                                    amplitude=2.0,
                                    background=10000
                                 },
                      velocityX = velocityX,
                      velocityY = 0.0,
                      velocityZ = 0.0,
                    }


-- Tracking              
racking = {
             label = 'track_momentum_left',
             folder = './',
             variable = {'momentum'},
             shape = {kind = 'canoND', object= { origin ={0., 0., 0.} } },
             time = {
                      useIterations = false,
                      min = 0,
                      max = sim_control.time_control.max,
                      interval = sim_control.time_control.max/10.0,
                    },
             format = 'ascii'
           }

 -- Boundary definitions
 boundary_condition = {
                         {
                           label = 'wall_1',
                           kind = 'outflow',
                           pressure = 100000,
                           density = 'extrapolate'
                         }
                         ,
                         {
                           label = 'wall_2',
                           kind = 'outflow',
                           pressure = 100000,
                           density = 'extrapolate'
                         }
                         ,
                         {
                           label = 'wall_3',
                             kind = 'inflow',
                             pressure = 'extrapolate',
                             density = 1.225,
                             v_x = velocityX,
                             v_y = 0.0,
                             v_z = 0.0
                         }
                         ,
                         {
                           label = 'wall_4',
                           kind = 'outflow',
                           pressure = 100000,
                           density = 'extrapolate'
                         }
                         ,
                         {
                           label = 'wall_5',
                           kind = 'outflow',
                           pressure = 100000,
                           density = 'extrapolate'
                         }
                         ,
                         {
                           label = 'wall_6',
                           kind = 'outflow',
                           pressure = 100000,
                           density = 'extrapolate'
                         }
                         ,
                       }
