-- Simple setup to test sponge layer in linearized Euler equations.
-- It uses a 1D gaussian pulse in pressure, with a line of elements
-- and a zero_gradient boundary on the left and sponge with Dirichlet
-- zero boundaries on the right.
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
degree = 5
poly_space = 'Q'
gamma = 1.4
pressure = 1
speed_of_sound = 1

logging = { level = 2 }


--...Configuration of simulation time
sim_control = { 
  time_control = {
    max      = 6.0/speed_of_sound,
    min      = 0.0, 
    interval = {iter = 10} 
  }
}
--------------------------------------------------------------------------------
-- Definition of the test-case.

-- Mesh definitions --
mesh = 'mesh_'

variable = {
  { name = "spongelayer_var", -- >>>>>>>>>>>>>
    ncomponents = 6,
    vartype = "st_fun",
    st_fun = {
      predefined = 'combined',
      spatial = {
        predefined = 'spongelayer',
        plane_origin = {1.0, 0, 0},
        plane_normal = {1.0-1/(degree+1), 0, 0},
        damp_factor = 4*speed_of_sound,
        damp_exponent = 1,
        target_state = {
          density   = 0.0,
          velocityX = 0.0,
          velocityY = 0.0,
          velocityZ = 0.0,
          pressure =  0.0
        }
      },
      --temporal = {1},
      shape = {
        kind = 'canoND',
        object= {
          origin = { 1.000001, -0.000001, -0.000001 },
          vec = {
            {1,0,0},
            {0,1.000002,0},
            {0,0,1.000002}
          },
          segments = {5,5,5}
        }
      }
    }
  } -- <<<<<<<<<<< spongelayer_var
}

-- Tracking              
tracking = {
  {
    label = 'press_east',
    folder = 'track_ml_',
    variable = {'pressure'},
    shape = { kind = 'canoND',
              object= { origin = {1.9,0.0,0.0} }
    },
    time_control = {
      max = sim_control.time_control.max,
      min = 0,
      interval = {iter=1},
    },
    output = { format = 'ascii', use_get_point = true }
  },
  {
    label = 'press',
    folder = 'vis_ml_',
    variable = {'pressure'},
    shape = { kind = 'global'},
    time_control = {
      max = sim_control.time_control.max,
      min = 0,
      interval = 0.2/speed_of_sound,
    },
    output = { format = 'vtk'}
  }
}
ply_sampling = {
  nlevels = 4,
  adaptiveDofReduction = true,
  tolerance = 0.001
}

-- Equation definitions --
density = gamma*pressure/speed_of_sound^2
equation = {
  name      = 'linearEuler',
  isen_coef = 1.4,
  background = {
    density = density,
    velocityX = 0.0,
    velocityY = 0.0,
    velocityZ = 0.0,
    pressure = pressure
  }
}

source = {
  -- Outlet sponge
  spongelayer = "spongelayer_var"
}


-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg',
               m =  degree,
               modg_space = poly_space
               }, 
    -- the temporal discretization scheme
    temporal = {
                 name = 'explicitRungeKutta', 
                 steps = 4,
              -- how to control the timestep
                 control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.95,     -- CourantÐFriedrichsÐLewy number
                         },
               },
}

ampl_pressure = 0.01*equation.background.pressure

function ic_gauss_pressure(x,y,z)
  r = (x-0.5)/0.25
  return( ampl_pressure * math.exp(r*r*math.log(0.5)) )
end

-- Initial Condition definitions --
initial_condition = { density   = 0.0,
                      velocityX = 0.0,
                      velocityY = 0.0,
                      velocityZ = 0.0,
                      pressure  = ic_gauss_pressure
                    }

-- Boundary conditions
boundary_condition = {
  {
    label = 'west',
    kind = 'zero_gradient',
    enforce_zero_grad = true
  },
  {
    label = 'east',
    kind = 'primitives',
    density = 0.0,
    velocityX = 0.0,
    velocityY = 0.0,
    velocityZ = 0.0,
    pressure = 0.0
  }
}

-- ...the general projection table
projection = {
  kind   = 'fpt',
  factor = 1.0
}
