-- Configuration for the acoustic pulse.
--
-- The setup simulates an acoustic wave by an initial Gaussian pulse in
-- pressure at the origin.
--
-- Setup taken from Tam: Computational Aeroacoustics, Appendix G.3.
--------------------------------------------------------------------------------

-- Parameters to vary --
degree = 3
poly_space = 'Q'
meshlevel = 6
--------------------------------------------------------------------------------

-- Equation definitions --
--  * Ratio of specific heats
gamma = 1.4
equation = {
             name   = 'linearEuler',
             isen_coef = gamma,
             background = {
             -- Mean field values (see Tam, implementation p. 441)
                 density = 1.0,
                 velocityX = 0.0,
                 velocityY = 0.0,
                 velocityZ = 0.0,
                 pressure = 1.0/gamma 
             }
}

-- Setup (gaussian pulse parameters)
gp_center = {0.0, 0.0, 0.0}
gp_halfwidth = 3.0
gp_amplitude = 0.001
gp_background = 0.0
gp_c = math.sqrt( gamma * equation.background.pressure
                        / equation.background.density )

variable = {
    name = 'exact_p',
    ncomponents = 1,
    vartype = 'st_fun',
    st_fun = {
        predefined = 'acoustic_pulse',
        center         = gp_center,
        halfwidth      = gp_halfwidth,
        amplitude      = gp_amplitude,
        background     = gp_background,
        speed_of_sound = gp_c
    }
}
--------------------------------------------------------------------------------


-- Initial Condition --
initial_condition = { density = 0.0,
                      velocityX = 0.0,
                      velocityY = 0.0,
                      velocityZ = 0.0,
                      pressure = { predefined = 'gausspulse',
                                   center     = gp_center,
                                   halfwidth  = gp_halfwidth,
                                   amplitude  = gp_amplitude,
                                   background = gp_background
                      }
}

-- Configuration of simulation time
sim_control = { time_control = { max = 35.0, -- final Simulated time
                                 min = 0.0, 
                                 interval = {iter = 1} 
                }
}

--------------------------------------------------------------------------------

logging = { level = 2 }

-- Check for Nans and unphysical values
check =  { interval = 1 }

-- ...the general projection table
projection = {
    kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
    -- for fpt the  nodes are automatically 'chebyshev'
    -- for l2p the  nodes are automatically 'gauss-legendre'
    factor = 1.0   -- dealising factor
    -- blocksize = 32,
}

rest_prefix = 'restart_'..poly_space..degree..'_L'..meshlevel..'_'

--...Configuration for the restart file
restart = { 
            -- file to restart from (if existing)
            read = rest_prefix..'simulation_lastHeader.lua',
            write = rest_prefix,
            init_on_missing = true,
            -- temporal definition of restart write
            time_control = { max = sim_control.time_control.max,
                             min = 0.0,
                             interval = {iter = 25}
            }
}
--------------------------------------------------------------------------------
-- Definition of the test-case.

---- Mesh definition --
mesh = {
        predefined = 'cube',
        origin = { -32.0, -32.0, -32.0 },
        length = 64.0,
        refinementLevel = meshlevel
}


-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
        name = 'modg',
        m =  degree,
        modg_space = poly_space
    }, 
    -- the temporal discretization scheme
    temporal = {
        name = 'explicitRungeKutta', 
        steps = 4,
        control = {
            name = 'cfl',
            cfl  = 0.95
        }
    }
}
