-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Nernst-Planck equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of standing waves for electric and magnetic fields. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- To verify diffusion and dispersion relations this testcases allows you to modify the spatial harmonics by
-- varying the integer mode number in x and y direction by modifying the lua variables m and n. Please notice,
-- this testcase is correct only for homogenous media with D = 1 (see equations table).
-- This testcase can be run in serial (only one execution unit).


-- Some global parameters for the T_{nm} mode testcase
-- ...the number of mpi ranks
nprocs = 1
-- ...the uniform refinement level for the periodic cube
level = 1
-- ...the length of the cube
cubeLength = 2.0
-- ...how many waves in x direction (initial condition)
waveNumber = 0
-- ...nernstplanck diffusivity
diffusivity = 1


-- The analytic soultion for this testcase is given by the following function:
-- ... concentration
function concentration(x,y,z,t)
  D = diffusivity;
  L = cubeLength;
  k = waveNumber
  return math.exp(-D*(2*math.pi*k/L)^2 * t) * math.cos((2*math.pi*k/L) * x)
end

-- The initial condition for the diffusive flux. This is just a dummy, see ticket.
function diffusiveFlux(x,y,z) 
  return 0
end

-- global simulation options
simulation_name = 'nerplanck_cos_modg' -- the name of the simualtion
sim_control = {
                time_control = { max  = 1.0 } -- final simulation time
              }

--commpattern = 'gathered_type'

-- Mesh definitions --
mesh = { predefined = 'cube',
         origin = { 
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0
                  },
         length = cubeLength,
         refinementLevel = level
       }

-- DEBUG OPTIONS
debug = { 
         debugMode = true, 
         debugFiles = true,
         verbose = true,
         dumpTreeIDs = true,
         dumpDependencies = true,
         checkSteps = true,
         dumpState = true,
--         debugStates = { 
--                        write = {
--                                 folder    = './debug/',  -- the folder for the debug mesh 
--                                 interval  = 1,           -- dump restart file interval
--                                 tmin      = 1,           -- first timestep to output
--                                 tmax      = tmax+1       -- last timestep to output
--                                }
--                       }
        }


-- Restart settings
estart = { 
--            -- folder to write restart data to
            write = './restart/nerplanck/',                                        
            -- temporal definition of restart write
            time_control = {   
              min = 0, 
              max = sim_control.time_control.max, 
              interval = sim_control.time_control.max/10.0
            }
          }

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'        

-- Tracking of a certain variable at a certain point in space over time.           
racking = { 
             -- Give a name for the tracked object.
             label = 'concentration', 
             -- the variable of the solver you track
             variable = {'concentration'},
             -- we track a point at the origin
             shape={  
                      kind = 'canoND', 
                      object = {origin ={cubeLength/4.0,cubeLength/4.0,cubeLength/4.0} }
                   },
             -- the temporal definition of the tracking: when do we start?
             time_control = {
               min = 0, 
               -- the maximum time: max = -1 assigns max=tmax
               max = sim_control.time_control.max, 
               interval = sim_control.time_control.max/100.0
             }, 
             format = 'ascii',
             folder = './tracking/nerplanck/'
         }

-- Equation definitions --
equation = {
    name   = 'nernstPlanck',                 -- we solve the nernst-planck equations
    D = diffusivity                       -- the diffusivity
}

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg',            -- we use the modal discontinuous Galerkin scheme 
               m = 5,                    -- the maximal polynomial degree for each spatial direction
               }, 
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitRungeKutta',  --'explicitEuler',
               steps = 4,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.03,     -- CourantFriedrichsLewy number
                         },
               },
}

-- ...the initial condition table. 
-- ...initial condition function for electric field (z component)
function ic_concentration(x,y,z)
  return concentration(x,y,z,0.0)
end


-- ...the general projection table
projection = {
           --   kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattopoints = false  -- if lobatto points should be used, default = false,
                                     -- only working for chebyshev points --> fpt
              factor = 10.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
           -- blocksize = 32,        -- for fpt, default -1
           -- fftmultithread = false -- for fpt, logical, default false
             }

initial_condition = { 
                      concentration = ic_concentration,
                      diffusiveFluxX = diffusiveFlux,
                      diffusiveFluxY = diffusiveFlux,
                      diffusiveFluxZ = diffusiveFlux
                    }

