-- Configuration file for Ateles --

--!--------------------------------------------!
-- This is a small testcase for qualitatively testing the
-- sponge layer. The initial condition is a weak gaussian 
-- pulse hitting the sponge and getting absorbed by it.
--!--------------------------------------------!



-- ...the length of the cube
cubeLength = 4.0

logging = {level = 3}

-- the refinement level of the octree
level = 4

back_press = 100000
-- smallness parameter
eps = cubeLength/(2.0^(level+8))
dx = 0.25

-- Transport velocity of the pulse in x direction.
velocityX = 100

simulation_name = 'sponge_2d_modg'


-- global simulation options
sim_control = {
              time_control = {   
                      min = 0, 
                      max = { sim = cubeLength/(2*velocityX)},--, iter=1},              -- final simulation time
                      interval = {iter = 50}
                     }
                   }
iniVel = velocityX 
iniDens = 1.225
characteristic = 0.0
function relax_velocity(x,y,z,t)
  return {0.0, 0.0}
end
relax_temperature = 0.0

variable = {
  {
    name = "spongelayer_var",
    ncomponents = 5,
    vartype = "st_fun",
    st_fun = {
      predefined = 'combined',
      spatial = { 
        predefined = 'spongelayer_2d',
        plane_origin = { 2.0, 0.0, 0 },
        plane_normal = { 2.0, 0, 0 },
        damp_factor = 800,
        damp_exponent = 2,
        target_state = {
          density = 1.225,
          velx = iniVel,
          vely = 0.0,
          pressure = back_press
        }
      },
      shape = {
        kind = 'canoND',
        object= {
          origin = { 2.0 - eps, -eps, -eps },
          vec = {
            { 2.0 + eps, 0, 0 },
            { 0, cubeLength + eps, 0 },
            { 0, 0, dx + eps }
          },
          segments = { 50, 50, 3 }
        }
      }
    }
  },
  -- This is the global material for Euler 2D. It consists of three different 
  -- components, characteristics with one scalar, relax_velocity with two 
  -- scalars and relax_temperature with one scalar, thus we need five scalars
  -- for this equation system.
  -- As this is the global fallback material, we define each material to be a 
  -- neutral term, which in this case is 0.
  {
     name = 'characteristic',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = characteristic
  },
  {
     name = 'relax_velocity',
     ncomponents = 2,
     vartype = "st_fun",
     st_fun = relax_velocity
  },
  {
     name = 'relax_temperature',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = relax_temperature
  }
}

-- Mesh definitions --
mesh = { predefined = 'slice',
         origin = {
                    0.0,
                    0.0,
                    0.0
                  },
         length = cubeLength,
         refinementLevel = level
       }


-- Equation definitions --
equation = {
    name   = 'euler_2d',
    therm_cond = 2.555e-02,
    isen_coef = 1.4,
    r      = 296.0,
    material = {
                 characteristic = 'characteristic',
                 relax_velocity = 'relax_velocity',
                 relax_temperature = 'relax_temperature'
               }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg_2d',            -- we use the modal discontinuous Galerkin scheme 
               m = 12,                   -- the maximal polynomial degree for each spatial direction
            --   dealiasFactor = 1.0,      -- factor to remove aliases: 1.0 means no additional dealiasing
            --   blocksize = 32,           -- The minimal blocksize for the FPT 
             --  fftMultiThread = true,   -- Use multithreaded version of FFTW.
               }, 
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitRungeKutta',  --'explicitEuler',
               steps = 4,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.6,     
                         },
               },
}

function dens(x,y,z)
  return x
end

projection = {
              kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
             }

function ic_gauss (x,y,z)
  d= (x-1)*(x-1) + (y-2)*(y-2)
  return( 1.225 + 0.01* math.exp(-d/0.05*math.log(2.0)) )
end


-- This is a very simple example to define constant boundary condtions.
initial_condition = { density = ic_gauss,
                      pressure = back_press, 
                      velocityX = velocityX,
                      velocityY = 0,
                    }


restart = {
             label = 'sponge',
             write = './restart/',
             time_control = {
                      min = 0,
                      max = sim_control.time_control.max.sim,
                      interval = sim_control.time_control.max.sim/100.0
                    },
           }


-- Sponge definition
source = {
  spongelayer = "spongelayer_var"
}

