-- Configuration file for Ateles --

-- ...the length of the cube
cubeLength = 2.0

logging = {level = 10}

-- the refinement level of the octree
level = 3

-- smallness parameter
eps = cubeLength/(2.0^(level+8))

-- Transport velocity of the pulse in x direction.
velocityX = 100

simulation_name = 'sponge_2d_modg'

dx = cubeLength/(2^(level))

-- global simulation options
sim_control = {
              time_control = {   
                      min = 0, 
                      max = { sim = 0.008}--, iter=1},              -- final simulation time
                     }
                   }
iniVel = velocityX 
iniDens = 1.225
characteristic = 0.0
function relax_velocity(x,y,z,t)
  return {0.0, 0.0}
end
relax_temperature = 0.0

variable = {
  {
    name = "spongelayer_var",
    ncomponents = 5,
    vartype = "st_fun",
    st_fun = {
      predefined = 'combined',
      spatial = { 
        predefined = 'spongelayer_2d',
        plane_origin = { 0.25, 0.0, 0 },
        plane_normal = { 0.6, 0, 0 },
        damp_factor = 1500,
        target_state = {
          density = 1.225,
          velocityX = iniVel,
          velocityY = 0.0,
          pressure =100000
        }
      },
      shape = {
        kind = 'canoND',
        object= {
          origin = { 0.2, -cubeLength/2.0, -cubeLength/2.0 },
          vec = {
            { 0.8, 0, 0 },
            { 0, cubeLength, 0 },
            { 0, 0, dx }
          },
          segments = { 50, 50, 3 }
        }
      }
    }
  },
  -- This is the global material for Euler 2D. It consists of three different 
  -- components, characteristics with one scalar, relax_velocity with two 
  -- scalars and relax_temperature with one scalar, thus we need five scalars
  -- for this equation system.
  -- As this is the global fallback material, we define each material to be a 
  -- neutral term, which in this case is 0.
  {
     name = 'characteristic',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = characteristic
  },
  {
     name = 'relax_velocity',
     ncomponents = 2,
     vartype = "st_fun",
     st_fun = relax_velocity
  },
  {
     name = 'relax_temperature',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = relax_temperature
  }
}

-- Mesh definitions --
mesh = { predefined = 'slice',
         origin = {
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0
                  },
         length = cubeLength,
         refinementLevel = level
       }

timing = { filename = 'timing.res' }

-- Equation definitions --
equation = {
    name   = 'euler_2d',
    therm_cond = 2.555e-02,
    isen_coef = 1.4,
    r      = 296.0,
    material = {
                 characteristic = 'characteristic',
                 relax_velocity = 'relax_velocity',
                 relax_temperature = 'relax_temperature'
               }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg_2d',            -- we use the modal discontinuous Galerkin scheme 
               m = 3,                   -- the maximal polynomial degree for each spatial direction
            --   dealiasFactor = 1.0,      -- factor to remove aliases: 1.0 means no additional dealiasing
            --   blocksize = 32,           -- The minimal blocksize for the FPT 
             --  fftMultiThread = true,   -- Use multithreaded version of FFTW.
               }, 
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitRungeKutta',  --'explicitEuler',
               steps = 4,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.9,     
                         },
               },
}

function dens(x,y,z)
  return x
end

projection = {
              kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattoPoints = false  -- if lobatto points should be used, default = false
              factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
              blocksize = 32,        -- for fpt, default -1
              fftMultiThread = false -- for fpt, logical, default false
             }

function ic_gauss (x,y,z)
  d= x*x + y*y
  return( 1.225 + 2* math.exp(-d/0.1*math.log(2.0)) )
end


-- This is a very simple example to define constant boundary condtions.
initial_condition = { density = ic_gauss,--{
                                -- predefined='gausspulse',
                                -- center={-0.5, 0.0, 0.0},
                                -- halfwidth=0.20,
                                -- amplitude=1.0,
                                -- background=1.225
                                --},
                      pressure = 100000, 
                      velocityX = velocityX,
                      velocityY = 0,
                    }



-- Tracking
tracking = {
             label = 'sponge_2d',
             folder = './',
             variable = {'density'},
                       shape = {   kind = 'canoND', 
                           object= {
                                   origin = {0.55 , 0, -0.875},}},
             time_control = {
                      min = 0,
                      --max = sim_control.time_control.max.sim,
                      interval =  sim_control.time_control.max.sim/10.0
                    },
             output = { format = 'ascii', ndofs = 1 }, -- 'asciiSpatial', 'harvester', 'convergence'
           }

estart = {
             label = 'sponge',
             write = './restart/',
             time_control = {
                      min = 0,
                      max = sim_control.time_control.max.sim,
                      interval = sim_control.time_control.max.sim/10.0
                    },
           }


-- Sponge definition
source = {
  spongelayer = "spongelayer_var"
}


--                 -- Outlet sponge
--  spongeLayer = {
--    shape = {
--      kind = 'canoND',
--      object= {
--        origin = { 0.2, -cubeLength/2.0, -cubeLength/2.0 },
--        vec = {
--          { 0.8, 0, 0 },
--          { 0, cubeLength, 0 },
--          { 0, 0, dx }
--        },
--        segments = {50,50,3},
--      }
--    },
--    kind = 'combined',
--    spatial = {
--      predefined = 'spongeLayer_2d',
--      plane_origin = {0.25,0.0,0},
--      plane_normal = {0.6,0,0},
--      damp_factor = 1500,
----      damp_exponent = {},
--      target_state = {
--        density = 1.225,
--        velx = iniVel,
--        vely = 0.0,
--        pressure =100000
--      }
--    },
----    temporal = {1},
--  }
--}

