require('seeder_2d')

-- Fluid parameters
isen_coef = 1.4
r = 296

-- global simulation options
name = 'shock_vortex'          
sim_control = {
              time_control = {
                                min = 0.0,
                                max = 2.0,      
                                interval = { iter = 100 },
                             },
             }


-- Mesh definitions --
mesh = './mesh/'

check = {
          interval = 5,
        }

-- Restart settings
restart = { 
            write = './restart/',                                        
            time_control = {   
                             min = 0, 
                             max = sim_control.time_control.max, 
                             interval = sim_control.time_control.max/100, 
                          },  
          }

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'        

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg_2d', 
               m = 31,           --TODO 15,
               }, 
    -- the stabilzation of the scheme
    stabilization = {
                      {
                       name = 'spectral_viscosity',
                       alpha = 36,
                       order = 30,
                      },
                    },
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitSSPRungeKutta',  --'explicitEuler',
               steps = 2,
               -- how to control the timestep
               control = {
                          name = 'cfl',   
                          cfl  = 0.1,      
                          cfl_visc  = 0.1,      
                         },
               },
}

-- The state right of the shock
densR = 1.0
velXR = 0.0
velYR = 0.0
pressR = 1.0

-- Shock properties
shockMach = 1.25
shockXCoord = -1.2
shockSpeed = shockMach * math.sqrt(isen_coef * pressR / densR )

-- The state left of the shock (evaluated by Rankine-Huginoit condition)
chi = ( velXR - shockSpeed ) / math.sqrt(isen_coef * pressR / densR )
densL = densR * ( ((isen_coef+1)*chi*chi) / ((isen_coef-1)*chi*chi+2) )  
velXL = shockSpeed + ( velXR - shockSpeed ) * (densR/densL) 
velYL = 0.0
pressL = pressR * ( (2*isen_coef*chi*chi-(isen_coef-1)) / (isen_coef+1) ) 

-- The vortex properties
Gamma = 0.25
r0 = 0.2
r1 = 1.0
xCenter = 0.0
yCenter = 0.0
function velPolar(r)
  if( r <= r0 ) then
    return Gamma * r * ( (1/(r0*r0)) - (1/(r1*r1)) )
  elseif( r <= r1 ) then
    return Gamma * r * ( (1/(r*r)) - (1/(r1*r1)) )
  else
    return 0.0
  end 
end
function vortexVelXR(x,y,z,t)
  r = math.sqrt(x*x + y*y)
  theta = math.atan2(y-yCenter,x-xCenter)
  return (-1.0)*velPolar(r)*math.sin(theta)
end
function vortexVelYR(x,y,z,t)
  r = math.sqrt(x*x + y*y)
  theta = math.atan2(y-yCenter,x-xCenter)
  return velPolar(r)*math.cos(theta)
end

-- The definition of boundary value functions 
function velX(x,y,z,t)
  if( x < shockXCoord ) then
    return velXL
  else
    return velXR + vortexVelXR(x,y,z,t)
  end
end
function velY(x,y,z,t)
  if( x < shockXCoord ) then
    return velYL
  else
    return velYR + vortexVelYR(x,y,z,t)
  end
end
function dens(x,y,z,t)
  if( x < shockXCoord ) then
    return densL
  else
    return densR
  end
end 
function press(x,y,z,t)
  if( x < shockXCoord ) then
    return pressL
  else
    return pressR
  end
end 

-- The definition of initial value functions 
function iniVelX(x,y,z)
  return velX(x,y,z,0.0)
end
function iniVelY(x,y,z)
  return velY(x,y,z,0.0)
end
function iniDens(x,y,z)
  return dens(x,y,z,0.0)
end
function iniPress(x,y,z)
  return press(x,y,z,0.0)
end

-- This is a very simple example to define constant boundary condtions.
initial_condition = { density = iniDens,
                      pressure = iniPress,
                      velocityX = iniVelX,
                      velocityY = iniVelY,
                      velocityZ = 0.0,
                    }

-- ...the general projection table
projection = {
              kind = 'fpt', 
              lobattopoints = true, 
              factor = 1.0,          
           -- blocksize = 32,        
           -- fftmultithread = false 
             }

-- Boundary definitions
boundary_condition = {
                       { 
                       label = 'inflow',  
                       kind = 'inflow_normal',      
                       density = dens,
                       v_norm = velX,
                       v_tan = 0.0,
                       }
                       ,
                       { 
                       label = 'outflow',  
                       kind = 'outflow',       
                       pressure = press,
                       }
                       ,
                      --TODO { 
                      --TODO label = 'slipSouth', 
                      --TODO kind = 'slipwall',       
                      --TODO }
                      --TODO ,
                      --TODO { 
                      --TODO label = 'slipNorth', 
                      --TODO kind = 'slipwall',       
                      --TODO }
                      --TODO ,
                      --TODO { 
                      --TODO label = 'slipBottom', 
                      --TODO kind = 'slipwall',       
                      --TODO }
                      --TODO ,
                      --TODO { 
                      --TODO label = 'slipTop', 
                      --TODO kind = 'slipwall',       
                      --TODO }
                      --TODO ,
                     }  

-- Sponge definition
source = {
  spongeLayer = {
    shape = {
      kind = 'canoND',
      object= {
        origin = { 0.0, -cubeLength/2, -cubeLength/2 },
        vec = { 
          { cubeLength, 0, 0 },
          { 0, cubeLength, 0 },
          { 0, 0, cubeLength }
        },
        segments = {
          cubeLength/smalleps,
          cubeLength/smalleps,
          cubeLength/smalleps
        }
      }
    },
    kind = 'combined',
    spatial = { 
      predefined = 'spongeLayer_2d', 
      plane_origin = { largeeps/2, 0.0, 0 },
      plane_normal = { 1.0, 0, 0 },
      damp_factor = 100,
--      damp_exponent = {},
      target_state = {
        density = densR,
        velx = velXR,
        vely = velYR,
        pressure= pressR
      },
    }
  }
}
