-- global simulation options
simulation = {
              name          = 'shock_vortex',          
              time          = {
                                useIterations = false,
                                min = 0.0,
                                max = 2.0,      
                              },
             }


-- Mesh definitions --
mesh = './mesh/'

timestep_info = 100
check = {
          interval = 1,
        }

-- Restart settings
restart = { 
--            -- file to restart from
--            read = './restart/plate_euler_modg_lastHeader.lua',                        
--            -- folder to write restart data to
            write = './restart/',                                        
            -- temporal definition of restart write
            time = {   
                      useIterations = false,  
                      min = 0, 
                      max = simulation.time.max, 
                      interval = simulation.time.max/2e+01,
                   },  
          }

-- timing settings (i.e. output for performance measurements, this table is otional)
timing = {
          folder = './',          
          filename = 'timing.res', 
         }

-- Equation definitions --
equation = {
    name   = 'euler',
    isen_coef = 1.4,
    r      = 296.0,
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg', 
               m = 31,
               dealiasFactor = 1.0, 
               blocksize = 32,
               lobattoPoints = true,
               }, 
    -- the stabilzation of the scheme
    stabilization = {
                      {
                       name = 'spectral_viscosity',
                       alpha = 36,
                       order = 10
                      },
                      {
                         name = 'cons_positivity_preserv',
                         eps = 1.0e-08,
                      },
                    },
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitSSPRungeKutta',  --'explicitEuler',
               steps = 2,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.1,       -- CourantFriedrichsLewy number
                         },
               },
}


-- The state right of the shock
densR = 1.0
velXR = 0.0
velYR = 0.0
velZR = 0.0
pressR = 1.0

-- Shock properties
shockMach = 1.25
shockXCoord = -1.2
shockSpeed = shockMach * math.sqrt(equation.isen_coef * pressR / densR )

-- The state left of the shock (evaluated by Rankine-Huginoit condition)
chi = ( velXR - shockSpeed ) / math.sqrt(equation.isen_coef * pressR / densR )
densL = densR * ( ((equation.isen_coef+1)*chi*chi) / ((equation.isen_coef-1)*chi*chi+2) )  
velXL = shockSpeed + ( velXR - shockSpeed ) * (densR/densL) 
velYL = 0.0
velZL = 0.0
pressL = pressR * ( (2*equation.isen_coef*chi*chi-(equation.isen_coef-1)) / (equation.isen_coef+1) ) 

-- transformation functions for cartesian to polar coord
function cartToPolar(x, y, z)
  r = math.sqrt(x*x + y*y + z*z)
  phi = math.atan2(y,x)
  theta = math.acos( z/r )
  return r , phi, theta
end
function polarToCart( r, phi, theta )
  x = r * math.sin(theta) * math.cos(phi)
  y = r * math.sin(theta) * math.sin(phi)
  z = r * math.cos(theta)
  return x, y, z 
end 

-- The vortex properties
Gamma = 0.25
r1 = 0.25
r0 = 0.05
R = 0.75
function vortexVelXR(x,y,z,t)
  -- convert to polar coordinates
  r_polar, phi_polar, theta_polar = cartToPolar(x,y,z)

  -- center point of the vortex ring
  x_center, y_center, z_center = polarToCart(R, phi_polar, math.pi/2.0)

  -- move coordinates to local coordinate of the vortex ring center
  x_local = x - x_center
  y_local = y - y_center
  z_local = z - z_center

  -- convert to local polar coordinates
  r_loc, phi_loc, theta_loc = cartToPolar( x_local, y_local, z_local )

  if( r_loc <= r0 ) then
    vel = Gamma * r * ( (1/(r0*r0)) - (1/(r1*r1)) )
  elseif( r_loc <= r1 ) then
    vel = Gamma * r * ( (1/(r*r)) - (1/(r1*r1)) )
  else
    vel = 0.0
  end 

  velTmpX = vel * math.cos(phi_polar) * math.cos(theta_loc)
  return velTmpX

end 
function vortexVelYR(x,y,z,t)
  -- convert to polar coordinates
  r_polar, phi_polar, theta_polar = cartToPolar(x,y,z)

  -- center point of the vortex ring
  x_center, y_center, z_center = polarToCart(R, phi_polar, math.pi/2.0)

  -- move coordinates to local coordinate of the vortex ring center
  x_local = x - x_center
  y_local = y - y_center
  z_local = z - z_center

  -- convert to local polar coordinates
  r_loc, phi_loc, theta_loc = cartToPolar( x_local, y_local, z_local )

  if( r_loc <= r0 ) then
    vel = Gamma * r * ( (1/(r0*r0)) - (1/(r1*r1)) )
  elseif( r_loc <= r1 ) then
    vel = Gamma * r * ( (1/(r*r)) - (1/(r1*r1)) )
  else
    vel = 0.0
  end 

  velTmpY = vel * math.sin(phi_polar) * math.cos(theta_loc)
  return velTmpY

end 
function vortexVelZR(x,y,z,t)
  -- convert to polar coordinates
  r_polar, phi_polar, theta_polar = cartToPolar(x,y,z)

  -- center point of the vortex ring
  x_center, y_center, z_center = polarToCart(R, phi_polar, math.pi/2.0)

  -- move coordinates to local coordinate of the vortex ring center
  x_local = x - x_center
  y_local = y - y_center
  z_local = z - z_center

  -- convert to local polar coordinates
  r_loc, phi_loc, theta_loc = cartToPolar( x_local, y_local, z_local )

  if( r_loc <= r0 ) then
    vel = Gamma * r * ( (1/(r0*r0)) - (1/(r1*r1)) )
  elseif( r_loc <= r1 ) then
    vel = Gamma * r * ( (1/(r*r)) - (1/(r1*r1)) )
  else
    vel = 0.0
  end 

  velTmpZ = vel * math.sin(theta_loc)
  if( r_polar > R ) then
    velTmpZ = velTmpZ * (-1.0)
  end
  return velTmpZ

end 

-- The definition of boundary value functions 
function velX(x,y,z,t)
  if( x < shockXCoord ) then
    return velXL
  else
    return velXR + vortexVelXR(x,y,z,t)
  end
end
function velY(x,y,z,t)
  if( x < shockXCoord ) then
    return velYL
  else
    return velYR + vortexVelYR(x,y,z,t)
  end
end
function velZ(x,y,z,t)
  if( x < shockXCoord ) then
    return velZL
  else
    return velZR + vortexVelZR(x,y,z,t)
  end
end
function dens(x,y,z,t)
  if( x < shockXCoord ) then
    return densL
  else
    return densR
  end
end 
function press(x,y,z,t)
  if( x < shockXCoord ) then
    return pressL
  else
    return pressR
  end
end 

-- The definition of initial value functions 
function iniVelX(x,y,z)
  return velX(x,y,z,0.0)
end
function iniVelY(x,y,z)
  return velY(x,y,z,0.0)
end
function iniVelZ(x,y,z)
  return velZ(x,y,z,0.0)
end
function iniDens(x,y,z)
  return dens(x,y,z,0.0)
end
function iniPress(x,y,z)
  return press(x,y,z,0.0)
end

-- This is a very simple example to define constant boundary condtions.
initial_condition = { density = iniDens,
                      pressure = iniPress,
                      velocityX = iniVelX,
                      velocityY = iniVelY,
                      velocityZ = iniVelZ,
                    }

-- ...the general projection table
projection = {
              kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattopoints = false  -- if lobatto points should be used, default = false,
                                     -- only working for chebyshev points --> fpt
              factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
           -- blocksize = 32,        -- for fpt, default -1
           -- fftmultithread = false -- for fpt, logical, default false
             }


-- Boundary definitions
boundary_condition = {
                       { 
                       label = 'inflow',  
                       kind = 'inflow_normal',      
                       density = dens,
                       v_norm = velX,
                       v_tan = 0.0,
                       }
                       ,
                       { 
                       label = 'outflow',  
                       kind = 'outflow',       
                       pressure = press,
                       }
                       ,
                       { 
                       label = 'slipSouth', 
                       kind = 'slipwall',       
                       }
                       ,
                       { 
                       label = 'slipNorth', 
                       kind = 'slipwall',       
                       }
                       ,
                       { 
                       label = 'slipBottom', 
                       kind = 'slipwall',       
                       }
                       ,
                       { 
                       label = 'slipTop', 
                       kind = 'slipwall',       
                       }
                       ,
                     }  
