-- Configuration file for Ateles --
require('seeder')

spatial_order = 64

-- global simulation options
simulation = {
              name = 'penalize_euler', 
              time = {
                       useIterations = false, 
                       min = 0, 
                       max = 0.4,
                     }
             }

-- Mesh definitions --
mesh = 'mesh/'

timestep_info = 10


-- Restart settings
io_buffer_size = 10
restart = { 
            write = './restart/',                                        
            time = {   
                      useIterations = false,  
                      min = 0, 
                      max = simulation.time.max, 
                      interval = simulation.time.max/4,
                   },  
          }

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'        

-- ...the initial condition table. 
isen = 1.4
boltz= 296.0
rho  = 1.0
p    = 1.0
u    = 0.0
temp = p /( rho * boltz)

-- ... the smallness parameter of the penalization
penalization_eps = 0.001
penalization_alpha = 0.01
-- ... the center of the obstacle
x_center = 3.0*channel_length/4.0
-- ... the radius of the obstacle
radius = channel_length/4.0
function obstacle_function(x,y,z,t)
  -- Calculate the distance from the circle center
  dist = math.sqrt( math.pow(x-x_center,2.0) )
  -- Check if the point is inside the circle
  if dist <= radius  then
    return { 1.0, 0.0, 0.0, 0.0, temp }
  else
    return { 0.0, 0.0, 0.0, 0.0, 0.0}
  end 
end

-- Initial condition definitions --
-- ... properties of the pulse
pulse_amplitude = 0.05*p
pulse_center = 1.0*channel_length/4.0
pulse_width = channel_length/10.0
function pressure(x,y,z)
  return p + pulse_amplitude*math.exp( (-36.0)*math.pow((x-pulse_center)/pulse_width,2) )
end
initial_condition = { 
                      density  = rho,
                      pressure = pressure,
                      velocity_X = u,
                      velocity_Y = 0.0,
                      velocity_Z = 0.0,
                    }

-- Equation definitions --
equation = {
    name   = 'euler',                 
    -- The definition of an obstacle by penalization
    penalization = {
                     {
                        kind = 'lua_fun',
                        fun = obstacle_function,
                        shape={ 
                                  kind = 'canoND', 
                                  object = {
                                             origin = {
                                                       x_center-radius,
                                                       channel_width/2.0,
                                                       channel_width/2.0,
                                                      },
                                             vec = {
                                                      {2.0*radius,0.0,0.0},
                                                      {0.0,dx,0.0},
                                                      {0.0,0.0,dx},
                                                   },
                                             segments = {100,100,100},
                                           },
                              },
                     },
                     global = {
                                kind = 'const',
                                const = {0.0, 0.0, 0.0, 0.0, 0.0},
                              },
                   }, 
    -- Physical parameters of the gas phase
    isen_coef            = isen,
    r                    = boltz,
    -- Parameters of the penalization
    porosity             = penalization_eps,
    viscous_permeability = penalization_alpha*penalization_eps,
    thermal_permeability = penalization_alpha*penalization_eps,
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Check for Nans and unphysical values
check =  {
           interval = 1,
         }

-- Scheme definitions --
scheme = {
    spatial =  {
                  name = 'modg',          
                  m = spatial_order-1,                 
                  modg_space = 'Q' 
               }, 
    temporal = {
                  name = 'explicitSSPRungeKutta',  
                  steps = 2,
                  control = {
                             name = 'cfl', 
                             cfl  = 0.3,  
                            },
               },
    stabilization = {
                       name = 'spectral_viscosity',
                       alpha = 36,
                       order = 26, 
                    }
}


-- ...the general projection table
projection = {
              kind = 'fpt', 
              lobattopoints = true,
              factor = 1.0,  
              blocksize = 32,
              -- fftmultithread = true, 
             }

-- Boundary conditions
boundary_condition = {
                        { 
                         label = 'inlet', 
                         kind = 'slipwall',
                         --kind = 'inflow_normal',
                         --density = rho,
                         --v_norm = u,
                         --v_tan = 0.0,
                        }
                         ,
                        { 
                         label = 'outlet', 
                         kind = 'slipwall',
                         --kind = 'outflow',
                         --pressure = p,
                         }
                         ,
                        { 
                         label = 'bottom', 
                         kind = 'slipwall', 
                         }
                         ,
                        { 
                         label = 'top', 
                         kind = 'slipwall', 
                         }
                         ,
                        { 
                         label = 'south', 
                         kind = 'slipwall', 
                         }
                         ,
                        { 
                         label = 'north', 
                         kind = 'slipwall', 
                         }
}
