-- Configuration file for Ateles --
require 'seeder_2d'


-- global simulation options
simulation_name = 'jet_euler_modg'
sim_control = {
              time_control  = {
                                min = 0.0,
                                max = 40.0,
                                interval = { iter=1000 },
                              },
             }

check = {
          interval = 100,
        }

-- Mesh definitions --
mesh = './mesh/'


-- Restart settings
restart = { 
            --read = './restart/jet_euler_modg_lastHeader.lua',
            write = './restart/',                                        
            time_control = {   
                             useIterations = false,  
                             min = 0, 
                             max = sim_control.time_control.max, 
                             interval = sim_control.time_control.max/400.0,
                           },  
          }

-- the filename of the timing results i.e. output for performance measurements
timing_file = 'timing.res'        

-- Equation definitions --
equation = {
    name   = 'euler_2d',
    therm_cond = 2.555e-02,
    isen_coef = 1.4,
    r      = 296.0,
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
                  name = 'modg_2d',            -- we use the modal discontinuous Galerkin scheme 
                  m = 31,        -- the maximal polynomial degree for each spatial direction
               }, 
    stabilization = {
                      {
                       name = 'spectral_viscosity',
                       alpha = 36,
                       order = 10,
                      },
                    },
    -- the temporal discretization scheme
    temporal = {
               name = 'explicitSSPRungeKutta', 
               steps = 2,
               control = {
                          name = 'cfl', 
                          cfl  = 0.6, 
                         },
               },
}

-- The ambient medium
press = 1
dens = 1

-- the Mach number of the jet
mach = 0.4

-- the velocity amplitude in z direction
velAmpl = mach*math.sqrt(equation.isen_coef*press/dens) -- 0.1
densAmpl = 2.0


-- the radius of the jet
jet_radius = 0.1
jet_center = 1e-4
momentum_thickness = jet_radius/20

function velX_inlet(x,y,z,t)
  r = math.sqrt( (y-jet_center)*(y-jet_center) )
  return velAmpl * (1/2) * ( 1 +  math.tanh((jet_radius-r)/(2*momentum_thickness) ) )
end 

function dens_inlet(x,y,z,t)
  tmpVel = velX_inlet(x,y,z,t)/velAmpl
  return densAmpl / ( 1 + (equation.isen_coef-1)/2*mach*mach*tmpVel*(1-tmpVel)  )
end


--TODO -- random number for x and y velocity at initial time 
--TODO math.randomseed(1234)
--TODO function velXYIni(x,y,z)
--TODO   r = math.sqrt( x*x + y*y )
--TODO   return velAmpl * 1e-06 * math.exp( (-3/jetwidth) * r ) * ( math.tanh( -D * ( z - minzcoord) ) + 1.0 ) * math.random()
--TODO end

-- This is a very simple example to define constant boundary condtions.
initial_condition = { density = dens,
                      pressure = press,
                      velocityX = 0.0,
                      velocityY = 0.0,
                    }

-- ...the general projection table
projection = {
              kind = 'fpt',  
              factor = 1.0,          
              lobattoPoints=true,
             }


-- Boundary definitions
boundary_condition = {
                       { 
                       label = 'inflowWest',  
                       kind = 'inflow_normal',      
                       density = dens_inlet,
                       v_norm = velX_inlet,
                       v_tan = 0.0,
                       }
                       ,
                       { 
                       label = 'outflowEast',  
                       kind = 'outflow',       
                       pressure = press,
                       }
                       ,
                       { 
                       label = 'outflowNorth', 
                       kind = 'slipwall',       
                       --kind = 'outflow',       
                       --pressure = press,
                       }
                       ,
                       { 
                       label = 'outflowSouth', 
                       kind = 'slipwall',       
                       --kind = 'outflow',       
                       --pressure = press,
                       }
                       ,
                     }  
source = {
  spongeLayer = {
    shape = {   
      kind = 'canoND', 
      object= {
        origin = {
          bounding_cube.length-3*elemsize+eps ,
          -cubeLength/2 ,
          -eps ,
        },
        vec = {
          { 2 * elemsize, 0, 0 },
          { 0, cubeLength, 0 },
          { 0, 0, elemsize }
        },
        segments = { 400, 400, 12 }
      } 
    },
    kind = 'combined',
    spatial = { 
      predefined = 'spongeLayer_2d', 
      plane_origin = { bounding_cube.length - 3 * elemsize, 0.0, 0 },
      plane_normal = { 1.0, 0,0 },
      damp_factor = 50,
      damp_exponent = 2,
      target_state = {
        density = dens,
        velx = 0.0,
        vely = 0.0,
        pressure = press
      },
    }
  }
}
