-- --------------- General options --------------- --
simulation_name = 'convected_vortex'
sim_control = {
  time_control = {
    min = 0,
    max = 6,
    interval = {iter=1}
  },
  abort_criteria = {stop_file = 'stop'}
}
-- --------------- General options --------------- --
-- ----------------------------------------------- --


-- ------------------- --
-- ------ Mesh ------- --
mesh = {
  predefined = 'slice',
  origin = { -3.0, -3.0, -0.125 },
  length = 6,
  refinementLevel = 2
}
-- ------ Mesh ------- --
-- ------------------- --


-- --------------------------------------------------------- --
-- ----------------------- Equation ------------------------ --
gamma = 1.4
equation = {
  name = 'euler_2d',
  therm_cond = 24.35e-3, -- mW/(K*m)
  isen_coef = gamma,
  r = 1,
  material = {
    characteristic = 0,
    relax_velocity = {0, 0},
    relax_temperature = 1
  }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)
-- ----------------------- Equation ------------------------ --
-- --------------------------------------------------------- --


-- ----------------------- Scheme -------------------------- --
scheme = {
  -- the spatial discretization scheme
  spatial =  {
    name = 'modg_2d',
    m = 31
  },
  -- the temporal discretization scheme
  temporal = {
    name = 'explicitRungeKutta',
    steps = 4,
    -- how to control the timestep
    control = {
      name = 'cfl',
      use_modal_estimate = true,
      cfl  = 0.9
    }
  }
}

projection = {
  kind = 'l2p',
  factor = 1.0
}
-- ----------------------- Scheme -------------------------- --
-- --------------------------------------------------------- --


-- ------ Initial conditions ------- --
-- Vortex strength:
beta = 2
center = {0,0}
u_0 = 1
v_0 = 0

-- helpers:
twopi_q = beta/(2*math.pi)
pigamma_q = twopi_q^2/(2*gamma)
densexpo = 1/(gamma-1)

dist = function(x,y)
  return 1 - ((x-center[1])^2 + (y-center[2])^2)
end

Tvar = function(x,y)
  return -(gamma-1) * math.exp(dist(x,y)) * pigamma_q
end

vx = function(x,y)
  return -twopi_q * math.exp(0.5*dist(x,y))*y
end

vy = function(x,y)
  return twopi_q * math.exp(0.5*dist(x,y))*x
end

inidens = function(x,y,z)
  return (1 + Tvar(x,y))^densexpo
end

initial_condition = {
  density = inidens,
  pressure = function(x,y,z) return inidens(x,y,z)*(1+Tvar(x,y)) end,
  velocityX = function(x,y,z) return u_0 + vx(x,y) end,
  velocityY = function(x,y,z) return v_0 + vy(x,y) end
}
-- ------ Initial conditions ------- --
-- --------------------------------- --


-- --------------- Restart ---------------- --
restart = {
  write = 'restart_',
  time_control = {
    min = 0,
    max = sim_control.time_control.max,
    interval = {sim=0.1, clock=3600},
    align_trigger = {sim=true}
  }
}
-- ---------------------------------------- --


-- -------------------- Tracking ---------------------- --
tracking = {
  label = 'point_series',
  folder = './',
  variable = { 'density', 'momentum', 'energy' },
  shape = {
    kind = 'canoND',
    object= { origin = { 0.7, 0.7, 0} }
  },
  time_control = {
    min = 0,
    max = sim_control.time_control.max,
    interval = {iter=1}
  },
  output = { format = 'ascii', ndofs = 1 }
}
-- ---------------------------------------------------- --
