-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin
-- Solver ATELES. It provides a testcase for the simulation of Euler equations
-- in a homogenous media. The simulation domain is a periodic cube with edge
-- length 2.0. Therefore this is a very good way to verify your algorithmic
-- implementations, since this testcase does not involve any boundary
-- conditions.
-- The testcase simulates the temporal development of Gaussian pulse in density.
-- Since we are considering a very simple domain, an analytic solution is well
-- known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- This testcase can be run in serial (only one execution unit) or in parallel
-- (with multiple mpi ranks).  To specify the number of ranks please modify
-- nprocs variable. To calculate a grid convergence behavior please modify the
-- level variable. An increment of one will half the radius of your elements.
logging = { level = 10 }

timestep_info = 10

-- ...the length of the cube
cubeLength = 2.0

-- the refinement level of the octree
level = 1

-- Transport velocity of the pulse in x direction.
velocityX = 100

-- global simulation options
simulation_name = 'gPulseDens_euler_modg' -- the name of the simualtion
sim_control = {
  time_control = {
    min = 0,
    max = cubeLength/velocityX/4 -- final simulation time
  }
}

characteristic = 0.0
function relax_velocity(x,y,z,t)
  return {0.0, 0.0, 0.0}
end
relax_temperature = 0.0

variable = {
  -- This is the global material for Euler 3D. It consists of three different
  -- components, characteristics with one scalar, relax_velocity with three
  -- scalars and relax_temperature with one scalar, thus we need five scalars
  -- for this equation system.
  -- As this is the global fallback material, we define each material to be a
  -- neutral term, which in this case is 0.
  {
     name = 'characteristic',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = characteristic
  },
  {
     name = 'relax_velocity',
     ncomponents = 3,
     vartype = "st_fun",
     st_fun = relax_velocity
  },
  {
     name = 'relax_temperature',
     ncomponents = 1,
     vartype = "st_fun",
     st_fun = relax_temperature
  },
}

-- Mesh definitions --
mesh = {
  predefined = 'cube',
  origin = {
    (-1.0)*cubeLength/2.0,
    (-1.0)*cubeLength/2.0,
    (-1.0)*cubeLength/2.0
  },
  length = cubeLength,
  refinementLevel = level
}


---- Restart settings
estart = {
--            -- file to restart from
--            read = './restart/maxwell/per_osci_maxwell_modg_lastHeader.lua',
--            -- folder to write restart data to
  write = './restart/',
  -- temporal definition of restart write
  time_control = {
    min = 0,
    max = sim_control.time_control.max,
    interval = sim_control.time_control.max/8.0
  }
}

-- timing settings (i.e. output for performance measurements, this table is otional)
timing = {
  folder = './',                  -- the folder for the timing results
  filename = 'timing.res'         -- the filename of the timing results
}

-- Equation definitions --
equation = {
  name = 'euler',
  therm_cond = 2.555e-02,
  isen_coef = 1.4,
  r = 296.0,
  material = {
    characteristic = 'characteristic',
    relax_velocity = 'relax_velocity',
    relax_temperature = 'relax_temperature'
  }
}
-- (cv) heat capacity and (r) ideal gas constant
equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Scheme definitions --
scheme = {
  -- the spatial discretization scheme
  spatial =  {
    name = 'modg',            -- we use the modal discontinuous Galerkin scheme
    m = 7,                   -- the maximal polynomial degree for each spatial direction
    --   dealiasFactor = 1.0,      -- factor to remove aliases: 1.0 means no additional dealiasing
    --   blocksize = 32,           -- The minimal blocksize for the FPT
    --  fftMultiThread = true,   -- Use multithreaded version of FFTW.
  },
  -- the temporal discretization scheme
  temporal = {
    -- Explicit Runge Kutta
    name = 'explicitRungeKutta',
    steps = 4,
    -- Strong stability preserving explicit Rung Kutta:
    -- name = 'explicitSSPRungeKutta',
    -- steps = 2,
    -- Explicit Euler (unstable for higher spatial orders):
    -- name = 'explicitEuler',
    control = {
      name = 'cfl',   -- the name of the timestep control mechanism
      cfl  = 0.8,     -- Courant-Friedrichs-Lewy number
    }
  }
}

function dens(x,y,z)
  return x
end

-- ...the general projection table
projection = {
  kind = 'fpt', -- 'fpt' or 'l2p', default 'l2p'
  factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
}

-- This is a very simple example to define constant boundary condtions.
initial_condition = {
  density = {
    predefined = 'gausspulse',
    center = { 0.0, 0.0, 0.0 },
    halfwidth = 0.20,
    amplitude = 2.0,
    background = 1.225
  },
  pressure = 100000,
  velocityX = velocityX,
  velocityY = 0.0,
  velocityZ = 0.0,
}


-- Tracking
tracking = {
  label = 'track_momentum',
  folder = './',
  variable = {'momentum'},
  shape = {kind = 'canoND', object= { origin ={0., 0., 0.} } },
  time_control = {
    min = 0,
    max = sim_control.time_control.max,
    interval = sim_control.time_control.max/8.0
  },
  output = { format = 'ascii' , ndofs = 1}, -- 'asciiSpatial', 'harvester', 'convergence'
}


