-- Configuration file for Ateles --


-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Maxwell equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of standing waves for displacement and magnetic fields. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- To verify diffusion and dispersion relations this testcases allows you to modify the spatial harmonics by
-- varying the integer mode number in x and y direction by modifying the lua variables m and n. Please notice,
-- this testcase is correct only for homogenous media with epsi = mu = 1 (see equations table).
-- This testcase can be run in serial (only one execution unit) or in parallel (with multiple mpi ranks). To calculate a grid convergence behavior please modify the level variable. An increment of one will half the radius of your elements.


-- Some global parameters for the T_{nm} mode testcase
-- ...the length of the cube
cubeLength = 2.0
-- ...the integer number of the mode in x direction
amplX = 1.0
-- ...the integer number of the mode in y direction
amplY = 1.0

-- The analytic soultion for this testcase is given by the following functions (only correct for epsi = mu = 1):
-- ... definition of temporal angular frequency
w = math.sqrt(amplX^2 + amplY^2)
-- ... E_x = 0.0
function displacementX(x,y,z,t)
  return 0.0
end 
-- ... E_y = 0.0
function displacementY(x,y,z,t)
  return 0.0
end 
-- ... E_z(x,y,z,t) = sin(amplX \pi x) sin(amplY \pi y) cos(w t)
function displacementZ(x,y,z,t)
  return math.sin(amplX*math.pi*x)*math.sin(amplY*math.pi*y)*math.cos(w*t)
end 
-- ... B_x(x,y,z,t) = -\frac{\pi n}{w} sin(m \pi x) cos(n \pi y) sin(w t)
function magneticX(x,y,z,t)
  return (-1.0)*(math.pi*amplY/w)*math.sin(amplX*math.pi*x)*math.cos(amplY*math.pi*y)*math.sin(w*t)
end 
-- ... B_y(x,y,z,t) = \frac{\pi m}{w} cos(m \pi x) sin(n \pi y) sin(w t)
function magneticY(x,y,z,t)
  return (math.pi*amplX/w)*math.cos(amplX*math.pi*x)*math.sin(amplY*math.pi*y)*math.sin(w*t)
end 
-- ... B_z = 0.0
function magneticZ(x,y,z,t)
  return 0.0 
end


-- global simulation options
simulation_name = 'per_osci_max_modg' -- the name of the simualtion
sim_control = {
                time_control = { 
                  max = 1.0 -- final simulation time
                }
              }

--commpattern = 'gathered_type'

-- Mesh definitions --
mesh = './mesh/'


-- Restart settings
restart = { 
--            -- file to restart from
--            read = './restart/maxwell/per_osci_maxwell_modg_lastHeader.lua',                        
--            -- folder to write restart data to
            write = './restart/maxwell/',                                        
            -- temporal definition of restart write
            time_control = {   
              min = 0, 
              max = sim_control.time_control.max, 
              interval = sim_control.time_control.max/2.0
            }  
          }

-- DEBUG OPTIONS
--debug = { 
--         debugMode = true, 
--         debugFiles = true,
--         verbose = true,
--         dumpTreeIDs = true,
--         dumpDependencies = true,
--         checkSteps = true,
--         dumpState = true,
--         debugStates = { 
--                        write = {
--                                 folder    = './debug/',  -- the folder for the debug mesh 
--                                 interval  = 1,           -- dump restart file interval
--                                 tmin      = 1,           -- first timestep to output
--                                 tmax      = tmax+1       -- last timestep to output
--                                }
--                       }
--        }

-- timing settings (i.e. output for performance measurements, this table is otional)
timing = {
          folder = './',                  -- the folder for the timing results
          filename = 'timing.res'         -- the filename of the timing results
         }

-- Tracking of a certain variable at a certain point in space over time.           
eps = 1.0e-08
tracking = { 
             -- Give a name for the tracked object.
             label = 'trackIt', 
             -- the variable of the solver you track
             variable = {{'displacement_field'},{'magnetic_field'}},
             shape={  
                      kind = 'canoND', 
                      object = {
                                origin = { 
                                          (-1.0)*cubeLength/2.0 + eps,
                                          (-1.0)*cubeLength/2.0 + eps,
                                          (-1.0)*cubeLength/2.0 + eps,
                                         },
                                length = cubeLength/2.0 - 2.0*eps,
                                segments = { 100, 100, 100 } ,
                               }
                   },
             -- the temporal definition of the tracking: when do we start?
             time_control = {
               min = 0, 
               -- the maximum time: max = -1 assigns max=tmax
               max = sim_control.time_control.max, 
               interval = sim_control.time_control.max/2.0
             },     
             format = 'harvester',
             folder = './tracking/maxwell/'
         }

-- Equation definitions --
equation = {
    name   = 'maxwell',                   -- we solve maxwell's equations
    material     = {
                     -- The background permitivity and permeability
                     global = {
                                kind = 'const',
                                const = {1.0, 1.0, 0.0},
                              },
                   },
}

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg',            -- we use the modal discontinuous Galerkin scheme 
               m = 5 ,                    -- the maximal polynomial degree for each spatial direction
               }, 
    -- the temporal discretization scheme
    temporal = {
               name = --'explicitEuler',
               'explicitRungeKutta', 
               steps = 4,
               -- how to control the timestep
               control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.03,     -- CourantFriedrichsLewy number
                         },
               },
}

-- ...the initial condition table. 
-- ...initial condition function for displacement field (z component)
function ic_displacementZ(x,y,z)
  return displacementZ(x,y,z,0.0)
end
function ic_displacementY(x,y,z)
  return displacementY(x,y,z,0.0)
end

-- ...initial condition function for magnetic field (x component)
function ic_magneticX(x,y,z)
  return magneticX(x,y,z,0.0)
end
-- ...initial condition function for magnetic field (y component)
function ic_magneticY(x,y,z)
  return magneticY(x,y,z,0.0)
end
function ic_magneticZ(x,y,z)
  return magneticZ(x,y,z,0.0)
end

initial_condition = { 
                      displacement_fieldX = 0.0,     -- displacement field , x component
                      displacement_fieldY = ic_displacementY,  -- displacement field , z component
                      displacement_fieldZ = ic_displacementZ,  -- displacement field , z component
                      magnetic_fieldX = ic_magneticX,  -- magnetic induction , x component
                      magnetic_fieldY = ic_magneticY,  -- magnetic induction , y component
                      magnetic_fieldZ = ic_magneticZ,  -- magnetic induction , z component

                      projOverSampling = 10,          -- oversampling for quadrature projeciton of ini cond (per direction)
                    }

--function curDens(x, y, z, t)
--  -- calculate the minimal distance from the z axis
--  r= (x-0.001)^2
--  return {0.0,0.0,0.1*math.sin(w*t)*math.exp((-10.0)*r)}
--end
--sourceBoxLength = 0.0001
--source = {
--  currentDensity =          -- the name of the source
--  {
--    -- source terms will be active on a cube. The sources will be
--    -- active only on the inner most fine elements.
--    -- shape = all,-- {kind = 'canoND',  object = {
--    --                          origin={(-1)*cubeLength/2,
--    --                                  (-1)*cubeLength/2,
--    --                                  (-1)*cubeLength/2},
--    --                          vec= { 
--    --                                {cubeLength,0.0,0.0}, 
--    --                                {0.0,cubeLength,0.0}, 
--    --                                {0.0,0.0,cubeLength}, 
--    --                               },
--    --                          segments = { 100, 100, 100 }
--    --                        } 
--    -- },
--    fun = curDens
--  }
--}

