-- Configuration file for Ateles, solving acoustic equation with a source term
-- loacted in the domain  --

-- This is a configuration file for the Finite Volume / Discontinuous Galerkin Solver ATELES. 
-- It provides a testcase for the simulation of Acoustic equations in a homogenous media. The simulation domain
-- is a periodic cube with edge length 2.0. Therefore this is a very good way to verify your algorithmic implementations, 
-- since this testcase does not involve any boundary conditions. 
-- The testcase simulates the temporal development of standing waves for acoustic equation. Since we 
-- are considering a very simple domain, an analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- To verify diffusion and dispersion relations this testcases allows you to modify the spatial harmonics by
-- varying the integer mode number in x and y direction by modifying the lua variables m and n.
-- This testcase can be run in serial (only one execution unit) or in parallel (with multiple mpi ranks).
-- To calculate a grid convergence behavior please modify the level variable. An increment of one will half the radius of your elements.

--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
-- Parameters to vary --
degree =7
poly_space = 'Q'

timestep_info = 1

-- Check for Nans and unphysical values
check =  {
           interval = 1,
         }

-- ...the general projection table
projection = {
              kind = 'l2p',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattopoints = false  -- if lobatto points should be used, default = false,
                                     -- only working for chebyshev points --> fpt
              factor = 2.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
           -- blocksize = 32,        -- for fpt, default -1
           -- fftmultithread = false -- for fpt, logical, default false
             }

--...Configuration of simulation time
simulation_name = 'acoustic_source'  
sim_control = {
                time_control = { 
                  max =  {iter=100, sim=math.sqrt(2)},            -- Number of iterations / Simulated time
                  min = 0.0
                }
              }

--...Configuration for the restart file
restart = { 
--            -- file to restart from
--            read = './restart/maxwell/per_osci_maxwell_modg_lastHeader.lua',                        
--            -- folder to write restart data to
            write = 'restart/',
            -- temporal definition of restart write
            time_control = { 
              max =  math.sqrt(2),            -- Number of iterations / Simulated time
              min = 0.0,
              interval = {iter =10}
            }
          }

-- End  Parameters to vary --
--------------------------------------------------------------------------------
-- Definition of the test-case.

-- Mesh definitions --
-- ...the uniform refinement level for the periodic cube
level = 2
-- ...the length of the cube
cubeLength = 2.0
mesh = { predefined = 'cube',
         origin = { 
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0
                  },
         length = cubeLength,
         refinementLevel = level
       }

-- Equation definitions --
equation = {
             name   = 'acoustic',
             background = {
                 density = 1.225, 
                 velocityX = 0.0,
                 velocityY = 0.0,
                 velocityZ = 0.0,
                 pressure = 100000.0
                 }
           }

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg',            -- we use the modal discontinuous Galerkin scheme 
               m =  degree,                   -- the maximal polynomial degree for each spatial direction
               modg_space = poly_space
               }, 
    -- the temporal discretization scheme
    temporal = {
                  name = 'explicitRungeKutta', 
                  steps = 4,
                 -- how to control the timestep
                 control = {
                             name = 'cfl',   -- the name of the timestep control mechanism
                             cfl  = 0.95,     -- CourantÐFriedrichsÐLewy number
                            },
               },
             }

-- Initial Condition definitions --
initial_condition = { 
                      density = 0.0, --equation.background.density, 
                      velocityX = 0.0, -- equation.background.velocityX,
                      velocityY = 0.0, -- equation.background.velocityY,
                      velocityZ = 0.0, -- equation.background.velocityZ
                    }

-- Source term
-- ... radius of the source
r=0.2
-- ... parameters for the analytic solution
freq = ( 2.0*math.pi ) *10
-- ... the temporal period of the waveguide
T = 2.0*math.pi/freq
-- ... amplitude for the pertubation of denisty
ampl=equation.background.density/math.sqrt(equation.background.pressure/equation.background.density)

-- Source term definition, i.e. in Maxwell equations we are talking about space charges and 
-- current densities. In general they can depend on spatial coordinates and time.
function mass(x, y, z, t)
 -- d = math.sqrt(x*x+y*y+z*z)
 -- if d <= r then
  if math.abs(x) <= r then
    f = equation.background.density*ampl* math.exp(-(x*x)/0.08 * math.log(2)) * math.sin(freq*t)
    return {f}
  else
    return {0.0}
  end
end

source = {
  massInjection = {
    fun = mass,
    shape = {
      kind = 'canoND', 
      object = {
        origin = { -0.2, -0.2, -0.2 },
        vec = {
          { 0.4, 0.0, 0.0 },
          { 0.0, 0.4, 0.0 },
          { 0.0, 0.0, 0.4 }
        },
        segments = { 50, 50, 50 }
      } 
    }
  }
}
