-- Configuration file for Ateles (Periodic Oscillator) --

-- This is a configuration file for the Finite Volume / Discontinuous Galerkin
-- Solver ATELES. 
-- It provides a testcase for the simulation of Acoustic equations in a
-- homogenous media. The simulation domain is a periodic cube with edge length
-- 2.0. Therefore this is a very good way to verify your algorithmic
-- implementations, since this testcase does not involve any boundary
-- conditions. The testcase simulates the temporal development of standing waves
-- for acoustic equation. Since we are considering a very simple domain, an
-- analytic solution is well known and given as Lua functions in this script.
-- Therefore we suggest this testcase to verify one of the following topics
-- ... algorihtmic correctness
-- ... spatial and temporal order of your scheme
-- ... diffusion and dispersion relations of your scheme
-- ... and many more.
-- To verify diffusion and dispersion relations this testcases allows you to
-- modify the spatial harmonics by varying the integer mode number in x and y
-- direction by modifying the lua variables m and n.
-- This testcase can be run in serial (only one execution unit) or in parallel
-- (with multiple mpi ranks).
-- To calculate a grid convergence behavior please modify the level variable.
-- An increment of one will half the radius of your elements.

--------------------------------------------------------------------------------
--------------------------------------------------------------------------------
-- Parameters to vary --
degree = 12
poly_space = 'Q'
-- ...the uniform refinement level for the periodic cube
level = 3 

-- Check for Nans and unphysical values
check =  {
           interval = 100,
         }

-- ...the general projection table
projection = {
              kind = 'fpt',  -- 'fpt' or 'l2p', default 'l2p'
              -- for fpt the  nodes are automatically 'chebyshev'
              -- for lep the  nodes are automatically 'gauss-legendre'
           -- lobattopoints = false  -- if lobatto points should be used, default = false,
                                     -- only working for chebyshev points --> fpt
              factor = 1.0,          -- dealising factpr for fpt, oversampling factor for l2p, float, default 1.0
           -- blocksize = 32,        -- for fpt, default -1
           -- fftmultithread = false -- for fpt, logical, default false
             }

--...Configuration of simulation time
sim_control = { 
                time_control = { max = 0.004, -- 0.007,  -- final Simulated time
                                 min = 0,
                                 interval = {iter = 100} 
                                }
              }

-- Tracking              
cubeLength = 2.0
eps=cubeLength/(2^(level+1))
tracking = {
             label = 'track_density',
             folder = './',
             variable = {'density'},
             shape = {kind = 'canoND', object= { origin = {0.5,0.5,0.0+eps}, 
                                               --  vec = { {cubeLength-eps,0.0,0.0}},
                                               --  segments = {200},
                                               --  distribution='equal'
                                                }
             },
             time_control = { max = sim_control.time_control.max,  -- final Simulated time
                             min = 0,
                             interval = sim_control.time_control.max/20.0 
                            },
             output = { format = 'ascii', ndofs = 1 }
           }
-- End  Parameters to vary --
--------------------------------------------------------------------------------
-- Definition of the test-case.

-- Mesh definitions --
-- ...the length of the cube
mesh = { predefined = 'slice',
         origin = { 
                    (-1.0)*cubeLength/2.0,
                    (-1.0)*cubeLength/2.0,
                  },
         length = cubeLength,
         refinementLevel = level
       }


-- Equation definitions --
equation = {
             name   = 'acoustic_2d',
             background = {
                 density = 1.225, 
                 velocityX = 0.0,
                 velocityY = 0.0,
                 pressure = 100000.0
                 }
           }

-- Scheme definitions --
scheme = {
    -- the spatial discretization scheme
    spatial =  {
               name = 'modg_2d',            -- we use the modal discontinuous Galerkin scheme 
               m =  degree,                   -- the maximal polynomial degree for each spatial direction
               modg_space = poly_space
               }, 
    -- the temporal discretization scheme
    temporal = {
                 name = 'explicitRungeKutta', 
                 steps = 4,
              -- how to control the timestep
                 control = {
                          name = 'cfl',   -- the name of the timestep control mechanism
                          cfl  = 0.95,     -- CourantÐFriedrichsÐLewy number
                         },
               },
             }


-- variables for gaussian pluse             
c = math.sqrt(equation.background.pressure / equation.background.density)
ampl_density= equation.background.density/c 

function ic_gauss_density(x,y)
d= x*x+y*y 
return( ampl_density * math.exp(-d/0.002*math.log(2)) )
end

-- Initial Condition definitions --
initial_condition = { density = ic_gauss_density,
                      velocityX = 0.0,
                      velocityY = 0.0,
                      velocityZ = 0.0,
                    }


--...Configuration for the restart file
--restart = { 
----            -- file to restart from
----            read = './restart/maxwell/per_osci_maxwell_modg_lastHeader.lua',                        
----            -- folder to write restart data to
--            write = 'restart_2d/',
--            -- temporal definition of restart write
--            time_control = { --max = sim_control.time_control.max,  -- final Simulated time
--                             min = 0,
--                             interval = sim_control.time_control.max/25.0 
--                            }
--          }
