! Copyright (c) 2013 Jens Zudrop <j.zudrop@grs-sim.de>
! Copyright (c) 2013 Harald Klimach <harald.klimach@uni-siegen.de>
! Copyright (c) 2013-2017 Peter Vitt <peter.vitt2@uni-siegen.de>
! Copyright (c) 2014 Verena Krupp <verena.krupp@uni-siegen.de>
! Copyright (c) 2016 Tobias Girresser <tobias.girresser@student.uni-siegen.de>
!
! Permission to use, copy, modify, and distribute this software for any
! purpose with or without fee is hereby granted, provided that the above
! copyright notice and this permission notice appear in all copies.
!
! THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHORS DISCLAIM ALL WARRANTIES
! WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
! MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR
! ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
! WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
! ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
! OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
! **************************************************************************** !

!> Routines to derive quantities from the state in the Euler equation system.
module atl_eqn_euler_1d_derive_module
  use env_module,           only: rk

  use atl_equation_module,  only: atl_equations_type

  implicit none

  private

  public :: atl_eqn_euler_1d_cons2prim
  public :: atl_eqn_euler_1d_prim2cons
  public :: atl_eqn_euler_1d_cons2prim_elems
  public :: atl_eqn_euler_1d_prim2cons_elems

contains

! ******************************************************************************!
  !> Convert primitive varibales to conservative variables.
  !!
  !! The interface has to comply to the abstract interface
  !! atl_equation_module#eqn_var_trafo "eqn_var_trafo".
  subroutine atl_eqn_euler_1d_prim2cons(equation, instate, outstate, material)
    ! --------------------------------------------------------------------------!
    !> Description of the equation system.
    class(atl_equations_type),   intent(in)    :: equation

    !> Primitive variables to convert. If outstate is not provided, conversion
    !! will take in place of instate.
    real(kind=rk),           intent(inout) :: instate(:,:)

    !> Converted variables.
    real(kind=rk), optional, intent(out)   :: outstate(:,:)

    !> The material information.
    real(kind=rk), optional, intent(in)    :: material(:,:)
    ! --------------------------------------------------------------------------!

    if(present(outstate)) then
      outstate(:,1) = instate(:,1)
      outstate(:,2) = instate(:,1) * instate(:,2)
      outstate(:,3) = instate(:,3) / (equation%euler%isen_coef - 1.0_rk)         &
        & + 0.5_rk * instate(:,1) * ( instate(:,2)**2 )
    else
      instate(:,3) = instate(:,3) / (equation%euler%isen_coef - 1.0_rk)         &
        & + 0.5_rk * instate(:,1) * ( instate(:,2)**2 )
      instate(:,2) = instate(:,1)*instate(:,2)
    end if
  end subroutine atl_eqn_euler_1d_prim2cons
! ******************************************************************************!


! ******************************************************************************!
  !> Convert conservative to primitive variables.
  !!
  !! The interface has to comply to the abstract interface
  !! atl_equation_module#eqn_var_trafo "eqn_var_trafo".
  subroutine atl_eqn_euler_1d_cons2prim(equation, instate, outstate, material)
    ! --------------------------------------------------------------------------!
    !> Description of the equation system.
    class(atl_equations_type),   intent(in)    :: equation

    !> Primitive variables to convert. If outstate is not provided, conversion
    !! will take in place of instate.
    real(kind=rk),           intent(inout) :: instate(:,:)

    !> Converted variables.
    real(kind=rk), optional, intent(out)   :: outstate(:,:)

    !> The material information.
    real(kind=rk), optional, intent(in)    :: material(:,:)

    ! --------------------------------------------------------------------------!

    if(present(outstate)) then
      outstate(:,1) = instate(:,1)
      outstate(:,2) = instate(:,2)/instate(:,1)
      outstate(:,3) = (equation%euler%isen_coef - 1.0_rk) * ( instate(:,3) &
        & - 0.5_rk * instate(:,1) * (outstate(:,2)**2 ) )
    else
      instate(:,2) = instate(:,2)/instate(:,1)
      instate(:,3) = (equation%euler%isen_coef - 1.0_rk) * ( instate(:,3) &
        & - 0.5_rk * instate(:,1) * ( instate(:,2)**2 ) )
    end if
  end subroutine atl_eqn_euler_1d_cons2prim
! ******************************************************************************!

! ******************************************************************************!
  !> Convert primitive varibales to conservative variables.
  !!
  !! The interface has to comply to the abstract interface
  !! atl_equation_module#eqn_var_trafo "eqn_var_trafo".
  subroutine atl_eqn_euler_1d_prim2cons_elems( equation, instate, outstate, &
    &                                          nElems                       )
    ! --------------------------------------------------------------------------!
    !> Description of the equation system.
    class(atl_equations_type),   intent(in)    :: equation

    !> Primitive variables to convert. If outstate is not provided, conversion
    !! will take in place of instate.
    real(kind=rk),           intent(inout) :: instate(:,:,:)

    !> Converted variables.
    real(kind=rk), optional, intent(out)   :: outstate(:,:,:)

    !> Number of elements to act on (first index in the state arrays).
    integer,                 intent(in)    :: nElems
    ! --------------------------------------------------------------------------!

    if (present(outstate)) then
      outstate(1:nElems,:,1) = instate(1:nElems,:,1)
      outstate(1:nElems,:,2) = instate(1:nElems,:,1)*instate(1:nElems,:,2)
      outstate(1:nElems,:,3) =                                            &
        & instate(1:nElems,:,3) / (equation%euler%isen_coef-1.0_rk)       &
        &   + 0.5_rk * instate(1:nElems,:,1) * ( instate(1:nElems,:,2)**2 )
    else
      instate(1:nElems,:,3) =                                             &
        & instate(1:nElems,:,3) / (equation%euler%isen_coef-1.0_rk)       &
        &   + 0.5_rk * instate(1:nElems,:,1) * ( instate(1:nElems,:,2)**2 )
      instate(1:nElems,:,2) = instate(1:nElems,:,1)*instate(1:nElems,:,2)
    end if
  end subroutine atl_eqn_euler_1d_prim2cons_elems
! ******************************************************************************!

! ******************************************************************************!
  !> Convert conservative to primitive variables.
  !!
  !! The interface has to comply to the abstract interface
  !! atl_equation_module#eqn_var_trafo "eqn_var_trafo".
  subroutine atl_eqn_euler_1d_cons2prim_elems( equation, instate, outstate, &
    &                                          nElems                       )
    ! --------------------------------------------------------------------------!
    !> Description of the equation system.
    class(atl_equations_type),   intent(in)    :: equation

    !> Primitive variables to convert. If outstate is not provided, conversion
    !! will take in place of instate.
    real(kind=rk),           intent(inout) :: instate(:,:,:)

    !> Converted variables.
    real(kind=rk), optional, intent(out)   :: outstate(:,:,:)

    !> Number of elements to act on (first index in the state arrays).
    integer,                 intent(in)    :: nElems
    ! --------------------------------------------------------------------------!

    if (present(outstate)) then
      outstate(1:nElems,:,1) = instate(1:nElems,:,1)
      outstate(1:nElems,:,2) = instate(1:nElems,:,2)/instate(1:nElems,:,1)
      outstate(1:nElems,:,3) = (equation%euler%isen_coef - 1.0_rk) &
        & * ( instate(1:nElems,:,3) - 0.5_rk * instate(1:nElems,:,1) &
        &   * (outstate(1:nElems,:,2)**2) )
    else
      instate(1:nElems,:,2) = instate(1:nElems,:,2)/instate(1:nElems,:,1)
      instate(1:nElems,:,3) = (equation%euler%isen_coef - 1.0_rk) &
        & * ( instate(1:nElems,:,3) - 0.5_rk * instate(1:nElems,:,1) &
        &   * (instate(1:nElems,:,2)**2) )
    end if
  end subroutine atl_eqn_euler_1d_cons2prim_elems
! ******************************************************************************!


end module atl_eqn_euler_1d_derive_module
