! Copyright (c) 2014-2016 Harald Klimach <harald.klimach@uni-siegen.de>
! Copyright (c) 2014-2017 Peter Vitt <peter.vitt2@uni-siegen.de>
! Copyright (c) 2015 Verena Krupp <verena.krupp@uni-siegen.de>
! Copyright (c) 2014 Jens Zudrop <j.zudrop@grs-sim.de>
! Copyright (c) 2016 Tobias Girresser <tobias.girresser@student.uni-siegen.de>
!
! Permission to use, copy, modify, and distribute this software for any
! purpose with or without fee is hereby granted, provided that the above
! copyright notice and this permission notice appear in all copies.
!
! THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHORS DISCLAIM ALL WARRANTIES
! WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
! MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR
! ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
! WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
! ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
! OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
! **************************************************************************** !

!> atl_init_projectio_module
!!
!! This module contains the subroutine for initilize the poly_project_datatype
!! First the poly_project_type is defined according to the header,
!! maxpolydegree, basistype

module atl_load_project_module
  use env_module,                  only: rk
  use flu_binding,                 only: flu_state
  use aot_table_module,            only: aot_table_open, &
    &                                    aot_table_close
  use tem_aux_module,              only: tem_abort
  use tem_logging_module,          only: logUnit
  use ply_dynArray_project_module, only: dyn_ProjectionArray_type, &
    &                                    ply_fill_dynProjectArray, &
    &                                    ply_prj_init_type, append

  use atl_materialPrp_module,      only: atl_material_type
  use atl_scheme_module,           only: atl_scheme_type,        &
    &                                    atl_modg_scheme_prp,    &
    &                                    atl_modg_2d_scheme_prp, &
    &                                    atl_modg_1d_scheme_prp

  use atl_boundary_module,         only: atl_level_boundary_type

  implicit none

  private

  public :: atl_load_projection

contains


! ******************************************************************************!
  subroutine atl_load_projection(minLevel, maxLevel, poly_proj_pos,  &
    &                            dynprojectArray, conf, scheme_list, &
    &                            source_projPos, boundary_list,      &
    &                            boundary_list_stab, ic_pos_list,    &
    &                            material_list                       )
    ! --------------------------------------------------------------------------!
    integer, intent(in)                 :: minLevel
    integer, intent(in)                 :: maxLevel
    integer, intent(inout)              :: poly_proj_pos(minlevel:maxLevel)
    type(dyn_ProjectionArray_type), intent(inout) :: dynprojectArray
    type(flu_State), intent(in)         :: conf
    type(atl_scheme_type), intent(in)   :: scheme_list(minlevel:maxLevel)
    type(atl_level_boundary_type), intent(inout) &
      & :: boundary_list(minlevel:maxLevel), &
      &    boundary_list_stab(minlevel:maxLevel)
    !> The levelwise positions of the projections that has to be used for the
    !! sources
    integer, intent(inout)              :: source_projPos(minlevel:maxLevel)
    integer, intent(inout), allocatable :: ic_pos_list(:)
    type(atl_material_type), intent(inout) :: material_list(minlevel:maxLevel)
    ! --------------------------------------------------------------------------!
    integer :: iLevel
    integer :: thandle
    ! --------------------------------------------------------------------------!
    allocate (ic_pos_list(minLevel:maxLevel))
    write(logUnit(1),*) 'Loading projection tables...'
    ! open the projection table
    call aot_table_open(L = conf,  thandle = thandle,  key = 'projection')
    if (thandle /=0) then
      ! load general projection
      call atl_load_general_projection( poly_proj_pos   = poly_proj_pos,   &
        &                               dynprojectArray = dynprojectArray, &
        &                               minLevel        = minLevel,        &
        &                               maxLevel        = maxLevel,        &
        &                               scheme_list     = scheme_list,     &
        &                               conf            = conf,            &
        &                               parent          = thandle          )
       ! check for source terms
       ! load individual projection method for source
       call atl_load_source_projection( minLevel         = minLevel,         &
         &                              maxLevel         = maxLevel,         &
         &                              source_projPos   = source_projPos,   &
         &                              dynprojectArray  = dynprojectArray,  &
         &                              general_pos_list = poly_proj_pos,    &
         &                              scheme_list      = scheme_list,      &
         &                              conf             = conf,             &
         &                              parent           = thandle           )

       ! load individual projection method for source
       call atl_load_ic_projection( minLevel         = minlevel,        &
         &                          maxLevel         = maxlevel,        &
         &                          ic_pos_list      = ic_pos_list,     &
         &                          dynprojectArray  = dynProjectArray, &
         &                          general_pos_list = poly_proj_pos,   &
         &                          scheme_list      = scheme_list,     &
         &                          conf             = conf,            &
         &                          parent           = thandle          )

       ! load individual projection method for boundary condition
       call atl_load_bc_projection( minLevel         = minLevel,         &
         &                          maxLevel         = maxLevel,         &
         &                          boundary_list    = boundary_list,    &
         &                          dynprojectArray  = dynprojectArray,  &
         &                          general_pos_list = poly_proj_pos,    &
         &                          scheme_list      = scheme_list,      &
         &                          conf             = conf,             &
         &                          parent           = thandle           )

       ! load individual projection method for boundary condition of the
       ! stabilization
       call atl_load_bc_projection( minLevel         = minLevel,           &
         &                          maxLevel         = maxLevel,           &
         &                          boundary_list    = boundary_list_stab, &
         &                          dynprojectArray  = dynprojectArray,    &
         &                          general_pos_list = poly_proj_pos,      &
         &                          scheme_list      = scheme_list,        &
         &                          conf             = conf,               &
         &                          parent           = thandle             )
       ! load individual projection method for material
       call atl_load_material_projection( minLevel         = minLevel,        &
         &                                maxLevel         = maxLevel,        &
         &                                material_list    = material_list,   &
         &                                dynprojectArray  = dynprojectArray, &
         &                                general_pos_list = poly_proj_pos,   &
         &                                scheme_list      = scheme_list,     &
         &                                conf             = conf,            &
         &                                parent           = thandle          )
    else
      ! only load the gneral one and copy all the other position pointers
      ! load general projection
      call atl_load_general_projection( poly_proj_pos   = poly_proj_pos,   &
        &                               dynprojectArray = dynprojectArray, &
        &                               minLevel        = minLevel,        &
        &                               maxLevel        = maxLevel,        &
        &                               scheme_list     = scheme_list,     &
        &                               conf            = conf,            &
        &                               parent          = thandle          )
      do iLevel = minLevel, maxLevel
        boundary_list(iLevel)%poly_proj_pos = poly_proj_pos(iLevel)
        boundary_list_stab(iLevel)%poly_proj_pos = poly_proj_pos(iLevel)
        source_projPos(iLevel) = poly_proj_pos(iLevel)
        ic_pos_list(iLevel) = poly_proj_pos(iLevel)
        material_list(iLevel)%poly_proj_pos = poly_proj_pos(iLevel)
      end do
    end if
    call aot_table_close(L = conf, thandle = thandle)
  end subroutine atl_load_projection
! ******************************************************************************!

! ******************************************************************************!
  subroutine atl_load_general_projection( poly_proj_pos, dynprojectArray,  &
    &                                     minLevel, maxLevel, scheme_list, &
    &                                     conf, parent                     )
    ! --------------------------------------------------------------------------!
    integer, intent(in)           :: minLevel
    integer, intent(in)           :: maxLevel
    integer, intent(inout) :: poly_proj_pos(minlevel:maxLevel)
    type(dyn_ProjectionArray_type), intent(inout) :: dynprojectArray
    type(atl_scheme_type), intent(in)   :: scheme_list(minlevel:maxLevel)
    type(flu_State), intent(in)   :: conf
    integer, intent(in)           :: parent
    ! --------------------------------------------------------------------------!
    integer :: iLevel
    ! --------------------------------------------------------------------------!

    do iLevel= minLevel, maxLevel
    write(logUnit(1),*) ''
    write(logUnit(1),*) '...for Level', iLevel

    select case(scheme_list(iLevel)%scheme)
      case (atl_modg_scheme_prp)
        call ply_fill_dynProjectArray(                                    &
          & proj_pos            = poly_proj_pos(iLevel),                  &
          & dyn_projectionArray = dynProjectArray,                        &
          & maxPolyDegree       = scheme_list(iLevel)%modg%maxPolyDegree, &
          & basisType           = scheme_list(iLevel)%modg%basisType,     &
          & conf                = conf,                                   &
          & parent              = parent                                  )

      case (atl_modg_2d_scheme_prp)
        call ply_fill_dynProjectArray(                                       &
          & proj_pos            = poly_proj_pos(iLevel),                     &
          & dyn_projectionArray = dynProjectArray,                           &
          & maxPolyDegree       = scheme_list(iLevel)%modg_2d%maxPolyDegree, &
          & basisType           = scheme_list(iLevel)%modg_2d%basisType,     &
          & conf                = conf,                                      &
          & parent              = parent                                     )

      case (atl_modg_1d_scheme_prp)
        call ply_fill_dynProjectArray(                                       &
          & proj_pos            = poly_proj_pos(iLevel),                     &
          & dyn_projectionArray = dynProjectArray,                           &
          & maxPolyDegree       = scheme_list(iLevel)%modg_1d%maxPolyDegree, &
          & basisType           = scheme_list(iLevel)%modg_1d%basisType,     &
          & conf                = conf,                                      &
          & parent              = parent                                     )

      case default
        write(logUnit(1),*) 'ERROR in init_projection: unknown scheme name '
        write(logUnit(1),*) 'Stopping....'
        call tem_abort()

      end select

    end do
  end subroutine atl_load_general_projection
! ******************************************************************************!


! ******************************************************************************!
  subroutine atl_load_bc_projection(minLevel, maxLevel, boundary_list,      &
    &                               dynprojectArray, general_pos_list,conf, &
    &                               scheme_list, parent                     )
    ! --------------------------------------------------------------------------!
    integer, intent(in)           :: minLevel
    integer, intent(in)           :: maxLevel
    type(atl_level_boundary_type), intent(inout) :: &
      & boundary_list(minlevel:maxLevel)
    type(dyn_ProjectionArray_type), intent(inout) :: dynprojectArray
    integer, intent(in)           :: general_pos_list(minLevel:maxLevel)
    type(atl_scheme_type), intent(in) :: scheme_list(minlevel:maxLevel)
    type(flu_State), intent(in)   :: conf
    integer, intent(in)           ::parent
    ! --------------------------------------------------------------------------!
    integer :: iLevel
    integer :: thandle
    ! --------------------------------------------------------------------------!
    write(logUnit(1),*) ''
    write(logUnit(1),*) 'Load the individual projection method for ' &
      & // 'polynomials for boundary condition...'

    call aot_table_open( L       = conf,                &
      &                  parent  = parent,              &
      &                  thandle = thandle,             &
      &                  key     = 'boundary_condition' )

    if (thandle /= 0) then ! --> there is a projection table for source terms
      ! levelwise we open the projection table and fill the projection_init_type
      ! to append the DA for proejction.
      do iLevel= minLevel, maxLevel
        select case(scheme_list(iLevel)%scheme)
          case (atl_modg_scheme_prp)
            call ply_fill_dynProjectArray(                                    &
              & proj_pos            = boundary_list(iLevel)%poly_proj_pos,    &
              & dyn_projectionArray = dynProjectArray,                        &
              & maxPolyDegree       = scheme_list(iLevel)%modg%maxPolyDegree, &
              & basisType           = scheme_list(iLevel)%modg%basisType,     &
              & conf                = conf,                                   &
              & parent              = thandle                                 )
          case (atl_modg_2d_scheme_prp)
            call ply_fill_dynProjectArray(                                   &
              & proj_pos            = boundary_list(iLevel)%poly_proj_pos,   &
              & dyn_projectionArray = dynProjectArray,                       &
              & maxPolyDegree       = scheme_list(iLevel)%modg_2d            &
              &                                          %maxPolyDegree,     &
              & basisType           = scheme_list(iLevel)%modg_2d%basisType, &
              & conf                = conf,                                  &
              & parent              = thandle                                )
          case (atl_modg_1d_scheme_prp)
            call ply_fill_dynProjectArray(                                   &
              & proj_pos            = boundary_list(iLevel)%poly_proj_pos,   &
              & dyn_projectionArray = dynProjectArray,                       &
              & maxPolyDegree       = scheme_list(iLevel)%modg_1d            &
              &                                          %maxPolyDegree,     &
              & basisType           = scheme_list(iLevel)%modg_1d%basisType, &
              & conf                = conf,                                  &
              & parent              = thandle                                )
        end select
      end do
    else
      ! There is no projection table, means the position of the general
      ! projection for all level is used
      ! position should be stored in the source datatype
      write(logUnit(1),*) ' No individual projection table for boundary'//&
        &                 ' condition is provided, using general projection.'
      ! store the position of the general projection method
      boundary_list(:)%poly_proj_pos = general_pos_list(:)
    end if

    call aot_table_close(L=conf, thandle=thandle)

  end subroutine atl_load_bc_projection
! ******************************************************************************!


! ******************************************************************************!
  subroutine atl_load_source_projection( minLevel, maxLevel, source_projPos,  &
    &                                    dynprojectArray,                     &
    &                                    general_pos_list, conf, scheme_list, &
    &                                    parent                               )
    ! --------------------------------------------------------------------------!
    integer, intent(in)           :: minLevel
    integer, intent(in)           :: maxLevel
    !> The levelwise positions of the projections that has to be used for the
    !! sources
    integer, intent(inout)        :: source_projPos(minlevel:maxLevel)
    type(dyn_ProjectionArray_type), intent(inout) :: dynprojectArray
    integer, intent(in)           :: general_pos_list(minLevel:maxLevel)
    type(atl_scheme_type), intent(in) :: scheme_list(minlevel:maxLevel)
    type(flu_State), intent(in)   :: conf
    integer, intent(in)           ::parent
    ! --------------------------------------------------------------------------!
    integer :: iLevel
    integer :: thandle
    ! --------------------------------------------------------------------------!
    write(logUnit(1),*) ''
    write(logUnit(1),*) 'Load the individual projection method for ' &
      & // 'polynomials for source terms...'

    call aot_table_open( L       = conf,          &
      &                  parent  = parent,        &
      &                  thandle = thandle,       &
      &                  key     = 'source_terms' )

    if (thandle /= 0) then ! --> there is a projection table for source terms

      ! levelwise we open the projection table and fill the projection_init_type
      ! to append the DA for proejction.
      do iLevel= minLevel, maxLevel
        select case(scheme_list(iLevel)%scheme)
          case (atl_modg_scheme_prp)
            call ply_fill_dynProjectArray(                                    &
              & proj_pos            = source_projPos(iLevel),                 &
              & dyn_projectionArray = dynProjectArray,                        &
              & maxPolyDegree       = scheme_list(iLevel)%modg%maxPolyDegree, &
              & basisType           = scheme_list(iLevel)%modg%basisType,     &
              & conf                = conf,                                   &
              & parent              = thandle                                 )

          case (atl_modg_2d_scheme_prp)
            call ply_fill_dynProjectArray(                                   &
              & proj_pos            = source_projPos(iLevel),                &
              & dyn_projectionArray = dynProjectArray,                       &
              & maxPolyDegree       = scheme_list(iLevel)%modg_2d            &
              &                                          %maxPolyDegree,     &
              & basisType           = scheme_list(iLevel)%modg_2d%basisType, &
              & conf                = conf ,                                 &
              & parent              = thandle                                )

          case (atl_modg_1d_scheme_prp)
            call ply_fill_dynProjectArray(                                   &
              & proj_pos            = source_projPos(iLevel),                &
              & dyn_projectionArray = dynProjectArray,                       &
              & maxPolyDegree       = scheme_list(iLevel)%modg_1d            &
              &                                          %maxPolyDegree,     &
              & basisType           = scheme_list(iLevel)%modg_1d%basisType, &
              & conf                = conf ,                                 &
              & parent              = thandle                                )

        end select
      end do
    else
      ! There is no projection table, means the position of the general
      ! projection for all level is used
      ! position should be stored in the source datatype
      write(logUnit(1),*) ' No individual projection table for source terms' &
        & //              ' is provided, using general projection.'
      ! store the position of the general projection method
      source_projPos(:) = general_pos_list(:)
    end if

    call aot_table_close(L=conf, thandle=thandle)

  end subroutine atl_load_source_projection
! ******************************************************************************!


! ******************************************************************************!
  subroutine atl_load_ic_projection( minLevel, maxLevel, ic_pos_list,        &
    &                                dynprojectArray, general_pos_list,conf, &
    &                                scheme_list, parent                     )
    ! --------------------------------------------------------------------------!
    integer, intent(in)           :: minLevel
    integer, intent(in)           :: maxLevel
    integer, intent(inout)        :: ic_pos_list(minLevel:maxLevel)
    type(dyn_ProjectionArray_type), intent(inout) :: dynprojectArray
    integer, intent(in)             :: general_pos_list(minLevel:maxLevel)
    type(atl_scheme_type), intent(in)   :: scheme_list(minlevel:maxLevel)
    type(flu_State), intent(in)     :: conf
    integer, intent(in)             :: parent
    ! --------------------------------------------------------------------------!
    integer :: iLevel
    integer :: thandle
    ! --------------------------------------------------------------------------!
    write(logUnit(1),*) ''
    write(logUnit(1),*) 'Load the individual projection method for ' &
      & // 'polynomials for initial condition...'

    call aot_table_open( L       = conf,               &
      &                  parent  = parent,             &
      &                  thandle = thandle,            &
      &                  key     = 'initial_condition' )

    if (thandle /= 0) then ! --> there is a projection table for initial_condition
    ! levelwise we fill  the projection_init_type and to append the DA for
    ! proejction.

      do iLevel= minLevel, maxLevel
        select case(scheme_list(iLevel)%scheme)
          case (atl_modg_scheme_prp)
            call ply_fill_dynProjectArray(                                    &
              & proj_pos            =  ic_pos_list(iLevel),                   &
              & dyn_projectionArray = dynProjectArray,                        &
              & maxPolyDegree       = scheme_list(iLevel)%modg%maxPolyDegree, &
              & basisType           = scheme_list(iLevel)%modg%basisType,     &
              & conf                = conf,                                   &
              & parent              = thandle                                 )
          case (atl_modg_2d_scheme_prp)
            call ply_fill_dynProjectArray(                                   &
              & proj_pos            = ic_pos_list(iLevel),                   &
              & dyn_projectionArray = dynProjectArray,                       &
              & maxPolyDegree       = scheme_list(iLevel)%modg_2d            &
              &                                          %maxPolyDegree,     &
              & basisType           = scheme_list(iLevel)%modg_2d%basisType, &
              & conf                = conf ,                                 &
              & parent              = thandle                                )
          case (atl_modg_1d_scheme_prp)
            call ply_fill_dynProjectArray(                                   &
              & proj_pos            =  ic_pos_list(iLevel),                  &
              & dyn_projectionArray = dynProjectArray,                       &
              & maxPolyDegree       = scheme_list(iLevel)%modg_1d            &
              &                                          %maxPolyDegree,     &
              & basisType           = scheme_list(iLevel)%modg_1d%basisType, &
              & conf                = conf,                                  &
              & parent              = thandle                                )
        end select
      end do
    else
      !If there is not projection table, the
      ! general projection for this level is used and the position should be
      ! stored oh the individuall position list
      write(logUnit(1),*) ' No individual projection table for initial' &
        & // ' condition provided, using general projection.'
      ! store the position of the general projection method
      ic_pos_list(:) = general_pos_list(:)
    end if

    call aot_table_close(L=conf, thandle=thandle)

  end subroutine atl_load_ic_projection
  ! ***************************************************************************!


  ! ***************************************************************************!
  subroutine atl_load_material_projection( minLevel, maxLevel, material_list, &
    &                                      dynprojectArray, general_pos_list, &
    &                                      conf, scheme_list, parent          )
    ! ------------------------------------------------------------------------!
    integer, intent(in)           :: minLevel
    integer, intent(in)           :: maxLevel
    type(atl_material_type), intent(inout) :: material_list(minlevel:maxLevel)
    type(dyn_ProjectionArray_type), intent(inout) :: dynprojectArray
    integer, intent(in)           :: general_pos_list(minLevel:maxLevel)
    type(atl_scheme_type), intent(in) :: scheme_list(minlevel:maxLevel)
    type(flu_State), intent(in)   :: conf
    integer, intent(in)           ::parent
    ! ------------------------------------------------------------------------!
    integer :: iLevel
    integer :: thandle
    integer :: matProjectPos
    integer :: genProjectPos
    real(kind=rk) :: matProjFact, State2MatFact
    type(ply_prj_init_type) :: proj_state2Mat
    ! ------------------------------------------------------------------------!
    write(logUnit(1),*) ''
    write(logUnit(1),*) 'Load the individual projection method for' &
      & // 'polynomials for material...'

    call aot_table_open(L = conf, parent = parent, thandle= thandle,&
      &                 key = 'material'                )
    if (thandle /=0) then ! --> there is a projection table for source terms

      ! levelwise we open the projection table and fill the projection_init_type
      ! to append the DA for proejction.
      do iLevel= minLevel, maxLevel
        select case(scheme_list(iLevel)%scheme)
          case (atl_modg_scheme_prp)
             call ply_fill_dynProjectArray( &
                &   proj_pos = material_list(iLevel)%poly_proj_pos,         &
                &   dyn_projectionArray = dynProjectArray,                  &
                &   maxPolyDegree = scheme_list(iLevel)%modg%maxPolyDegree, &
                &   basisType     = scheme_list(iLevel)%modg%basisType,     &
                &   conf =conf,                                             &
                &   parent = thandle                                        )

          case (atl_modg_2d_scheme_prp)
             call ply_fill_dynProjectArray( &
                &   proj_pos = material_list(iLevel)%poly_proj_pos,            &
                &   dyn_projectionArray = dynProjectArray,                     &
                &   maxPolyDegree = scheme_list(iLevel)%modg_2d%maxPolyDegree, &
                &   basisType     = scheme_list(iLevel)%modg_2d%basisType,     &
                &   conf =conf ,                                               &
                &   parent = thandle                                           )

          case (atl_modg_1d_scheme_prp)
             call ply_fill_dynProjectArray(                                    &
                &   proj_pos = material_list(iLevel)%poly_proj_pos,            &
                &   dyn_projectionArray = dynProjectArray,                     &
                &   maxPolyDegree = scheme_list(iLevel)%modg_1d%maxPolyDegree, &
                &   basisType     = scheme_list(iLevel)%modg_1d%basisType,     &
                &   conf =conf ,                                               &
                &   parent = thandle                                           )
        end select

        ! Copy the material transformation and change the factor and the
        ! polynomial degree to match with the parameter of the state
        ! transformation.
        write(logUnit(1),*) 'Preparing special state to material transformation'
        matProjectPos = material_list(iLevel)%poly_proj_pos
        genProjectPos = general_pos_list(iLevel)
        proj_state2Mat = dynProjectArray%val( matProjectPos )
        if (proj_state2Mat%header%kind == 'fpt') then
          matProjFact = dynProjectArray%val(matProjectpos)%header &
            &                          %fpt_header%factor
        else
          matProjFact = dynProjectArray%val(matProjectpos)%header &
            &                          %l2p_header%factor
        end if
        State2MatFact = matProjFact * real(dynProjectArray%val(matProjectPos) &
          &                                               %maxPolyDegree,     &
          &                                kind=rk)                           &
          &                         / real(dynProjectArray%val(genProjectPos) &
          &                                               %maxPolyDegree,     &
          &                                kind=rk)
        if (proj_state2Mat%header%kind == 'fpt') then
          proj_state2Mat%header%fpt_header%factor = State2MatFact
        else
          proj_state2Mat%header%l2p_header%factor = State2MatFact
        end if

        proj_state2Mat%maxPolyDegree = dynProjectArray%val(genProjectPos) &
          &                                           %maxPolyDegree

        call append( me  = dynProjectArray,                              &
          &          val = proj_state2Mat,                               &
          &          pos = material_list(iLevel)%poly_proj_pos_state2Mat )

        write(logUnit(1),*) ' * polydegree: ', proj_state2Mat%maxPolyDegree
        if (proj_state2Mat%header%kind == 'fpt') then
          write(logUnit(1),*) ' * FPT Oversamp factor: ', &
            &                 proj_state2Mat%header%fpt_header%factor
        else
          write(logUnit(1),*) ' * L2P Oversamp factor: ', &
            &                 proj_state2Mat%header%l2p_header%factor
        end if

      end do

    else

      ! There is no projection table, means the position of the general
      ! projection for all level is used
      ! position should be stored in the source datatype
      write(logUnit(1),*) ' No individual projection table for material'//&
        &                 ' is provided, using general projection.'
      ! store the position of the general projection method
      material_list(:)%poly_proj_pos = general_pos_list(:)
      material_list(:)%poly_proj_pos_state2Mat = general_pos_list(:)

    end if

    call aot_table_close(L=conf, thandle=thandle)

  end subroutine atl_load_material_projection
  ! ***************************************************************************!

end module atl_load_project_module
