--------------------------------------------------------------------------------
-- This is a sample Ateles configuration file                                 --
-- It makes use of the Lua scripting language, and you can have a look at its --
-- syntax at http://www.lua.org                                               --
-- Its main purpose is to show the options you might want to set for your     --
-- simulation. It thus serves as a little documentation for the configuration.--
-- It is supposed to always work with the current Ateles version              --
--                                                                            --
-- WARNING!!!                                                                 --
-- LEAVE THIS FILE UNCHANGED IN THE REPOSITORY                                --
-- unless you change configurable options, in this case you should update this--
-- script accordingly.                                                        --
--                                                                            --
-- This script should always serve as an example which runs "out of the box". --
-- Be aware that you have to provide the mesh and the folder for the restart  --
-- and tracking output.                                                       --
--                                                                            --
-- Thank you!                                                                 --
--------------------------------------------------------------------------------
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- Load general configuration options from the treelm library.
-- Includes definition of the mesh!
require 'treelm/config'
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- GLOBAL SIMULATION OPTIONS
-- with the simulation name you can put a label on your simulation run, to have
-- some concise reference.
simulation_name = 'sample'

sim_control = { -- controlling the execution
  time_control = {
    -- The time can be given in terms of iterations, sim time
    -- and wall clock.

    max = 1.0, -- Point in time when to stop the simulation.
               -- If not a table, this refers to the sim time.
               -- Whatever happens first will define the
               -- end of the simulation:
               -- max = {sim = fintime, iter = 1000, clock = 3600}

    interval = { iter = 1 }, -- The interval is used to define the
                             -- frequency at which information about
                             -- the current time is printed to the
                             -- screen.
                             -- Like max this is a time setting, and
                             -- can be provided in terms of the
                             -- three different definitions.

    -- Controlling the trigger status of the time control, and
    -- status of the simulation can involve communication and
    -- might have a performance impact if done every iteration
    -- and single iterations are too short.
    -- By setting check_iter the interval at which
    -- these checks are done, can be increased, and
    -- thus performance impacts reduced.
    -- CAUTION: be aware that increasing this to values
    --          larger than 1, decreases the accuracy for the
    --          time controlling, as only every check_iter
    --          iteration will do a check on the current time.
    --          Typically it is not necessary to increase this.
    -- default:
    -- check_iter = 1
  },

  abort_criteria = {
    -- Criteria upon which the simulation should be stopped.
    stop_file = '' -- Stop file, can be used to signal the simulation to
                   -- gracefully end.
                   -- If this is empty (the default), this is disabled.
                   -- Non-empty settings here, will cause the simulation to
                   -- come to an end when a file of this name exists in the
                   -- working directory.
                   -- Thus, with stop_file = 'stop', you can cause the
                   -- simulation to stop and write a restart by doing
                   -- "touch stop" in the working directory of the run.
                   -- Empty files like the one created by touch, will be
                   -- deleted by Ateles. If you want to keep the file, it
                   -- needs to have some content. You can achieve this for
                   -- example by "echo keep > stop".
  }
}

-- the name of the timing file can be set here, defaults to 'timing.res'
timing = { filename = 'timing.res' }
--------------------------------------------------------------------------------


--------------------------------------------------------------------------------
-- DESCRIPTION OF THE PROBLEM TO SOLVE
-- The analytic solution for this Maxwell testcase is given by the following
-- functions (only correct for epsi = mu = 1, used for IC below):
-- ... definition of temporal angular frequency
amplX = 1.0
amplY = amplX
w = math.sqrt(amplX^2 + amplY^2)
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- VARIABLE TABLE
-- This table contains user defined variables. These variables can be used to be
-- the data source for source terms or material parameters, but it's also
-- possible to define variables consisting of different other quantities to
-- use them for tracking.
function mass(x, y, z, t)
  return {1.1*x, 2.2*y, 3.3*z}
end
--function variableConductivity(x, y, z, t)
--  return conductivity
--end

electricX = 0.0
electricY = 0.0
-- ... E_z(x,y,z,t) = sin(amplX \pi x) sin(amplY \pi y) cos(w t)
function electricZ(x,y,z,t)
  return math.sin(amplX*math.pi*x)*math.sin(amplY*math.pi*y)*math.cos(w*t)
end
-- ... B_x(x,y,z,t) = -\frac{\pi n}{w} sin(m \pi x) cos(n \pi y) sin(w t)
function magneticX(x,y,z,t)
  return (-1.0)*(math.pi*amplY/w)*math.sin(amplX*math.pi*x)*math.cos(amplY*math.pi*y)*math.sin(w*t)
end
-- ... B_y(x,y,z,t) = \frac{\pi m}{w} cos(m \pi x) sin(n \pi y) sin(w t)
function magneticY(x,y,z,t)
  return (math.pi*amplX/w)*math.cos(amplX*math.pi*x)*math.sin(amplY*math.pi*y)*math.sin(w*t)
end
-- ... B_z = 0.0
magneticZ = 0.0
-- ...initial condition function for electric field (z component)
function ic_electricZ(x,y,z)
  return electricZ(x, y, z, 0.0)
end

variable = {
  {
    name = "mass",
    ncomponents = 3,
    vartype = "st_fun",
    st_fun = {
      fun = mass
--      shape = {
--        kind = 'canoND',
--        object= {
--          origin = { -0.2, -0.2, -0.2 },
--          vec = { { 0.4, 0.0, 0.0 }, { 0.0, 0.4, 0.0 }, { 0.0, 0.0, 0.4 } },
--          segments = { 50, 50, 50 }
--        }
--      }
    }
  },
  -- This is the global material for Maxwell. It consists of three different
  -- components, permeability, permittivity, and conductivity, each a scalar, so
  -- that we need three scalars for this equation system.
  -- As this is the global fallback material, we define each material to be a
  -- neutral term, which in this case is 0.
  --{
  --  name = "global_maxwell_permeability",
  --  ncomponents = 1,
  --  vartype = "st_fun",
  --  st_fun = { const = mu }
  --},
  --{
  --  name = "global_maxwell_permittivity",
  --  ncomponents = 1,
  --  vartype = "st_fun",
  --  st_fun = { const = epsi }
  --},
  --{
  --  name = "global_maxwell_conductivity",
  --  ncomponents = 1,
  --  vartype = "st_fun",
  --  st_fun = {
  --    fun = variableConductivity,
  --  }
  --}
  -- This is the global material for Maxwell with divergence correction. It
  -- consists of four different components, permeability, permittivity, gamma,
  -- and chi.
  -- As this is the global fallback material, we define each material to be a
  -- neutral term, which in this case is 1.
  --{
  --  name = "global_maxwellDivCorrection_material",
  --  ncomponents = 4,
  --  vartype = "st_fun",
  --  st_fun = {
  --    const = { 1.0, 1.0, 1.0, 1.0 }
  --  }
  --}
  -- This is the global material for Euler 1D. It consists of three different
  -- components, characteristics with one scalar, relax_velocity with one
  -- scalar and relax_temperature with one scalar, thus we need three scalars
  -- for this equation system.
  -- As this is the global fallback material, we define each material to be a
  -- neutral term, which in this case is 0.
  --{
  --  name = "global_euler1D_characteristic",
  --  ncomponents = 1,
  --  vartype = "st_fun",
  --  st_fun = { const = 0.0 }
  --},
  --{
  --  name = "global_euler1D_relax_velocity",
  --  ncomponents = 1,
  --  vartype = "st_fun",
  --  st_fun = { const = 0.0 }
  --},
  --{
  --  name = "global_euler1D_relax_temperature",
  --  ncomponents = 1,
  --  vartype = "st_fun",
  --  st_fun = { const = 0.0 }
  --}
  -- This is the global material for Euler 2D. It consists of three different
  -- components, characteristics with one scalar, relax_velocity with two
  -- scalars and relax_temperature with one scalar, thus we need four scalars
  -- for this equation system.
  -- As this is the global fallback material, we define each material to be a
  -- neutral term, which in this case is 0.
  --{
  --  name = "global_euler2D_characteristic",
  --  ncomponents = 1,
  --  vartype = "st_fun",
  --  st_fun = { const = 0.0 }
  --},
  --{
  --  name = "global_euler2D_relax_velocity",
  --  ncomponents = 2,
  --  vartype = "st_fun",
  --  st_fun = {
  --    const = { 0.0, 0.0 }
  --  }
  --},
  --{
  --  name = "global_euler2D_relax_temperature",
  --  ncomponents = 1,
  --  vartype = "st_fun",
  --  st_fun = { const = 0.0 }
  --}
  -- This is the global material for Euler 3D. It consists of three different
  -- components, characteristics with one scalar, relax_velocity with three
  -- scalars and relax_temperature with one scalar, thus we need five scalars
  -- for this equation system.
  -- As this is the global fallback material, we define each material to be a
  -- neutral term, which in this case is 0.
  --{
  --  name = "global_euler3D_characteristic",
  --  ncomponents = 1,
  --  vartype = "st_fun",
  --  st_fun = { const = 0.0 }
  --},
  --{
  --  name = "global_euler3D_relax_velocity",
  --  ncomponents = 3,
  --  vartype = "st_fun",
  --  st_fun = {
  --    const = { 0.0, 0.0, 0.0 }
  --  }
  --},
  --{
  --  name = "global_euler3D_relax_temperature",
  --  ncomponents = 1,
  --  vartype = "st_fun",
  --  st_fun = { const = 0.0 }
  --}
  -- @TODO PV 20151204: Fix the penalization
  -- This is the global fallback penalization for euler 3D equation. It provides
  -- five scalar values for all state components.
  --{
  --  name = 'euler3D_penalization',
  --  ncomponents = 5,
  --  vartype = "st_fun",
  --  st_fun = { const = { 0, 0, 0, 0, 0 } }
  --}
  -- This is the global fallback penalization for euler 2D equation. It provides
  -- four scalar values for all state components.
  --{
  --  name = 'euler2D_penalization',
  --  ncomponents = 4,
  --  vartype = "st_fun",
  --  st_fun = { const = { 0, 0, 0, 0 } }
  --}
  -- This is the global fallback penalization for euler 1D equation. It provides
  -- three scalar values for all state components.
  --{
  --  name = 'euler1D_penalization',
  --  ncomponents = 3,
  --  vartype = "st_fun",
  --  st_fun = { const = { 0, 0, 0 } }
  --}
}
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- EQUATION DEFINITIONS
-- Euler Equations
--equation = {
--  name = 'euler',              -- we solve euler equations
--  numflux = 'lax_friedrich',   -- numerical flux to use
--                               -- possible are 'lax_friedrich', 'godunov' or 'hll'
--  isen_coef = 1.4,             -- isentropic coefficient
--  r = 1.0,                     -- ideal gas constant
--  linear_limit = 0.0,          -- Allow the usage of a linearized physical
--                               -- flux.
--                               -- If this is set to a value >0, elements with
--                               -- a small indication will be treated with a
--                               -- linearized flux computation.
--
--  linearization_indicator = 'error'
--                               -- What kind of indicator to use to decide
--                               -- the linearization.
--                               -- Default is to use an error estimate.
--                               -- Alternatives are some simpler estimations:
--                               -- 'density': decide linearization just based
--                               --            on relative variation in density
--                               -- 'energy': decide linearization just based
--                               --           on relative variation in energy
--                               -- The 'error' indicator involves all state
--                               -- variables and should be the most accurate
--                               -- option.
--
--  ensure_positivity = .false., -- Limit the state to be used in the flux
--                               -- computation to positive values only.
--                               -- Only modes up to the point where the
--                               -- positivity of the polynomial can be
--                               -- guaranteed will be used for density and
--                               -- energy for computing the fluxes.
--                               -- Also the mean values on the surface that
--                               -- are to be used for numerical fluxes will
--                               -- be bound to positive values.
--                               -- Not that this decreases the accuracy in
--                               -- elements, where the state would assume
--                               -- negative density or energy, and the
--                               -- limitation will act quite early on as the
--                               -- minimum estimation is rather coarse.
--  material = {
--    mode_reduction = true, -- Enables mode reduction. When the material is
--                           -- constant for an element (volume and faces),
--                           -- computational effort can be reduced by only
--                           -- using the first mode for material.
--                           -- Default: false
--    reduction_threshold = 1, -- threshold to use for the material variable
--                             -- to decide whether to reduce the modes in
--                             -- flux computations.
--                             -- default is 1.
--    characteristic = 0.0,
--    relax_velocity = { 0.0, 0.0, 0.0 },
--    relax_temperature = 0.0
--  }
--}
---- Linear Euler Equations
--equation = {
--  name = 'linearEuler'         -- we solve linearized euler equations
--  numflux = 'godunov',         -- numerical flux to use
--                               -- possible are 'lax_friedrich'or 'godunov'
--  isen_coef = 1.4,             -- isentropic coefficient
--  background = {               -- background state
--    density = 1.225,           -- the equation system is linearized around
--    velocityX = 100.0,
--    velocityY = 0.0,
--    velocityZ = 0.0,
--    pressure = 100000.0
--  }
--}
---- Acoustic ( isothermal, background u should be 0)
--equation = {
--  name = 'acoustic'            -- we solve acoustic equations
--  background = {               -- background state
--    density = 1.225,           -- background velocity should be 0
--    velocityX = 0.0,
--    velocityY = 0.0,
--    velocityZ = 0.0,
--    pressure = 100000.0
--  }
--}
---- cv heat capacity
--equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)
--
-- Navier Stokes Equations
--equation = {
--  name = 'navier_stokes',      -- we solve navier stokes equations
--  mu = 1.831e-05,              -- viscosity  (default = 2.121996-314)
--  therm_cond = 2.55e-02,       -- thermal conductivity (default = 6.953195-310) (\todo does simlation work when therm_cond = 0 ?)
--  isen_coef = 1.4,             -- isentropic coefficient
--  r = 296.0,                   -- ideal gas constant
--  material = {
--    mode_reduction = true, -- Enables mode reduction. When the material is
--                           -- constant for an element (volume and faces),
--                           -- computational effort can be reduced by only
--                           -- using the first mode for material.
--                           -- Default: false
--    reduction_threshold = 1, -- threshold to use for the material variable
--                             -- to decide whether to reduce the modes in
--                             -- flux computations.
--                             -- default is 1.
--    characteristic = 0.0,
--    relax_velocity = { 0.0, 0.0, 0.0 },
--    relax_temperature = 0.0
--  }
--}
-- (cv) heat capacity and (r) ideal gas constant
--equation["cv"] = equation["r"] / (equation["isen_coef"] - 1.0)

-- Maxwell Equations
equation = {
  name = 'maxwell',-- we solve maxwell’s equations
  material = {
    mode_reduction = false, -- Enables mode reduction. When the material is
                            -- constant for an element (volume and faces),
                            -- computational effort can be reduced by only
                            -- using the first mode for material.
                            -- Note: Not yet implemented for maxwell!
                            -- Default: false
--    reduction_threshold = 1, -- threshold to use for the material variable
--                             -- to decide whether to reduce the modes in
--                             -- flux computations.
--                             -- default is 1.
    permeability = 1.0,
    permittivity = 1.0,
    conductivity = 0.0
  }
}


--
-- Filtered Navier Stokes Equations
--equation = {
--  name = 'filtered_navier_stokes',  -- we solve filtered navier stokes equations
--  mu = 1.831e-05,                   -- viscosity
--  therm_cond = 2.55e-02,            -- thermal conductivity
--  isen_coef = 1.4,                  -- isentropic coefficient
--  r = 296.0,                        -- ideal gas constant
--  nvar = 5,
--  -- Turbulence parameter for Smagorinsky model
--  turbulence_model = 'smagorinsky',
--  -- Germano-Lilli model
--  --turbulence_model = 'germano'
--  Cs = 0.04,                        -- suggested value between 0.1 and 0.2
--  Ci = 0.005,                       -- Value for Yoshizawa model
--  prandtl_sgs = 0.6,                -- suggested value
--  material = {
--    characteristic = 0.0,
--    relax_velocity = { 0.0, 0.0, 0.0 },
--    relax_temperature = 0.0
--  }
--}
--
-- Maxwell Equations with divergence correction
--equation = {
--  name = 'maxwellDivCorrection',    -- we solve maxwell’s equations with divergence correction
--  mu = perma,                       -- the magnetic permeability of vacuum
--  epsi = permit,                    -- the electric permitivity of vacuum
--  chi = 1.0,                        -- parameter for electric divergence cleaning
--  gam = 1.0,                        -- parameter for magnetic divergence cleaning
--  material = {
--    permeability = 1.0,
--    permittivity = 1.0,
--    conductivity = 0.0
--  }
--}
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- SCHEME DEFINITIONS
scheme = {
  spatial = {
-- Modal Discontinuous Galerkin (MODG)
    name = 'modg', -- Lower dimensions available as:
                   -- modg_2d
                   -- modg_1d
    modg_space = 'Q', -- Polynomial space to use (P or Q)
                   --                   Default is Q.
    m = 1, -- fixed polynomial degree for all levels
    -- OR
--    m = {
--      predefined = 'fixedFact' -- predefined degrees
--      -- fixedFact increases order by a fixed factor
--      factor = 2.0,     -- default = sqrt(2)
--      -- The default sqrt(2) factor results in roughly the
--      -- same time step restriction on all levels.
--      base_order = 1.0, -- default = 1.0
--      -- Polynomial degree on level ilevel is given by:
--      -- m(ilevel)
--      -- = nint(base_order*factor**(maxlevel-ilevel)-1.0)
--    },
      -- OR
--    -- specify the degree for each level in the mesh
--    m = { { level = 4, m = 15 } },
  },
-- Additional stabilization of the scheme, whenever necessary.
-- The stabilization filters are applied in order of specification.
--  stabilization = {
--    {
--      -- Spectral viscosity, i.e.:
--      -- u(x) = \sum_{k=0}^N \sigma_k \hat{u}_k L_k(x)
--      -- with
--      -- \sigma_k = exp(- \alpha (k/N)^order )
--      name = 'spectral_viscosity',
--      alpha = 36,
--      order = 60,
--      isAdaptive = true
--    },
--    {
--      -- Positivity preserving limiter:
--      -- eps denotes the limit below which density and pressure will be considered
--      -- as unphysical.
--      name = 'cons_positivity_preserv',
--      eps = 1.0e-08,
--    },
--  },
  temporal = {
    name = 'explicitRungeKutta',  -- we use explicit Runge-Kutta in time
    steps = 4,
    control = {                   -- how to control the timestep
      name = 'cfl',               -- the name of the timestep control mechanism
      cfl = 0.95,                 -- Courant Friedrichs Levy factor
      use_modal_estimate = false, -- For nonlinear equations: use a modal
                                  -- estimation for the dynamic timestep
                                  -- computation. This is less accurate (may do
                                  -- smaller timesteps), but avoids expensive
                                  -- transformations for the timestep
                                  -- computation.
      -- or to use a fixed timestep length:
      -- name = 'fixed',
      -- dt = 0.123
    }
  }
}
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- SPATIAL DISCRETIZATIONS

-- only modal DG schemes available!

-- TEMPORAL DISCRETIZATIONS

-- Explicit Runge Kutta
--temporal = {
--  name = 'explicitRungeKutta',  -- we use explicit runge kutta in time
--  steps = 4,                    -- we have a 4 stage
--  control = {                   -- how to control the timestep
--    name = 'cfl',      -- the name of the timestep control mechanism
--    cfl = 0.3          -- Courant Friedrichs Lewy number
--  }
--}

-- Strong Stability Preserving Explicit Runge Kutta
--temporal = {
--  name = 'explicitSSPRungeKutta',  -- we use ssp explicit runge kutta in time
--  steps = 2,                       -- we have a 2 stage
--  control = {                      -- how to control the timestep
--    name = 'cfl',         -- the name of the timestep control mechanism
--    cfl = 0.3             -- Courant Friedrichs Lewy number
--  }
--}

-- Explicit Euler
--temporal = {
--  name = 'explicitEuler',       -- we use explicit euler in time
--  control = {                   -- how to control the timestep
--    name = 'cfl',      -- the name of the timestep control mechanism
--    cfl = 0.3          -- Courant Friedrichs Lewy number
--  }
--}

--------------------------------------------------------------------------------
-- PROJECTION
-- In the projection table, the method for projection from nodal to modal space
-- for BC, IC and non-linear problems is given
projection = {
  -- kind = 'fpt',  -- 'fpt', 'fxt' or 'l2p',
  --                --  default: 'l2p'
  -- for fpt the  nodes are automatically 'chebyshev'
  -- for l2p and fxt the  nodes are automatically 'gauss-legendre'

  -- lobattoPoints = false  -- if lobatto points should be used,
                            -- default = false,
                            -- only working for Chebyshev points (fpt)!

  factor = 1.0,             -- dealising factor for fpt
  --                        -- oversampling factor to remove aliasing
  --                        -- effects by padding, default: 1
                            -- Note that for FXT an evenly oversampled
                            -- order is required, if this is not the
                            -- case, the next higher even order will be
                            -- used and the actual factor might
                            -- accordingly be higher.

  -- FXT settings:
  -- prec = 1.5e-8,         -- precision to use for the fast multipole
                            -- computation during initialization.
                            -- Defaults to sqrt of epsilon for double
                            -- precision numbers (1.5e-8).

  -- FPT settings:
  -- approx_terms = 18,     -- Number of terms used to approximate the
  --                        -- matrix multiplication for blocks, that
  --                        -- are detached from the diagonal.
  --                        -- The default of 18 is recommended for
  --                        -- double precision.
  -- blocksize = 64,        -- for FPT, default 64. The blocksize
  --                        -- defines how big the minimal block
  --                        -- should be that is approximated in
  --                        -- fast algorithm.
  --                        -- The smaller it is, the more operations
  --                        -- are merely approximated.
  --                        -- Recommended for double precision is a
  --                        -- setting of 64.
  --                        -- The fast algorithm will only be used
  --                        -- for m >= blocksize.
  --                        -- Note, that this has to be larger than
  --                        -- 2*approx_terms to provide any
  --                        -- reduction in operation counts.
  -- striplen = 256,        -- This provides the length for arrays to
  --                        -- apply the matrix operation to
  --                        -- simultaneously.
  --                        -- Default is the vlen from the tem_compileconf_module.
  -- subblockingWidth = 8   -- The subblockingWidth is used during the
  --                        -- unrolling of the diagonal multiplication
  --                        -- during the projection. By setting this
  --                        -- value to an appropriate value a better
  --                        -- cache usage can be achieved.
  --                        -- Default is 8
  -- adapt_factor_pow2 = true -- for FPT, default false. Should the
  --                          -- oversampling factor be adjusted to
  --                          -- obtain a power of 2 in the
  --                          -- oversampled order?
  -- fftMultiThread = false -- for FPT, default false. Should nested
  --                        -- multithreading be activated for FFTW?
  --
  -- individual projection methods for source terms
  -- if non is determined the general projection method is used
  source_terms = {
    -- the configuration parameter are similar to the
    -- general projection method
    -- kind = 'fpt',
    factor = 1.0
  },
  -- individual projection methods for initial condition
  -- if non is determined the general projection method is used
  initial_condition = {
    -- the configuration parameter are similar to the
    -- general projection method
    factor = 2.0
  },
  -- individual projection methods for boundary condition
  -- if non is determined the general projection method is used
  boundary_condition = {
    -- the configuration parameter are similar to the
    -- general projection method
    -- kind = 'fpt',
    factor = 2.0
  },
  -- individual projection methods for boundary condition
  -- if non is determined the general projection method is used
  material= {
    -- the configuration parameter are similar to the
    -- general projection method
    -- kind = 'fpt',
    factor = 2.0
  }
}

--------------------------------------------------------------------------------
-- INITIAL CONDITIONS
-- Each entry in the initial conditions are a given by a spatial function.
-- These might be predefined functions, Lua functions or just constants.
-- initial conditions for Maxwell
initial_condition = {
  displacement_fieldX = 0.0, -- electric field, x component
  displacement_fieldY = 0.0, -- electric field, y component
  displacement_fieldZ = ic_electricZ, -- electric field, z component
  magnetic_fieldX = 0.0, -- magnetic induction, x component
  magnetic_fieldY = 0.0, -- magnetic induction, y component
  magnetic_fieldZ = 0.0  -- magnetic induction, z component
--  -- for divergence correction
--  magneticCorrection = function or const. value,  -- magnetic div correction
--  electricCorrection =   function or const. value -- electric div correction
}

-- Initial conditions for Euler
--initial_condition = {
--  --density = 1.225,
--  density = {
--    predefined = 'gausspulse',
--    center = { centerX, centerY, centerZ }, -- Gauss pulse center
--    halfwidth = 2.0,               -- half width of gauss pulse from center
--    amplitude = 2.0,               -- height or magnitude of gauss pulse
--    background = 1.225             -- reference value. In case of density, it is reference density
--  },
--  pressure = 100000,
--  velocityX = velX,
--  velocityY = 0.0,
--  velocityZ = 0.0
--}

--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- BOUNDARY CONDITIONS
-- This minimal example is completely periodic and has no boundary conditions.
-- Each boundary condition is described by a kind, and space-time functions for
-- all required variables in the that boundary.
-- Space time functions are either predefined functions, Lua functions,
-- constants, or combined functions, where temporal and spatial parts are
-- superimposed.
-- If there are values that are to be extrapolated (Neumann boundary
-- condition), you can set enforce_zero_grad to true, to use an extrapolation
-- of a polynomial with zero gradient at the boundary.
-- This is achieved by computing the last mode to fulfill this condition.
-- If you set neumann_mode_fraction to a smaller value than 1, then only
-- this fraction of lower modes will be used in the enforce_zero_grad
-- procedure and higher modes will be set to 0.
--boundary_condition = {
--  -- INFLOW: subsonic: set two values (density, velocity) and extrapolate one value (pressure)
--  --         supersonic: set all values (density, velocity, pressure)
--  {
--    label = 'inflow',
--    kind = 'inflow',
--    pressure = 'extrapolate',
--    density = 1,
--    velocityX = 500,
--    velocityY = 0.0,
--    velocityZ = 0.0
--  },
--  -- OUTFLOW: subsonic: set one value (pressure) and extrapolate two values (density, velocity)
--  {
--    label = 'east',
--    kind = 'outflow',
--    pressure = 100001,
--    density = 'extrapolate'
--  },
--  -- NOSLIPWALL:
--  {
--    label = 'top',
--    kind = 'noslipwall',
--    velocityX = 0.0,
--    velocityY = 0.0,
--    velocityZ = 0.0
--  },
--  -- SLIPWALL
--  {
--    label = 'cylinder',
--    kind = 'slipwall',
--    velocityX = 0.0,
--    velocityY = 0.0,
--    velocityZ = 0.0
--  },
--  -- Periodic
--  -- don't specify peridic boundaries here!
--
--  -- PERFECTLY ELECTRIC CONDUCTING BOUNDARY (for Maxwell equation)
--  --{
--  --  label = 'innerCylinder',
--  --  kind = 'pec'
--  --}
--}
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
-- SOURCE TERM
--HK: Todo: actually put in a working example here. the one below does not
--          work, due the the comparison of math.abs(x) with r, which is not
--          defined!

source = {
  current_density = 'mass'
}
--source = {
--  charge = {             -- charge density
--    fun = …
--  },
--  currentDensityX = 0.0, -- current density, x component
--  currentDensityY = 0.0, -- current density, y component
--  currentDensityZ = {    -- current density, z component
--    fun = …
--  }
--}
--------------------------------------------------------------------------------

-- Subsampling for tracking, define a ply_sampling table to activate subsampling
-- for all tracking objects (except those with use_get_point).
-- Subsampling (for subresoloved color information):
ply_sampling = {
  nlevels = 1,    -- maximal level to use in subsampling
                  -- defaults to 0, which deactivates subsampling

  --method  = 'fixed', -- method to use for subsampling
                       -- 'adaptive': (recommended default) adaptive refinement
                       --             of the mesh based on solution
                       -- 'fixed': will refine all elements by nlevels and
                       --          evaluate the polynomials at the barycenter of
                       --          each fine element

  -- Parameters for adaptive sampling:
  --tolerance = 0,                -- threshold for ignoring higher modes,
                                  -- if the sum of absolute values of higher
                                  -- modes in relation to the first mode is
                                  -- below the tolerance, they will be cut off.
                                  -- Default: 0 (never ignore modes).
                                  -- Recommended: 0.01 - 0.05
  --dof_reduction = 0.5,          -- Factor to multiply the 1D number of degrees
                                  -- of freedom with in each refinement.
                                  -- (0.5 implies to only keep half the degrees
                                  -- of freedom after the refinement).
                                  -- Values smaller 0.5 are not useful here and
                                  -- 0.5 already yields quite poor polynomial
                                  -- representations in the sampling.
                                  -- Default: 0.5
                                  -- Recommended: 2/3 - 3/4, 0.7 generally looks
                                  -- quite good.
  --adaptiveDofReduction = false, -- Increase the number of modes to keep after
                                  -- refinement above the configured
                                  -- dof_reduction factor if no more than the
                                  -- original memory will be used.
                                  -- Default: false
                                  -- Recommended: true (generally advisable, no
                                  --                    big drawbacks)
  --absUpperBoundLevel = 0        -- Maximal absolute level to refine to.
                                  -- 0 means no absolute level, all elements
                                  -- will be limited in the refinement by the
                                  -- nlevels set above
                                  -- This is useful for multilevel meshes, where
                                  -- each element might start out on a different
                                  -- level.
                                  -- For a complete mesh overview you might want
                                  -- to use a very high nlevels setting but
                                  -- limit the refinement by absUpperBoundLevel
                                  -- to allow for a more uniform resolution in
                                  -- the sampled data.
                                  -- Default: 0 (no absolute upper bound)
}

-- Example tracking to generate vtk files:
tracking = {
  {
    label = 'E-field',
    variable = {
      'displacement_field'
    },
    shape = {
      kind='global'
    },
    time_control = {
      min = {
        iter = 0
      },
      max = sim_control.time_control.max,
      interval = sim_control.time_control.max
    },
    folder = 'track_',
    output = {
      format = 'vtk'
    }
  }
}
