/*
 * Copyright (C) 2017 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#include "TestHidDescriptor.h"
#include "TestHidSensorSpec.h"
#include <cstring>

/**
 * Example HID sensor descriptors in this file is extracted from published document
 * "HID Sensors Usage" (hid-sensors-usage.docx). They are added for testing of HidParser.
 *
 * It is slightly modified in order to compile.
 */
namespace {
/**
 * Two sensors collection skeleton example.
 */
const unsigned char col1_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_COLLECTION,
    HID_COLLECTION(Application),

    HID_REPORT_ID(1),
    HID_USAGE_PAGE_SENSOR,

    HID_USAGE_SENSOR_TYPE_MOTION_ACCELEROMETER_3D,
    HID_COLLECTION(Physical),

    //Feature Report Descriptor for Sensor Report ID = 1

    //Input Report Descriptor for Sensor Report ID = 1

    HID_END_COLLECTION, //for Report ID = 1

    HID_REPORT_ID(2),
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_MOTION_GYROMETER_3D,
    HID_COLLECTION(Physical),

    //Feature Report Descriptor for Sensor Report ID = 2

    //Input Report Descriptor for Sensor Report ID = 2

    HID_END_COLLECTION, //for Report ID = 2

    //More sensors follow using the same pattern

    HID_END_COLLECTION //Application
};


/**
 * Composite device descriptor example: two sensors with a keyboard and a mouse.
 */
const unsigned char col2_report_descriptor[] = {
    //keyboard
    0x05U, 0x01U,         // USAGE PAGE (Generic Desktop)
    0x09U, 0x06U,         // USAGE (Keyboard)
    0xA1U, 0x01U,         // COLLECTION (Application)

    0x85U, 0x03U,         // REPORT_ID (1)

    0x15U, 0x00U,         //   LOGICAL MINIMUM (0)
    0x25U, 0x01U,         //   LOGICAL MAXIMUM (1)
    0x75U, 0x01U,         //   REPORT SIZE (1)
    0x95U, 0x08U,         //   REPORT COUNT (8)
    0x05U, 0x07U,         //   USAGE PAGE (Keyboard)
    0x19U, 0xE0U,         //   USAGE MINIMUM (Keyboard LeftControl)
    0x29U, 0xE7U,         //   USAGE MAXIMUM (Keyboard Right GUI)
    0x81U, 0x02U,         //   INPUT (Var)

    0x75U, 0x08U,         //   REPORT SIZE (8)
    0x95U, 0x0AU,         //   REPORT COUNT (10)
    0x19U, 0x00U,         //   USAGE MINIMUM (No event)
    0x29U, 0x91U,         //   USAGE MAXIMUM (Keyboard LANG2)
    0x26U, 0xFFU, 0x00U,  //   LOGICAL MAXIMUM (0xFF)
    0x81U, 0x00U,         //   INPUT (Data,Ary,Abs)

    0xC0U,                // END COLLECTION, //keyboard

    // Two sensor collection skeleton example:
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_COLLECTION,
    HID_COLLECTION(Application),

    HID_REPORT_ID(2),
    HID_USAGE_PAGE_SENSOR,

    HID_USAGE_SENSOR_TYPE_MOTION_ACCELEROMETER_3D,
    HID_COLLECTION(Physical),

    //Feature Report Descriptor for Sensor Report ID = 2

    //Input Report Descriptor for Sensor Report ID = 2

    HID_END_COLLECTION, //for Report ID = 2

    HID_REPORT_ID(3),
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_MOTION_GYROMETER_3D,
    HID_COLLECTION(Physical),

    //Feature Report Descriptor for Sensor Report ID = 3

    //Input Report Descriptor for Sensor Report ID = 3

    HID_END_COLLECTION, //for Report ID = 3

    //More sensors follow using the same pattern

    HID_END_COLLECTION, //Sensor collection

    //mouse
    0x05U, 0x01U,         // USAGE PAGE (Generic Desktop)
    0x09U, 0x02U,         // USAGE (Mouse)
    0xA1U, 0x01U,         // COLLECTION (Application)
    0x85U, 0x04U,         // REPORT_ID (4)

    // 5 mouse buttons
    0x05U, 0x09U,         //   USAGE PAGE (Button)
    0x19U, 0x01U,         //   USAGE MINIMUM (Button 1)
    0x29U, 0x05U,         //   USAGE MAXIMUM (Button 5)
    0x15U, 0x00U,         //   LOGICAL MINIMUM (0)
    0x25U, 0x01U,         //   LOGICAL MAXIMUM (1)
    0x95U, 0x05U,         //   REPORT COUNT (5)
    0x75U, 0x01U,         //   REPORT SIZE (1)
    0x81U, 0x02U,         //   INPUT (Data,Var,Abs)

    //  3 unused buttons:
    0x95U, 0x01U,         //   REPORT COUNT (1)
    0x75U, 0x03U,         //   REPORT SIZE (3)
    0x81U, 0x03U,         //   INPUT (Cnst,Var,Abs)

    // mouse (delta x, delta y) position
    0x15U, 0x81U,         //   LOGICAL MINIMUM (-127)
    0x25U, 0x7fU,         //   LOGICAL MAXIMUM (+127)
    0x75U, 0x08U,         //   REPORT SIZE (8)
    0x95U, 0x02U,         //   REPORT COUNT (2)
    0x05U, 0x01U,         //   USAGE PAGE (Generic Desktop)
    0x09U, 0x30U,         //   USAGE(X)
    0x09U, 0x31U,         //   USAGE (Y)
    0x81U, 0x06U,         //   INPUT (Data,Var,Rel)

    0xC0U,                // END COLLECTION //mouse
};


/**
 * Composite device descriptor example: one sensor with a keyboard and a mouse.
 */
const unsigned char col3_report_descriptor[] = {
    //keyboard
    0x05U, 0x01U,         // USAGE PAGE (Generic Desktop)
    0x09U, 0x06U,         // USAGE (Keyboard)
    0xA1U, 0x01U,         // COLLECTION (Application)

    0x85U, 0x03U,         // REPORT_ID (1)

    0x15U, 0x00U,         //   LOGICAL MINIMUM (0)
    0x25U, 0x01U,         //   LOGICAL MAXIMUM (1)
    0x75U, 0x01U,         //   REPORT SIZE (1)
    0x95U, 0x08U,         //   REPORT COUNT (8)
    0x05U, 0x07U,         //   USAGE PAGE (Keyboard)
    0x19U, 0xE0U,         //   USAGE MINIMUM (Keyboard LeftControl)
    0x29U, 0xE7U,         //   USAGE MAXIMUM (Keyboard Right GUI)
    0x81U, 0x02U,         //   INPUT (Var)

    0x75U, 0x08U,         //   REPORT SIZE (8)
    0x95U, 0x0AU,         //   REPORT COUNT (10)
    0x19U, 0x00U,         //   USAGE MINIMUM (No event)
    0x29U, 0x91U,         //   USAGE MAXIMUM (Keyboard LANG2)
    0x26U, 0xFFU, 0x00U,  //   LOGICAL MAXIMUM (0xFF)
    0x81U, 0x00U,         //   INPUT (Data,Ary,Abs)

    0xC0U,                // END COLLECTION, //keyboard

    // One sensor collection skeleton example:
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_COLLECTION,
    HID_COLLECTION(Application),

    HID_REPORT_ID(2),
    HID_USAGE_PAGE_SENSOR,

    HID_USAGE_SENSOR_TYPE_MOTION_GYROMETER_3D,
    HID_COLLECTION(Physical),

    //Feature Report Descriptor for Sensor Report ID = 2

    //Input Report Descriptor for Sensor Report ID = 2

    HID_END_COLLECTION, //for Report ID = 2

    HID_END_COLLECTION, //Sensor collection

    //mouse
    0x05U, 0x01U,         // USAGE PAGE (Generic Desktop)
    0x09U, 0x02U,         // USAGE (Mouse)
    0xA1U, 0x01U,         // COLLECTION (Application)
    0x85U, 0x04U,         // REPORT_ID (3)

    // 5 mouse buttons
    0x05U, 0x09U,         //   USAGE PAGE (Button)
    0x19U, 0x01U,         //   USAGE MINIMUM (Button 1)
    0x29U, 0x05U,         //   USAGE MAXIMUM (Button 5)
    0x15U, 0x00U,         //   LOGICAL MINIMUM (0)
    0x25U, 0x01U,         //   LOGICAL MAXIMUM (1)
    0x95U, 0x05U,         //   REPORT COUNT (5)
    0x75U, 0x01U,         //   REPORT SIZE (1)
    0x81U, 0x02U,         //   INPUT (Data,Var,Abs)

    //  3 unused buttons:
    0x95U, 0x01U,         //   REPORT COUNT (1)
    0x75U, 0x03U,         //   REPORT SIZE (3)
    0x81U, 0x03U,         //   INPUT (Cnst,Var,Abs)

    // mouse (delta x, delta y) position
    0x15U, 0x81U,         //   LOGICAL MINIMUM (-127)
    0x25U, 0x7fU,         //   LOGICAL MAXIMUM (+127)
    0x75U, 0x08U,         //   REPORT SIZE (8)
    0x95U, 0x02U,         //   REPORT COUNT (2)
    0x05U, 0x01U,         //   USAGE PAGE (Generic Desktop)
    0x09U, 0x30U,         //   USAGE(X)
    0x09U, 0x31U,         //   USAGE (Y)
    0x81U, 0x06U,         //   INPUT (Data,Var,Rel)

    0xC0U,                 //   END COLLECTION //mouse
};


/**
 * Simple custom sensor example.
 */
const unsigned char cus1_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,         // USAGE_PAGE (Sensor)
    HID_USAGE_SENSOR_TYPE_OTHER_CUSTOM, // USAGE (Simple Custom)
    HID_COLLECTION(Physical),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_1,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_DATA_MOTION_SPEED value

    HID_END_COLLECTION
};


/**
 * More complex custom sensor example.
 */
const unsigned char cus2_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,         // USAGE_PAGE (Sensor)
    HID_USAGE_SENSOR_TYPE_OTHER_CUSTOM, // USAGE (Simple Custom)
    HID_COLLECTION(Physical),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_DATA_CUSTOM_USAGE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_TYPE_MOTION_SPEEDOMETER
    HID_USAGE_SENSOR_DATA_CUSTOM_BOOLEAN_ARRAY,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_TYPE_MOTION_SPEEDOMETER
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_1,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_DATA_MOTION_SPEED value
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_2,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_DATA_MOTION_SPEED value
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_3,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_DATA_MOTION_SPEED value
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_4,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_DATA_MOTION_SPEED value
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_5,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_DATA_MOTION_SPEED value
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_6,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_DATA_MOTION_SPEED value

    HID_END_COLLECTION
};


/**
 * Custom sensor example with features
 */
const unsigned char cus3_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,         // USAGE_PAGE (Sensor)
    HID_USAGE_SENSOR_TYPE_OTHER_CUSTOM, // USAGE (Simple Custom)
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_CUSTOM,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_CUSTOM,HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_CUSTOM,HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_CUSTOM_USAGE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_TYPE_MOTION_SPEEDOMETER
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_1,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs), // = HID_USAGE_SENSOR_DATA_MOTION_SPEED value

    HID_END_COLLECTION
};


/**
 * Full custom sensor example with feature.
 */
const unsigned char cus4_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,         // USAGE_PAGE (Sensor)
    HID_USAGE_SENSOR_TYPE_OTHER_CUSTOM, // USAGE (Simple Custom)
    HID_COLLECTION(Application),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_CUSTOM,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_CUSTOM,HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_CUSTOM,HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_CUSTOM_USAGE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_BOOLEAN_ARRAY,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_1,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_2,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_3,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_4,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_5,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_6,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),

#if 1 //define vendor-specific (non-spec) custom datafields
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_7,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_8,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_9,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_10,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_11,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_12,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_13,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_14,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_15,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_16,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_17,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_18,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_19,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_20,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_21,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_22,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_23,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_24,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_25,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_26,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_27,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_CUSTOM_VALUE_28,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale unit to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),
#endif
    HID_END_COLLECTION
};


/**
 * Human presence sensor example.
 */
const unsigned char pres_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_BIOMETRIC_PRESENCE,
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_BIOMETRIC_HUMAN_PRESENCE,
    HID_LOGICAL_MIN_8(0), // False
    HID_LOGICAL_MAX_8(1), // True
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs),

    HID_END_COLLECTION
};


/**
 * Proximity sensor example.
 */
const unsigned char prox_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_BIOMETRIC_PROXIMITY,
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_BIOMETRIC_HUMAN_PROXIMITY_RANGE,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0D), // scale default unit "meter" to "centimeter"
    // to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_BIOMETRIC_HUMAN_PROXIMITY_RANGE,
                          HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0D), // scale default unit "meter" to "centimeter"
    // to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_BIOMETRIC_HUMAN_PROXIMITY_RANGE,
                          HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0D), // scale default unit "meter" to "centimeter"
    // to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_BIOMETRIC_HUMAN_PROXIMITY_OUT_OF_RANGE,
    HID_LOGICAL_MIN_8(0), // False
    HID_LOGICAL_MAX_8(1), // True
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_BIOMETRIC_HUMAN_PROXIMITY_RANGE,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0D),  // scale default unit "meter" to "centimeter"
    // to provide 2 digits past decimal point
    HID_INPUT(Data_Var_Abs),

    HID_END_COLLECTION
};


/**
 * Barometer sensor example.
 */
const unsigned char bar_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_ENVIRONMENTAL_ATMOSPHERIC_PRESSURE,
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_ATMOSPHERIC_PRESSURE,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit "Bar" to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_ATMOSPHERIC_PRESSURE,
                          HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit "Bar" to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_ATMOSPHERIC_PRESSURE,
                          HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit "Bar" to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_ATMOSPHERIC_PRESSURE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit "Bar" to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),

    HID_END_COLLECTION
};


/**
 * Humidity sensor example.
 */
const unsigned char humi_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_ENVIRONMENTAL_HUMIDITY,
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_RELATIVE_HUMIDITY,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0x10,0x27), // 10000 = 0.00 to 100.00 percent with 2 digits past decimal point
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_RELATIVE_HUMIDITY,
                          HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0x10,0x27), // 10000 = 0.00 to 100.00 percent with 2 digits past decimal point
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_RELATIVE_HUMIDITY,
                          HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0x10,0x27), // 10000 = 0.00 to 100.00 percent with 2 digits past decimal point
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_RELATIVE_HUMIDITY,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0x10,0x27), // 10000 = 0.00 to 100.00 percent with 2 digits past decimal point
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit "percent"
    // to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),

    HID_END_COLLECTION
};


/**
 * Temperature sensor example.
 */
const unsigned char temp_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_ENVIRONMENTAL_TEMPERATURE,
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_TEMPERATURE,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit "Celsius"
    // to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_TEMPERATURE,
                          HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit "Celsius"
    // to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_TEMPERATURE,
                          HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit "Celsius"
    // to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(16),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_ENVIRONMENTAL_TEMPERATURE,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit "Celsius"
    // to provide 2 digits past the decimal point
    HID_INPUT(Data_Var_Abs),

    HID_END_COLLECTION
};


/**
 * Ambient light sensor example.
 */
const unsigned char als_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,         // USAGE_PAGE (Sensor)
    HID_USAGE_SENSOR_TYPE_LIGHT_AMBIENTLIGHT, // USAGE (AmbientLight)
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_LIGHT_ILLUMINANCE,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_REL_PCT),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0x10,0x27), // 10000 = 0.00 to 100.00 percent with 2 digits past decimal point
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E),  // scale default unit to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_LIGHT_ILLUMINANCE,HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_LIGHT_ILLUMINANCE,HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_FEATURE(Data_Var_Abs),

    //add this definition if required by the specific application
    HID_USAGE_SENSOR_PROPERTY_RESPONSE_CURVE,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(10), //as required for n pair of values
    HID_UNIT_EXPONENT(0x0), // scale default unit to provide 0 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_LIGHT_ILLUMINANCE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_LIGHT_COLOR_TEMPERATURE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_UNIT_EXPONENT(0),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_LIGHT_CHROMATICITY_X,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_UNIT_EXPONENT(0x0C), // scale default unit to provide 4 digits past decimal point
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_LIGHT_CHROMATICITY_Y,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_UNIT_EXPONENT(0x0C), // scale default unit to provide 4 digits past decimal point
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs),

    HID_END_COLLECTION
};


/**
 * 3D accelerometer sensor example.
 */
const unsigned char accel3_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_MOTION_ACCELEROMETER_3D,
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),

    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_MOTION_ACCELERATION,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E),  // scale default unit Gs to centi-Gs
    // to provide 2 digits past Gs decimal point
    HID_FEATURE(Data_Var_Abs),

    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_MOTION_ACCELERATION,HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E),  // scale default unit Gs to centi-Gs
    // to provide 2 digits past Gs decimal point
    HID_FEATURE(Data_Var_Abs),

    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_MOTION_ACCELERATION,HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E),  // scale default unit Gs to centi-Gs
    // to provide 2 digits past Gs decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_MOTION_ACCELERATION_X_AXIS,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E),  // scale default unit Gs to centi-Gs
    // to provide 2 digits past Gs decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_MOTION_ACCELERATION_Y_AXIS,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E),  // scale default unit Gs to centi-Gs
    // to provide 2 digits past Gs decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_MOTION_ACCELERATION_Z_AXIS,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E),  // scale default unit Gs to centi-Gs
    // to provide 2 digits past Gs decimal point
    HID_INPUT(Data_Var_Abs),

    //include the following datafield if required to support the shake event
    HID_USAGE_SENSOR_DATA_MOTION_STATE,
    HID_LOGICAL_MIN_8(0), // False = Still
    HID_LOGICAL_MAX_8(1), // True = In Motion
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_INPUT(Data_Var_Abs),

    HID_END_COLLECTION
};


/**
 * 3D gyroscope sensor example.
 */
const unsigned char gyro3_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_MOTION_GYROMETER_3D,
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_MOTION_ANGULAR_VELOCITY,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_MOTION_ANGULAR_VELOCITY,
                          HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_MOTION_ANGULAR_VELOCITY,
                          HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_MOTION_ANGULAR_VELOCITY_X_AXIS,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_MOTION_ANGULAR_VELOCITY_Y_AXIS,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_MOTION_ANGULAR_VELOCITY_Z_AXIS,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_INPUT(Data_Var_Abs),

    HID_END_COLLECTION
};


/**
 * 3D compass sensor example.
 */
const unsigned char comp3_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_ORIENTATION_COMPASS_3D,
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETIC_HEADING,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETIC_HEADING,
                          HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETIC_HEADING,
                          HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETIC_FLUX,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETIC_FLUX,
                          HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETIC_FLUX,
                          HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_ORIENTATION_COMPENSATED_MAGNETIC_NORTH,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_ORIENTATION_COMPENSATED_TRUE_NORTH,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETIC_NORTH,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_ORIENTATION_TRUE_NORTH,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0F), // scale default unit to provide 1 digit past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETIC_FLUX_X_AXIS,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0D), // scale default unit to "milliGauss"
    // to provide 3 digits past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETIC_FLUX_Y_AXIS,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0D), // scale default unit to "milliGauss"
    // to provide 3 digits past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETIC_FLUX_Z_AXIS,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0D), // scale default unit to "milliGauss"
    // to provide 3 digits past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY_LOW,
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY_MEDIUM,
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY_HIGH,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,

    HID_END_COLLECTION
};


/**
 * 3D inclinomater example.
 */
const unsigned char inc3_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_ORIENTATION_INCLINOMETER_3D,
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_TILT,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_TILT,HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_TILT,HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_ORIENTATION_TILT_X,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_ORIENTATION_TILT_Y,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_ORIENTATION_TILT_Z,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit to provide 2 digits past decimal point
    HID_INPUT(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY_LOW,
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY_MEDIUM,
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY_HIGH,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,

    HID_END_COLLECTION
};


/**
 * Device orientation sensor example. Note this maps to rotation vector sensor in android.
 */
const unsigned char devor_report_descriptor[] = {
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_TYPE_ORIENTATION_DEVICE_ORIENTATION,
    HID_COLLECTION(Physical),

    //feature reports (xmit/receive)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_PROPERTY_SENSOR_CONNECTION_TYPE,  // NAry
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_INTEGRATED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_ATTACHED_SEL,
    HID_USAGE_SENSOR_PROPERTY_CONNECTION_TYPE_PC_EXTERNAL_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_NO_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_ALL_EVENTS_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_REPORTING_STATE_THRESHOLD_EVENTS_WAKE_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_UNDEFINED_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D0_FULL_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D1_LOW_POWER_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D2_STANDBY_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D3_SLEEP_WITH_WAKE_SEL,
    HID_USAGE_SENSOR_PROPERTY_POWER_STATE_D4_POWER_OFF_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_FEATURE(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_PROPERTY_REPORT_INTERVAL,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_32(0xFF,0xFF,0xFF,0xFF),
    HID_REPORT_SIZE(32),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_PROPERTY_CHANGE_SENSITIVITY_ABS,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E), // scale default unit "meter" to provide 2 digits past the decimal point
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_QUATERNION,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_QUATERNION,
                          HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x01),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_QUATERNION,
                          HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x01),
    HID_FEATURE(Data_Var_Abs),

    //include this if the values are calculated in firmware
    //otherwise, the driver will calculate these values from the Quaternion
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_ROTATION_MATRIX,
                          HID_USAGE_SENSOR_DATA_MOD_CHANGE_SENSITIVITY_ABS),
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_16(0xFF,0xFF),
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_ROTATION_MATRIX,
                          HID_USAGE_SENSOR_DATA_MOD_MAX),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E),
    HID_FEATURE(Data_Var_Abs),
    HID_USAGE_SENSOR_DATA(HID_USAGE_SENSOR_DATA_ORIENTATION_ROTATION_MATRIX,
                          HID_USAGE_SENSOR_DATA_MOD_MIN),
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(1),
    HID_UNIT_EXPONENT(0x0E),
    HID_FEATURE(Data_Var_Abs),

    //input reports (transmit)
    HID_USAGE_PAGE_SENSOR,
    HID_USAGE_SENSOR_STATE,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(6),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_STATE_UNKNOWN_SEL,
    HID_USAGE_SENSOR_STATE_READY_SEL,
    HID_USAGE_SENSOR_STATE_NOT_AVAILABLE_SEL,
    HID_USAGE_SENSOR_STATE_NO_DATA_SEL,
    HID_USAGE_SENSOR_STATE_INITIALIZING_SEL,
    HID_USAGE_SENSOR_STATE_ACCESS_DENIED_SEL,
    HID_USAGE_SENSOR_STATE_ERROR_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_EVENT,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(5),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_EVENT_UNKNOWN_SEL,
    HID_USAGE_SENSOR_EVENT_STATE_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_PROPERTY_CHANGED_SEL,
    HID_USAGE_SENSOR_EVENT_DATA_UPDATED_SEL,
    HID_USAGE_SENSOR_EVENT_POLL_RESPONSE_SEL,
    HID_USAGE_SENSOR_EVENT_CHANGE_SENSITIVITY_SEL,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,
    HID_USAGE_SENSOR_DATA_ORIENTATION_QUATERNION,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(4),
    HID_UNIT_EXPONENT(0x0E),
    HID_INPUT(Data_Arr_Abs),

    //include this if the values are calculated in firmware
    //otherwise, the driver will calculate these values from the Quaternion
    HID_USAGE_SENSOR_DATA_ORIENTATION_ROTATION_MATRIX,
    HID_LOGICAL_MIN_16(0x01,0x80), //    LOGICAL_MINIMUM (-32767)
    HID_LOGICAL_MAX_16(0xFF,0x7F), //    LOGICAL_MAXIMUM (32767)
    HID_REPORT_SIZE(16),
    HID_REPORT_COUNT(9),
    HID_UNIT_EXPONENT(0x0F),
    HID_INPUT(Data_Arr_Abs),

    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY,
    HID_LOGICAL_MIN_8(0),
    HID_LOGICAL_MAX_8(2),
    HID_REPORT_SIZE(8),
    HID_REPORT_COUNT(1),
    HID_COLLECTION(Logical),
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY_LOW,
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY_MEDIUM,
    HID_USAGE_SENSOR_DATA_ORIENTATION_MAGNETOMETER_ACCURACY_HIGH,
    HID_INPUT(Data_Arr_Abs),
    HID_END_COLLECTION,

    HID_END_COLLECTION
};
} // annoymous namespace

/**
 * Global test vector entry.
 */
#define TEST_DESCRIPTOR(name) \
    { name ## _report_descriptor, sizeof(name ## _report_descriptor), #name }

const TestHidDescriptor gDescriptorArray[] = {
    TEST_DESCRIPTOR(accel3),
    TEST_DESCRIPTOR(col1),
    TEST_DESCRIPTOR(col2),
    TEST_DESCRIPTOR(col3),
    TEST_DESCRIPTOR(cus1),
    TEST_DESCRIPTOR(cus2),
    TEST_DESCRIPTOR(cus3),
    TEST_DESCRIPTOR(cus4),
    TEST_DESCRIPTOR(pres),
    TEST_DESCRIPTOR(prox),
    TEST_DESCRIPTOR(bar),
    TEST_DESCRIPTOR(humi),
    TEST_DESCRIPTOR(temp),
    TEST_DESCRIPTOR(als),
    TEST_DESCRIPTOR(gyro3),
    TEST_DESCRIPTOR(comp3),
    TEST_DESCRIPTOR(inc3),
    TEST_DESCRIPTOR(devor),
    {nullptr, 0, nullptr} //sentinel
};
#undef TEST_DESCRIPTOR

const TestHidDescriptor *findTestDescriptor(const char *name) {
    if (name == nullptr) {
        return nullptr;
    }

    for (const TestHidDescriptor *p = gDescriptorArray; ; ++p) {
        if (p->data == nullptr || p->len == 0) {
            break;
        }
        if (strcmp(p->name, name) == 0) {
            return p;
        }
    }
    return nullptr;
}
