<?php
/**
 * External Variable Class
 * @package	Adeli
 */
namespace Adelie;

require_once \sprintf("phar://%s/lib/UploadedFile.php", namespace\PHAR);

class ExternalVariable
{
	const NONE = 0;
	const GET = 1;
	const POST = 2;

	private static $instance;

	/**
	 * constructor
	 * @access	private
	 */
	private function __construct ()
	{
	}

	/**
	 * get the instance
	 * @access	public
	 * @static
	 * @return	\Adelie\ExternalVariable
	 */
	public static function getInstance ()
	{
		if (\is_null(self::$instance)) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * get an external variable
	 * @access	public
	 * @param	string	$key
	 * @param	integer	$type
	 * @return	mixed
	 */
	private function getVar ($key, $type=self::NONE)
	{
		switch ($type) {
			case self::GET:
				return isset($_GET[$key]) ? $_GET[$key] : null;
				break;
			case self::POST:
				return isset($_POST[$key]) ? $_POST[$key] : null;
				break;
			default:
				return isset($_REQUEST[$key]) ? $_REQUEST[$key] : null;
				break;
		}
	}

	/**
	 * get the external variable as integer
	 * @access	public
	 * @param	string	$key
	 * @param	integer	$type
	 * @return	integer|array
	 */
	public function getInt ($key, $type=self::NONE)
	{
		$var = $this->getVar($key, $type);
		if (\is_null($var)) {
			return 0;
		}
		return self::convInt($var);
	}

	/**
	 * get the external variable as integer
	 * @access	public
	 * @param	string	$key
	 * @param	integer	$type
	 * @return	float|array
	 */
	public function getDbl ($key, $type=self::NONE)
	{
		$var = $this->getVar($key, $type);
		if (\is_null($var)) {
			return 0.0;
		}
		return self::convDbl($var);
	}

	/**
	 * get the external variable as string
	 * @access	public
	 * @param	string	$key
	 * @param	integer	$type
	 * @return	string|array
	 */
	public function getStr ($key, $type=self::NONE)
	{
		$var = $this->getVar($key, $type);
		if (\is_null($var)) {
			return "";
		}
		return self::convStr($var);
	}

	/**
	 * get the external variable as boolean
	 * @access	public
	 * @param	string	$key
	 * @param	integer	$type
	 * @return	boolean|array
	 */
	public function getBool ($key, $type=self::NONE)
	{
		$var = $this->getVar($key, $type);
		if (\is_null($var)) {
			return false;
		}
		return self::convBool($var);
	}

	/**
	 * get the external variable as uploaded file
	 * @access	public
	 * @param	string	$key
	 * @param	string	$idx
	 * @return	null|\Adelie\UploadedFile
	 */
	public function getFile ($key, $idx="")
	{
		return new UploadedFile($key, $idx);
	}

	/**
	 * convert the variable to integer
	 * @param	mixed	$var
	 * @return	integer|array
	 */
	public static function convInt ($var)
	{
		if (\is_array($var)) {
			return \array_map("self::convInt", $var);
		} else {
			return \intval(\is_object($var) ? 0 : $var);
		}
	}

	/**
	 * convert the variable to double
	 * @param	mixed	$var
	 * @return	float|array
	 */
	public static function convDbl ($var)
	{
		if (\is_array($var)) {
			return \array_map("self::convDbl", $var);
		} else {
			return \floatval(\is_object($var) ? 0.0 : $var);
		}
	}

	/**
	 * convert the variable to string
	 * @param	mixed	$var
	 * @return	string|array
	 */
	public static function convStr ($var)
	{
		if (\is_array($var)) {
			return \array_map("self::convStr", $var);
		} else {
			return \strval($var);
		}
	}

	/**
	 * convert the variable to boolean
	 * @param	mixed	$var
	 * @return	boolean|array
	 */
	public static function convBool ($var)
	{
		if (\is_array($var)) {
			return \array_map("self::convBool", $var);
		} else {
			return \filter_var($var, \FILTER_VALIDATE_BOOLEAN);
		}
	}
}
