<?php
/**
 * MySQL Information Class
 * @package	Adelie
 * @copyright	Copyright (c) 2012, Adelie Development Team
 */
namespace Adelie;

require_once \sprintf("phar://%s/core/db/AbstractDbInfo.php", namespace\PHAR);
require_once \sprintf("phar://%s/core/db/DbMysql.php", namespace\PHAR);

final class DbMysqlInfo extends namespace\AbstractDbInfo
{
	/**
	 * constructor
	 * @access	public
	 * @param	\Adelie\AbstractDb	$dsn
	 */
	public function __construct (namespace\AbstractDb $db)
	{
		parent::__construct($db);

		// get table name
		$db->prepare("SELECT * FROM information_schema.columns WHERE TABLE_SCHEMA = :schema ");
		$db->bind("schema", $db->getDbName());
		$db->exec();
		while ($db->fetch()) {
			$name = \strtolower($db->get("column_name"));
			$this->columns[$db->get("table_name")][$name] = array(
				"php_type" => $this->getPhpType($db->get("data_type")),
				"pdo_type" => $this->getPdoType($db->get("data_type"))
			);
		}
		$db->free();
	}

	/**
	 * get PDO data type corresponding to the passed $type
	 * @access	protected
	 * @param	string	$type
	 * @return	integer
	 */
	protected function getPdoType ($type)
	{
		$type = \strtolower($type);
		switch ($type) {
			case "bit":
			case "tinyint":
			case "smallint":
			case "mediumint":
			case "integer":
				$cast = "\\PDO::PARAM_INT";
				break;
			case "bigint":
				$cast = (\PHP_INT_SIZE==8) ? "\\PDO::PARAM_INT" : "\\PDO::PARAM_STR";
				break;
			case "binary":
			case "varbinary":
			case "tinyblob":
			case "blob":
			case "mediumblob":
			case "longblob":
				$cast = "\\PDO::PARAM_LOB";
				break;
			default:
				$cast = "\\PDO::PARAM_STR";
				break;
		}
		return $cast;
	}

	/**
	 * get PHP data type corresponding to the passed $type
	 * @access	protected
	 * @param	string	$type
	 * @return	string
	 */
	protected function getPhpType ($type)
	{
		$type = \strtolower($type);
		switch ($type) {
			case "bit":
			case "tinyint":
			case "smallint":
			case "mediumint":
			case "integer":
				$cast = "intval";
				break;
			case "bigint":
				$cast = (\PHP_INT_SIZE==8) ? "intval" : "strval";
				break;
			case "float":
			case "double":
			case "decimal":
			case "real":
				$cast = "floatval";
				break;
			case "binary":
			case "varbinary":
			case "tinyblob":
			case "blob":
			case "mediumblob":
			case "longblob":
				$cast = "";
				break;
			default:
				$cast = "strval";
				break;
		}
		return $cast;
	}
}
