<?php
/**
 * Log Class
 * @package	Adelie
 */
namespace Adelie;

class Log
{
	// Error code
	const E_INVLID_ARG = 1;

	// Error Level
	const LVL_NONE = 0;
	const LVL_DEBUG = 10;
	const LVL_INFO = 20;
	const LVL_WARN = 30;
	const LVL_FATAL = 40;

	// Lotation mode
	const LOTATE_NONE = 0;
	const LOTATE_HOURLY = 10;
	const LOTATE_DAYLY = 20;
	const LOTATE_WEEKLY = 30;
	const LOTATE_MONTHLY = 40;
	const LOTATE_QUARTERLY = 50;
	const LOTATE_YEARLY = 60;

	private $path;
	private $level;
	private $lotate;

	/**
	 * constructor
	 * @access	public
	 * @param	string	$path	path to the log file
	 * @param	integer	$level	log level
	 * @param	integer	$lotate	lotate mode
	 */
	public function __construct ($path, $level=self::LVL_NONE, $lotate=self::LOTATE_NONE)
	{
		$this->path = $this->createPath($path);
		$this->setLevel($level);
		$this->setLotate($lotate);
	}

	/**
	 * set logging level
	 * @access	private
	 * @param	integer	$level
	 * @return	void
	 * @throws	\InvalidArgumentException
	 */
	private function setLevel ($level)
	{
		switch ($level) {
			case self::LVL_NONE:
			case self::LVL_DEBUG:
			case self::LVL_INFO:
			case self::LVL_WARN:
			case self::LVL_FATAL:
				$this->level = \intval($level);
				break;
			default:
				$msg = namespace\Error::get(namespace\Error::E_INVALID_ARG);
				throw new \InvalidArgumentException($msg, namespace\Error::E_INVALID_ARG);
				break;
		}
	}

	/**
	 * set lotation mode
	 * @access	private
	 * @param	integer	$lotate	Log::LOTATE_*を設定
	 * @return	void
	 * @throws	\InvalidArgumentException
	 */
	private function setLotate ($lotate)
	{
		switch ($lotate) {
			case self::LOTATE_NONE:
			case self::LOTATE_HOURLY:
			case self::LOTATE_DAYLY:
			case self::LOTATE_WEEKLY:
			case self::LOTATE_MONTHLY:
			case self::LOTATE_QUARTERLY:
			case self::LOTATE_YEARLY:
				$this->lotate = $lotate;
				break;
			default:
				$msg = namespace\Error::get(namespace\Error::E_INVALID_ARG);
				throw new \InvalidArgumentException($msg, namespace\Error::E_INVALID_ARG);
				break;
		}
	}

	/**
	 * create the path of log file into account the lotation mode
	 * @access	private
	 * @param	string	$path
	 * @return	string
	 */
	private function createPath ($path)
	{
		if ($this->lotate==self::LOTATE_NONE) {
			return $path;
		}

		// parse
		$dir = \realpath(\dirname($path));
		$file = \basename($path);
		$tmp = \explode(".", $file);
		$ext = (\count($tmp)>1) ? \array_pop($tmp) : "log";
		$name = \implode(".", $tmp);

		// create a lotate symbol
		switch ($this->lotate) {
			case self::LOTATE_HOURLY:
				$symbol = \date("YmdH");
				break;
			case self::LOTATE_DAYLY:
				$symbol = \date("Ymd");
				break;
			case self::LOTATE_WEEKLY:
				list($year, $week) = \explode("-", \date("Y-W"));
				$symbol = \sprintf("%04dw%02d", $year, $week);
				break;
			case self::LOTATE_MONTHLY:
				$symbol = \date("Ym");
				break;
			case self::LOTATE_QUARTERLY:
				list($year, $month) = \explode("-", \date("Y-n"));
				$quarter = \intval( ($month -1) / 3 ) + 1;
				$symbol = \sprintf("%04dq%01d", $year, $quarter);
				break;
			case self::LOTATE_YEARLY:
				$symbol = \date("Y");
				break;
			default:
				$msg = namespace\Error::get(namespace\Error::E_INVALID_ARG);
				throw new \InvalidArgumentException($msg, namespace\Error::E_INVALID_ARG);
				break;
		}

		$path = \sprintf("%s/%s.%s.%s", $dir, $name, $symbol, $ext);
		if (!\is_writable($path)) {
			if (!\is_writable(\dirname($path))) {
				$msg = namespace\Error::get(namespace\Error::E_WRITE);
				throw new namespace\AdelieException($msg, namespace\Error::E_WRITE);
			}
		}

		return $path;
	}

	/**
	 * get the path of log file
	 * @access	public
	 * @return	void
	 */
	public function getPath ()
	{
		return $this->path;
	}

	/**
	 * log
	 * @access	private
	 * @param	string	$msg
	 * @param	string	$level
	 * @return	void
	 * @throws	\RuntimeException
	 */
	private function log ($msg, $level)
	{
		$msg = \sprintf("[%s] %-7s %s\n", \date("Y-m-d H:i:s"), "<{$level}>", $msg);
		$ret = @\file_put_contents($this->path, $msg, \FILE_APPEND|\LOCK_EX);
		if ($ret===false) {
			$msg = namespace\Error::get(namespace\Error::E_WRITE);
			throw new namespace\AdelieException($msg, namespace\Error::E_WRITE, $this->path);
		}
	}

	/**
	 * log as debug mode
	 * @access	public
	 * @param	string	$msg
	 * @return	void
	 */
	public function debug ($msg)
	{
		if ($this->level<=self::LVL_DEBUG) {
			$this->log($msg, "DEBUG");
		}
	}

	/**
	 * log as info mode
	 * @access	public
	 * @param	string	$msg
	 * @return	void
	 */
	public function info ($msg)
	{
		if ($this->level<=self::LVL_INFO) {
			$this->log($msg, "INFO");
		}
	}

	/**
	 * log as warning mode
	 * @access	public
	 * @param	string	$msg
	 * @return	void
	 */
	public function warn ($msg)
	{
		if ($this->level<=self::LVL_WARN) {
			$this->log($msg.$this->level, "WARN");
		}
	}

	/**
	 * log as fatal mode
	 * @access	public
	 * @param	string	$msg
	 * @return	void
	 */
	public function fatal ($msg)
	{
		if ($this->level<=self::LVL_FATAL) {
			$this->log($msg, "FATAL");
		}
	}
}
