<?php
/**
 * DB Result Class
 * @package	Adelie
 * @copyright	Copyright (c) 2012, Adelie Development Team
 */
namespace Adelie;

final class DbResult implements \Iterator
{
	private $stmt;
	private $binds;
	private $time;
	private $row;
	private $affected_rows;

	/**
	 * constructor
	 * @access	public
	 * @param	\PDOStatement	$stmt
	 * @param	array	$binds
	 * @param	float	$time
	 */
	public function __construct (\PDOStatement $stmt, Array $binds, $time)
	{
		$this->stmt = $stmt;
		$this->binds = $binds;
		$this->time = $time;
		$this->row = null;
		$this->affected_rows = $stmt->rowCount();
	}

	/**
	 * アクセス不可能メソッドにアクセスした際の振る舞い
	 * @access	public
	 * @param	string	$name
	 * @param	array	$args
	 * @return	mixed
	 */
	public function __call ($name, Array $args)
	{
		$prefix = \substr($name, 0, 3);
		$key = namespace\String::toUscore(\substr($name, 3));
		switch ($prefix) {
			case "get":
				return isset($this->row->{$key}) ? $this->row->{$key} : "";
				break;
			default:
				throw new namespace\DbException("You accessed a inaccessible method.");
				break;
		}
	}

	/**
	 * アクセス不可能プロパティにアクセスした際の振る舞い
	 * @access	public
	 * @param	string	$name
	 * @return	mixed
	 */
	public function __get ($name)
	{
		return isset($this->row->{$name}) ? $this->row->{$name} : "";
	}

	/**
	 * 現在の値を取得
	 * @access	public
	 * @return	array
	 */
	public function current ()
	{
		return $this->valid() ? $this->row : null;
	}

	/**
	 * 現在のキーを取得(使用しないので常に0を返す)
	 * @access	public
	 * @return	integer
	 */
	public function key ()
	{
		return	0;
	}

	/**
	 * 前の値に移動
	 * @access	public
	 * @return	void
	 */
	public function prev ()
	{
		$this->row = $this->stmt->fetch(\PDO::FETCH_OBJ, \PDO::FETCH_ORI_PRIOR);
		if ($this->row===false) {
			$this->row = null;
		}
	}

	/**
	 * 次の値に移動
	 * @access	public
	 * @return	void
	 */
	public function next ()
	{
		$this->row = $this->stmt->fetch(\PDO::FETCH_OBJ, \PDO::FETCH_ORI_NEXT);
		if ($this->row===false) {
			$this->row = null;
		}
	}

	/**
	 * 最初の値に移動
	 * @access	public
	 * @return	void
	 */
	public function rewind ()
	{
		$this->row = $this->stmt->fetch(\PDO::FETCH_OBJ, \PDO::FETCH_ORI_FIRST);
		if ($this->row===false) {
			$this->row = null;
		}
	}

	/**
	 * 最後の値に移動
	 * @access	public
	 * @return	void
	 */
	public function end ()
	{
		$this->row = $this->stmt->fetch(\PDO::FETCH_OBJ, \PDO::FETCH_ORI_LAST);
		if ($this->row===false) {
			$this->row = null;
		}
	}

	/**
	 * 現在の値が存在するかを確認
	 * @access	public
	 * @return	boolean
	 */
	public function valid ()
	{
		return ($this->row===null) ? false : true;
	}

	/**
	 * 実行されたSQLを取得
	 * @access	public
	 * @return	string
	 */
	public function sql ()
	{
		$msg = $this->stmt->queryString;
		foreach ($this->binds as $key=>$val) {
			$bind_value = "";
			if (\is_array($val["val"])) {
				foreach ($val["val"] as $tmp) {
					$bind_value .= "{$tmp}, ";
				}
				$bind_value = \rtrim($bind_value, ", ");
			} else {
				$bind_value = $val["val"];
			}
			$msg .= \sprintf("\n[%s] %s", $key, $bind_value);
		}
		return $msg;
	}

	/**
	 * get the execution time of query
	 * @access	public
	 * @param	integer	$digit
	 * @return	float
	 */
	public function time ($digit=6)
	{
		return \sprintf("%.{$digit}f", $this->time);
	}

	/**
	 * the first row
	 * @access	public
	 * @return	void
	 */
	public function one ()
	{
		$this->rewind();
		$this->current();
		$this->next();
	}
}
