<?php
/**
 * Error Class
 * @package	Adelie
 * @copyright	Copyright (c) 2012, Adelie Development Team
 */
namespace Adelie;

final class Error
{
	// 正常
	const E_NO_ERROR = 0;

	// エラー
	const E_CLI = 1;
	const E_CTL = 2;
	const E_ACT = 3;
	const E_TPL = 4;
	const E_FILE = 10;
	const E_EXIST = 11;
	const E_READ = 12;
	const E_WRITE = 13;
	const E_MOVE = 14;
	const E_COPY = 15;
	const E_INHERIT = 100;
	const E_LOCK = 101;
	const E_CLONING = 102;
	const E_INVALID_ARG = 150;
	const E_DB = 201; // DB error
	const E_DB_CONN = 202; // DB connect error
	const E_UNEXPECTED = 255; // unexpected error

	/**
	 * get an error message
	 * @static
	 * @access	public
	 * @param	string	$code
	 * @param	string	$arg1 ...
	 * @return	string
	 */
	public static function get ()
	{
		$args = \func_get_args();
		$code = \array_shift($args);
		switch ($code) {
			case self::E_CLI: $msg = \sprintf("The CLI class (%s) isn't found.", @$args[0]); break;
			case self::E_CTL: $msg = \sprintf("The controller class (%s) isn't found.", @$args[0]); break;
			case self::E_ACT: $msg = \sprintf("The action method (%s) isn't found.", @$args[0]); break;
			case self::E_TPL: $msg = \sprintf("The template file (%s) isn't found.", @$args[0]); break;
			case self::E_EXIST: $msg = \sprintf("\"%s\" doesn't exists.", @$args[0]); break;
			case self::E_READ: $msg = \sprintf("Can't read \"%s\".", @$args[0]); break;
			case self::E_WRITE: $msg = \sprintf("Can't write data to \"%s\".", @$args[0]); break;
			case self::E_INHERIT: $msg = \sprintf("%s doesn't inherit from %s class.", @$args[0], @$args[1]); break;
			case self::E_LOCK: $msg = \sprintf("Can't lock."); break;
			case self::E_CLONING: $msg = \sprintf("Can't clone %s.", @$args[0]); break;
			case self::E_INVALID_ARG: $msg = \sprintf("Specified a invalid argment \"%s\".", @$args[0]); break;
			case self::E_DB: $msg = \sprintf("DB error has occured."); break;
			case self::E_DB_CONN: $msg = \sprintf("DB connection error has occured."); break;
			case self::E_UNEXPECTED: $msg = \sprintf("An unexpected error has occured."); break;
			default: $msg = ""; break;
		}
		return $msg;
	}

	/**
	 * get the last PHP error
	 * @access	public
	 * @static
	 * @return	string
	 */
	public static function last ()
	{
		$err = \error_get_last();
		if (!\is_array($err)) {
			return "";
		}
		switch ($err["type"]) {
			case E_ERROR: $type = "Error"; break;
			case E_WARNING: $type = "Warning"; break;
			case E_PARSE: $type = "Parse"; break;
			case E_NOTICE: $type = "Notice"; break;
			case E_CORE_ERROR: $type = "Core Error"; break;
			case E_CORE_WARNING: $type = "Core Warning"; break;
			case E_COMPILE_ERROR: $type = "Compile Error"; break;
			case E_COMPILE_WARNING: $type = "Compile Warning"; break;
			case E_USER_ERROR: $type = "User Error"; break;
			case E_USER_WARNING: $type = "User Warning"; break;
			case E_USER_NOTICE: $type = "User Notice"; break;
			case E_STRICT: $type = "Strict"; break;
			case E_RECOVERABLE_ERROR: $type = "Recoverable Error"; break;
			case E_DEPRECATED: $type = "Deprecated"; break;
			case E_USER_DEPRECATED: $type = "User Deprecated"; break;
			case E_ALL: $type = "All"; break;
			default: $type = "Unknown"; break;
		}
		return \sprintf("%s: %s in %s on line %d", $type, $err["message"], $err["file"], $err["line"]);
	}

	/**
	 * print error to stderr (for CLI)
	 * @static
	 * @access	public
	 * @param	\Exception	$e	exception
	 * @param	boolean	$verbose
	 * @return	void
	 */
	public static function stderr (\Exception $e, $verbose=true)
	{
		if ($verbose) {
			$err  = \sprintf("\n_/_/_/_/_/_/ Adelie Framework ERROR _/_/_/_/_/_/\n\n");
			$err .= \sprintf("<Message>\n");
			$err .= \sprintf("[%d] %s\n\n", $e->getCode(), \trim($e->getMessage()));
			$err .= \sprintf("<Exception>\n");
			$err .= \sprintf("%s\n\n", \get_class($e));
			$err .= \sprintf("<Trace>\n");
			$err .= \sprintf("%s\n\n", \trim($e->getTraceAsString()));
			$err .= \sprintf("<Previous>\n");
			$err .= \sprintf("%s\n\n", \trim(\get_class($e->getPrevious())));
			$err .= \sprintf("<Time>\n");
			$err .= \sprintf("Start  : %s\n", \date("Y-m-d H:i:s(D)"));
			$err .= \sprintf("Finish : %s\n", \date("Y-m-d H:i:s(D)"));
			$err .= \sprintf("Proc   : %.4f sec\n", \microtime(true)-namespace\START_MTIME);
			$err .= \sprintf("\n_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/_/\n\n");
		} else {
			$err = \sprintf("[%d] %s\n\n", $e->getCode(), \trim($e->getMessage()));
		}
		\file_put_contents("php://stderr", $err);
	}

	/**
	 * print error for web
	 * @static
	 * @access	public
	 * @param	\Exception	$e	例外
	 * @return	void
	 */
	public static function web (\Exception $e)
	{
		@\ob_clean();
		require_once \sprintf("phar://%s/etc/error.html", namespace\PHAR);
	}
}
