<?php
/**
 * CLI Class
 * @package	Adelie
 * @copyright	Copyright (c) 2012, Adelie Development Team
 */
namespace Adelie;

final class Cli
{
	const MODE_SYS = 1; // system mode
	const MODE_USER = 2; // user mode

	private static $instance;

	private $cli;
	private $mode;
	private $lock;

	/**
	 * constructor
	 * @access	public
	 */
	public function __construct ()
	{
		@\ob_end_flush();
		@\set_time_limit(0);
		$this->cli = null;
		$this->sys_mode = self::MODE_SYS;
		$this->lock = null;
	}

	/**
	 * get the instance
	 * @access	public
	 * @static
	 * @param	integer	$mode
	 * @return	Cli
	 */
	public static function getInstance ()
	{
		if (self::$instance==null) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * lock
	 * @access	private
	 * @return	void
	 */
	private function lock ()
	{
		if (!$this->cli->isLockMode()) {
			return;
		}

		$path = \sprintf("%s/%s%s.lock", namespace\DIR_LCK, ($this->mode==self::MODE_SYS ? "@" : ""), $this->cli->getProgName());
		$sfi = new \SplFileInfo($path);
		if (!$sfi->isReadable()) {
			\touch($path);
		}
		$sfo = $sfi->openFile("rb");
		$ret = $sfo->flock(\LOCK_EX|\LOCK_NB);
		if ($ret) {
			$this->lock = $sfo;
		} else {
			$msg = namespace\Error::get(namespace\Error::E_LOCK);
			throw new namespace\AdelieException($msg, namespace\Error::E_LOCK);
		}
	}

	/**
	 * unlock
	 * @access	private
	 * @return	boolean
	 */
	private function unlock ()
	{
		if ($this->lock) {
			$this->lock->flock(\LOCK_UN);
			$this->lock = null;
		}
	}

	/**
	 * get the CLI object
	 * @access	private
	 * @param	string	$obj_name
	 * @return	\Abstract\AbstractCli
	 */
	private function getCli ($obj_name)
	{
		$obj_name = namespace\String::toPascal($obj_name);
		if ($this->mode==self::MODE_SYS) {
			$path = \sprintf("phar://%s/cli/%sCli.php", namespace\PHAR, $obj_name);
			$obj_name = \sprintf("\\%s\\%sCli", __NAMESPACE__, $obj_name);
		} else {
			$path = \sprintf("%s/%sCli.php", namespace\DIR_CLI, $obj_name);
			$obj_name = \sprintf("\\%sCli", $obj_name);
		}

		if (!\file_exists($path)) {
			$msg = namespace\Error::get(namespace\Error::E_EXIST, $path);
			throw new namespace\AdelieException($msg, namespace\Error::E_EXIST);
		}

		require_once $path;
		if (!\class_exists($obj_name)) {
			$msg = namespace\Error::get(namespace\Error::E_CLI, $obj_name);
			throw new namespace\AdelieException($msg, namespace\Error::E_CLI);
		}

		$obj = new $obj_name();
		if (!\is_subclass_of($obj, "\\".__NAMESPACE__."\\AbstractCli")) {
			$msg = namespace\Error::get(namespace\Error::E_INHERIT, $obj_name);
			throw new namespace\AdelieException($msg, namespace\Error::E_INHERIT);
		}
		return $obj;
	}

	/**
	 * run
	 * @access	public
	 * @return	void
	 */
	public function run ()
	{
		try {
			// check run mode
			$obj_name = isset($_SERVER["argv"][1]) ? \strtolower($_SERVER["argv"][1]) : "@help";
			// helpやsetupの時は設定ファイルを読まない
			if (!\in_array($obj_name, array("@help", "@setup"))) {
				require_once \sprintf("%s/config.php", namespace\DIR_CNF);
			}
			if (\substr($obj_name, 0, 1)=="@") {
				$this->mode = self::MODE_SYS;
				$obj_name = \substr($obj_name, 1);
			} else {
				$this->mode = self::MODE_USER;
			}

			// run CLI
			$this->cli = $this->getCli($obj_name);
			$this->lock();
			$this->cli->run();
			$this->unlock();

		} catch (\Exception $e) {
			namespace\Error::stderr($e);
		}
	}
}
