/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.plugin.column;

import org.opengion.hayabusa.db.AbstractEditor;
import org.opengion.hayabusa.db.CellEditor;
import org.opengion.hayabusa.db.DBColumn;
import org.opengion.fukurou.util.XHTMLTag;
import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.fukurou.util.Attributes;
import org.opengion.fukurou.util.StringUtil;
import org.opengion.fukurou.util.TagBuffer;

/**
 * BARCODE エディターは、quagga.js を利用した、カメラ映像からバーコードを読み込むクラスです。
 * 読み込んだ文字列は テキストエリアに書き出します。
 *
 * ｽﾀｰﾄ/ｴﾝﾄﾞｷｬﾗｸﾀ(*)は、JavaScript側(quagga)で削除されるようです。
 * ﾁｪｯｸﾃﾞｼﾞｯﾄのﾓｼﾞｭﾗｽ43 は、既存のﾊﾞｰｺｰﾄﾞﾌｫﾝﾄ仕様との互換性から、付いていません。…つまり何もしません。
 *
 * 基本的な構造は、カメラ映像描画開始ボタン、映像表示領域(canvas)、textarea で構成されます。
 * JavaScriptのサンプルの都合で、Editor_QRCODE と同じように、映像表示領域(video) を使用できませんでした。
 *
 *    &lt;button type='button' id='barStart' onClick='barcodeStart();'&gt;Video Start&lt;/button&gt;
 *    &lt;input name='≪ｶﾗﾑ名≫' id='outdata' /&gt; &lt;br /&gt;
 *    &lt;canvas id='preview' style='background-color: black;'&gt; &lt;/canvas&gt;
 *
 * script に CDNサービス を使うと、無線環境(iPad等)ではものすごく遅くなったため、ローカルに配置することにします。
 * &lt;script src="https://cdn.jsdelivr.net/npm/quagga@0.12.1/dist/quagga.min.js"&gt;&lt;!-- --&gt;&lt;/script&gt;
 *
 * script は、quagga.min.js を使います。現在、1画面1つしかカメラは使えません。
 * これらは、使用する画面に、組み込んでください。
 * &lt;script src="{&#064;SYS.JSP}/option/quagga.min.js"&gt;&lt;!-- --&gt;&lt;/script&gt;
 * &lt;script src="{&#064;SYS.JSP}/option/videocamera.js"&gt;&lt;!-- --&gt;&lt;/script&gt;
 *
 * を使用するページに設定します。
 *
 * @og.rev 7.4.2.2 (2021/05/28) 新規作成
 * @og.group データ編集
 *
 * @version  7.4
 * @author   Kazuhiko Hasegawa
 * @since    JDK11.0,
 */
public class Editor_BARCODE extends AbstractEditor {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "7.4.2.2 (2021/05/28)" ;

//	// 7.4.2.2 (2021/05/28) ｼｽﾃﾑ定数のJSPを使用します。(※ SYS.JSP + SYS.IMAGE_DIR)
//	private static final String JSP_OPT =  HybsSystem.sys( "JSP" ) + "/option/" ;

//	private static final String JS_SRC = "<script src='" + JSP_OPT + "quagga.min.js' ><!-- --></script>"
//								+ CR +	 "<script src='" + JSP_OPT + "videocamera.js' ><!-- --></script>" ;

	private static final String BASE_HTML1 = "<button type='button' id='barStart' onClick='barcodeStart();'>Scan Start</button>" ;

	private static final String BASE_HTML2 = "<br /><canvas id='preview' style='background-color: black;'> </canvas>" ;

	/**
	 * デフォルトコンストラクター。
	 * このコンストラクターで、基本オブジェクトを作成します。
	 *
	 * @og.rev 7.4.2.2 (2021/05/28) 新規作成
	 *
	 */
	public Editor_BARCODE() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * コンストラクター。
	 *
	 * @og.rev 7.4.2.2 (2021/05/28) 新規作成
	 *
	 * @param	clm	DBColumnオブジェクト
	 */
	protected Editor_BARCODE( final DBColumn clm ) {
		super( clm );

		// Attributesの連結記述
		attributes.set( "id", "outdata" );				// id 固定

		tagBuffer.add( XHTMLTag.inputAttri( attributes ) );
	}

	/**
	 * 各オブジェクトから自分のインスタンスを返します。
	 * 自分自身をキャッシュするのか、新たに作成するのかは、各サブクラスの実装に
	 * まかされます。
	 *
	 * @og.rev 7.4.2.2 (2021/05/28) 新規作成
	 *
	 * @param	clm	DBColumnオブジェクト
	 *
	 * @return	CellEditorオブジェクト
	 * @og.rtnNotNull
	 */
	public CellEditor newInstance( final DBColumn clm ) {
		return new Editor_BARCODE( clm );
	}

	/**
	 * データの編集用文字列を返します。
	 *
	 * @og.rev 7.4.2.2 (2021/05/28) 新規作成
	 *
	 * @param   value 入力値
	 *
	 * @return  データの編集用文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final String value ) {
		// TagBufferの連結記述
		return	BASE_HTML1
			+ CR + super.getValue( value )
			+ CR + BASE_HTML2 ;
	}

	/**
	 * name属性を変えた、データ表示/編集用のHTML文字列を作成します。
	 * テーブル上の name に 行番号を付加して、名前_行番号 で登録するキーを作成し,
	 * リクエスト情報を１つ毎のフィールドで処理できます。
	 *
	 * @og.rev 7.4.2.2 (2021/05/28) 新規作成
	 *
	 * @param   row   行番号
	 * @param   value 入力値
	 *
	 * @return  データ表示/編集用の文字列
	 * @og.rtnNotNull
	 */
	@Override
	public String getValue( final int row,final String value ) {
		// TagBufferの連結記述
		return	BASE_HTML1
			+ CR + super.getValue( row, value )
			+ CR + BASE_HTML2 ;
	}
}
