/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.hayabusa.taglib;

import org.opengion.hayabusa.common.HybsSystem;
import org.opengion.hayabusa.resource.ResourceManager;
import org.opengion.hayabusa.resource.ResourceFactory;
import org.opengion.fukurou.util.ToString;						// 6.1.1.0 (2015/01/17)

import static org.opengion.fukurou.util.StringUtil.nval ;

import javax.servlet.jsp.PageContext;

/**
 * ローカルリソースを構築します(有効範囲は、ページスコープ 内です)。
 *
 * ページスコープ内で使用されますので、主に、ページ先頭にて、ローカルリソースの
 * 設定を行います。そのページ内では、ここで定義した ResourceManager が使用されます。
 * これは、システムIDの異なる画面を、リソースを分離した形で使用できるようになります。
 * 例えば、systemId="ZY" のシステムに、systemId="MR" の画面を組み込む場合に、
 * その画面で、"MR" のリソースをそのままその画面に適用することで、同一カラム名に
 * 対して、異なる(その画面だけ、特別な)リソースを適用することが出来ます。
 *
 * @og.formSample
 * ●形式：&lt;og:localResource systemId="GE" lang="zh" initLoad="false" /&gt;
 * ●body：なし
 *
 * ●Tag定義：
 *   &lt;og:localResource
 *       systemId           【TAG】ローカルリソース設定用のシステムID をセットします
 *       lang               【TAG】ローカルリソースを構築する場合の、言語(lang)を指定します
 *       initLoad           【TAG】リソース情報の先読み込みの可否をセットします(初期値:false)
 *       remove             【TAG】ページスコープ内のResourceManager を削除します(初期値:false)
 *       debug              【TAG】デバッグ情報を出力するかどうか[true/false]を指定します(初期値:false)
 *   /&gt;
 *
 * ●使用例
 *      この設定以降のページにローカルリソースを適用します。
 *      ・&lt;og:localResource
 *               systemId  ="GE"       システムID
 *               lang      ="zh"       言語(ja,en,zh)
 *               initLoad  ="false"    初期読み込み可否(true:先読み込みする/false:しない)
 *        /&gt;
 *
 *      ページの途中でローカルリソースを破棄します。
 *      ・&lt;og:localResource
 *               remove  ="true"       ページスコープ内のResourceManager を削除します。
 *        /&gt;
 *
 * @og.group 画面制御
 *
 * @version  4.0
 * @author   Kazuhiko Hasegawa
 * @since    JDK5.0,
 */
public class LocalResourceTag extends CommonTagSupport {
	/** このプログラムのVERSION文字列を設定します。	{@value} */
	private static final String VERSION = "6.4.2.0 (2016/01/29)" ;
	private static final long serialVersionUID = 642020160129L ;

	private String	systemId	;		// システムID
	private String	lang		;		// 言語
	private boolean initLoad	;		// リソース初期読み込み可否(true:先読み込みする)
	private boolean remove		;		// ページスコープ内のリソースを削除します。

	/**
	 * デフォルトコンストラクター
	 *
	 * @og.rev 6.4.2.0 (2016/01/29) PMD refactoring. Each class should declare at least one constructor.
	 */
	public LocalResourceTag() { super(); }		// これも、自動的に呼ばれるが、空のメソッドを作成すると警告されるので、明示的にしておきます。

	/**
	 * Taglibの終了タグが見つかったときに処理する doEndTag() を オーバーライドします。
	 *
	 * @return	後続処理の指示
	 */
	@Override
	public int doEndTag() {
		debugPrint();		// 4.0.0 (2005/02/28)
		if( remove ) {
			pageContext.removeAttribute( HybsSystem.LOCAL_RES_KEY,PageContext.PAGE_SCOPE );
		}
		else {
			lang = nval( lang,getLanguage() );

			final ResourceManager resource ;
			if( systemId == null ) {
				resource = ResourceFactory.newInstance( lang );
			}
			else {
				resource = ResourceFactory.newInstance( systemId,lang,initLoad );
			}

			pageContext.setAttribute( HybsSystem.LOCAL_RES_KEY,resource );
		}

		return EVAL_PAGE ;
	}

	/**
	 * タグリブオブジェクトをリリースします。
	 * キャッシュされて再利用されるので、フィールドの初期設定を行います。
	 *
	 */
	@Override
	protected void release2() {
		super.release2();
		systemId = null;
		lang     = null;
		initLoad = false;
		remove   = false;
	}

	/**
	 * 【TAG】ローカルリソース設定用のシステムID をセットします。
	 *
	 * @og.tag
	 * このシステムIDを利用して、ResourceManager を構築します。
	 * 指定のない場合は、システムプロパティーの SYSTEM_ID パラメータ値です。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @param	sysId	システムID
	 */
	public void setSystemId( final String sysId ) {
		systemId = nval( getRequestParameter( sysId ),systemId );
	}

	/**
	 * 【TAG】ローカルリソースを構築する場合の、言語(lang)を指定します。
	 *
	 * @og.tag ここで言語を指定しない場合は、language 属性が使用されます。
	 *
	 * @param   lng 言語
	 */
	public void setLang( final String lng ) {
		lang= nval( getRequestParameter( lng ),lang );
	}

	/**
	 * 【TAG】リソース情報の先読み込みの可否[true:する/false:しない]をセットします(初期値:false)。
	 *
	 * @og.tag
	 * リソース情報は、FGLOADフラグが'1'の情報を先読み込みします。
	 * ローカルリソースは、全リソースを使用することはないため、通常、
	 * 先読み込みする必要はありません。また、一度読み込んだリソースは、
	 * 内部でキャッシュされるため、最初の使用時のみ、DBアクセスが発生します。
	 * (先読み込みは、Tomcat起動時の、初めてのアクセス時に行われます。)
	 * 初期値は、false:先読みしないです。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @param	inLoad	先読み可否 [true:する/false:しない]
	 */
	public void setInitLoad( final String inLoad ) {
		initLoad = nval( getRequestParameter( inLoad ),initLoad );
	}

	/**
	 * 【TAG】ページスコープ内のResourceManager を削除します(初期値:false)。
	 *
	 * @og.tag
	 * ページスコープ内のResourceManager を削除すると、初期化されます。
	 * これは、ページ内の部分的にリソースを適用する場合に、使用します。
	 * 初期値は、削除しない(false)です。
	 *
	 * @og.rev 4.0.0.0 (2005/01/31) 新規追加
	 *
	 * @param   flag リソースの削除 [true:削除する/false:削除しない]
	 */
	public void setRemove( final String flag ) {
		remove = nval( getRequestParameter( flag ),remove );
	}

	/**
	 * このオブジェクトの文字列表現を返します。
	 * 基本的にデバッグ目的に使用します。
	 *
	 * @return このクラスの文字列表現
	 * @og.rtnNotNull
	 */
	@Override
	public String toString() {
		return ToString.title( this.getClass().getName() )
				.println( "VERSION"		,VERSION	)
				.println( "systemId"	,systemId	)
				.println( "lang"		,lang		)
				.println( "initLoad"	,initLoad	)
				.println( "remove"		,remove		)
				.println( "Other..."	,getAttributes().getAttribute() )
				.fixForm().toString() ;
	}
}
