/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.opengion.fukurou.util;

import org.opengion.fukurou.system.OgRuntimeException ;				// 6.4.2.0 (2016/01/29)
import java.awt.Color;
// import java.util.LinkedHashMap;									// 6.4.6.1 (2016/06/03)
import java.util.Map;
import java.util.HashMap;											// 7.0.1.1 (2018/10/22)
// import java.util.List;											// 6.9.9.2 (2018/09/18)
// import java.util.ArrayList;										// 6.9.9.2 (2018/09/18)
import java.util.Locale ;

import static org.opengion.fukurou.system.HybsConst.CR;				// 6.1.0.0 (2014/12/26) refactoring

/**
 * ColorMap.java は、共通的に使用される Color関連メソッドを集約した、クラスです。
 * StringUtil から、Color関係の部分を独立させました。
 *
 * カラー文字列は、java.awt.Color static 定義色、CSS1 / HTML3–4 / VGA の色名称、
 * X11の色名称 が指定できます。
 *
 * カラー番号は、ビビッドを、0～11 (V0～V11) , パステルを、12～23 (P0～P11) として登録します。
 * 従来のカラー番号と異なりますので、ご注意ください。
 *
 * @og.group ユーティリティ
 * @og.rev 6.0.2.1 (2014/09/26) 新規作成
 * @og.rev 6.4.3.3 (2016/03/04) ViewForm_ImageTimeBar#FlgColorMap 内部クラスから機能移動
 * @og.rev 7.0.1.1 (2018/10/22) 内部色コードを文字列として管理、配列は、パステルとビビッドから作成します。
 *
 * @version  6.0
 * @author	 Kazuhiko Hasegawa
 * @since    JDK8.0,
 */
public final class ColorMap {
//	// 6.9.9.2 (2018/09/18) Map から、CLR_ARY を作成します。
//	// 6.4.3.3 (2016/03/04) ViewForm_ImageTimeBar#FlgColorMap 内部クラスから移動
//	private static final Color[] CLR_ARY = new Color[] {
//			Color.BLUE      ,Color.CYAN   ,Color.GRAY ,Color.GREEN ,Color.LIGHT_GRAY ,Color.MAGENTA ,
//			Color.DARK_GRAY ,Color.ORANGE ,Color.PINK ,Color.RED   ,Color.YELLOW
//	};

	/** 色種別ｷｰﾜｰﾄﾞ	{@value}	*/
	public static final String VIVID  = "VIVID"  ;		// 7.0.1.3 (2018/11/12)
	/** 色種別ｷｰﾜｰﾄﾞ	{@value}	*/
	public static final String PASTEL = "PASTEL" ;		// 7.0.1.3 (2018/11/12)

	// 7.0.1.1 (2018/10/22) ビビッドカラー の色ｺｰﾄﾞ配列追加
	private static final String[] VIVID_KEY = new String[] {
		"#0000FF" , "#00FF00" , "#FF0000" , 			// 5,  9, 1
		"#00FFFF" , "#FFFF00" , "#FF00FF" , 			// 7, 11, 3
		"#00FF7F" , "#FF7F00" , "#7F00FF" ,	 			// 8, 12, 4
		"#007FFF" , "#7FFF00" , "#FF007F" 				// 6, 10, 2
 	};

	// 6.9.9.2 (2018/09/18) パステルカラー の色ｺｰﾄﾞ配列追加
	private static final String[] PSTL_KEY = new String[] {
		"#7F7FFF" , "#7FFF7F" ,  "#FF7F7F" , 			// 5,  9, 1
		"#7FFFFF" , "#FFFF7F" ,  "#FF7FFF" , 			// 7, 11, 3
		"#7FFFBF" , "#FFBF7F" ,  "#BF7FFF" ,  			// 8, 12, 4
		"#7FBFFF" , "#BFFF7F" ,  "#FF7FBF"  			// 6, 10, 2
 	};

	private static final String[] COLOR_KEY = new String[VIVID_KEY.length + PSTL_KEY.length];
	static {
		System.arraycopy( VIVID_KEY, 0, COLOR_KEY, 0				, VIVID_KEY.length );
		System.arraycopy( PSTL_KEY , 0, COLOR_KEY, VIVID_KEY.length	, PSTL_KEY.length );
	}

//
//	// 4.0.3.0 (2007/12/26) 色コードにPURPLE を追加
//	// 5.7.8.0 (2014/07/04) 透明追加
//	private static final List<Color>		CLR_LIST = new ArrayList<>();			// 6.9.9.2 (2018/09/18) 色配列ではなく、リストに変更します。
	/** staticイニシャライザ。読み取り専用(Collections.unmodifiableMap(Map))や、ConcurrentHashMap を使用しません。 */
//	private static final Map<String,Color>	CLR_MAP  = new LinkedHashMap<>();		// 6.4.6.1 (2016/06/03) 登録順をキープします。
	private static final Map<String,String>	CLR_MAP  = new HashMap<>();			// 7.0.1.1 (2018/10/22) 文字列として管理します。
	static {
		// java.awt.Color static 定義色
		CLR_MAP.put( "BLUE"			, "#0000FF" );		// Color.BLUE
		CLR_MAP.put( "CYAN"			, "#00FFFF" );		// Color.CYAN
		CLR_MAP.put( "GRAY"			, "#808080" );		// Color.GRAY
		CLR_MAP.put( "GREEN"		, "#008000" );		// Color.GREEN
		CLR_MAP.put( "LIGHTGRAY"	, "#D3D3D3" );		// Color.LIGHT_GRAY
		CLR_MAP.put( "MAGENTA"		, "#FF00FF" );		// Color.MAGENTA
		CLR_MAP.put( "DARKGRAY"		, "#A9A9A9" );		// Color.DARK_GRAY
		CLR_MAP.put( "ORANGE"		, "#FFA500" );		// Color.ORANGE
		CLR_MAP.put( "PINK"			, "#FFC0CB" );		// Color.PINK
		CLR_MAP.put( "RED"			, "#FF0000" );		// Color.RED
		CLR_MAP.put( "YELLOW"		, "#FFFF00" );		// Color.YELLOW

//		// 6.9.9.2 (2018/09/18) CSS1 / HTML3–4 / VGA の色名称の後ろに移動。
//		CLR_MAP.put( "WHITE"		,Color.WHITE		);
//		CLR_MAP.put( "BLACK"		,Color.BLACK		);
//		CLR_MAP.put( "TRANSPARENT"	,new Color( 255,255,255,0 )	);		// 5.7.8.0 (2014/07/04) 透明追加

		// 6.0.2.1 (2014/09/26) CSS1 / HTML3–4 / VGA の色名称 追加
		CLR_MAP.put( "LIME"			, "#00FF00" );
		CLR_MAP.put( "AQUA"			, "#00FFFF" );
		CLR_MAP.put( "SILVER"		, "#C0C0C0" );
		CLR_MAP.put( "MAROON"		, "#800000" );		// X11では、#B03060
		CLR_MAP.put( "OLIVE"		, "#808000" );
		CLR_MAP.put( "TEAL"			, "#008080" );
		CLR_MAP.put( "NAVY"			, "#000080" );
		CLR_MAP.put( "FUCHSIA"		, "#FF00FF" );
		CLR_MAP.put( "PURPLE"		, "#800080" );		// X11では、#A020F0

//		// 6.9.9.2 (2018/09/18) Map から、CLR_LIST を作成するのは、ここまで。
//		CLR_LIST.addAll( CLR_MAP.values() );
//
//		// 6.9.9.2 (2018/09/18) パステルカラー の色ｺｰﾄﾞ配列追加
//		for( final String col : PSTL_KEY ) {
//			CLR_LIST.add( Color.decode( col ) );
//		}

		CLR_MAP.put( "WHITE"		, "#FFFFFF" );		// WHITE
		CLR_MAP.put( "BLACK"		, "#000000" );		// BLACK
//		CLR_MAP.put( "TRANSPARENT"	, new Color( 255,255,255,0 )	);		// 5.7.8.0 (2014/07/04) 透明追加
		CLR_MAP.put( "TRANSPARENT"	, "rgba( 255,255,255,0 )"		);		// 7.0.1.2 (2018/11/04) 透明

		// 6.0.2.1 (2014/09/26) X11の色名称 追加
		CLR_MAP.put( "ALICEBLUE"			, "#F0F8FF" );
		CLR_MAP.put( "ANTIQUEWHITE"			, "#FAEBD7" );
		CLR_MAP.put( "AQUAMARINE"			, "#7FFFD4" );
		CLR_MAP.put( "AZURE"				, "#F0FFFF" );
		CLR_MAP.put( "BEIGE"				, "#F5F5DC" );
		CLR_MAP.put( "BISQUE"				, "#FFE4C4" );
		CLR_MAP.put( "BLANCHEDALMOND"		, "#FFEBCD" );
		CLR_MAP.put( "BLUEVIOLET"			, "#8A2BE2" );
		CLR_MAP.put( "BROWN"				, "#A52A2A" );
		CLR_MAP.put( "BURLYWOOD"			, "#DEB887" );
		CLR_MAP.put( "CADETBLUE"			, "#5F9EA0" );
		CLR_MAP.put( "CHARTREUSE"			, "#7FFF00" );
		CLR_MAP.put( "CHOCOLATE"			, "#D2691E" );
		CLR_MAP.put( "CORAL"				, "#FF7F50" );
		CLR_MAP.put( "CORNFLOWERBLUE"		, "#6495ED" );
		CLR_MAP.put( "CORNSILK"				, "#FFF8DC" );
		CLR_MAP.put( "CRIMSON"				, "#DC143C" );
		CLR_MAP.put( "DARKBLUE"				, "#00008B" );
		CLR_MAP.put( "DARKCYAN"				, "#008B8B" );
		CLR_MAP.put( "DARKGOLDENROD"		, "#B8860B" );
		CLR_MAP.put( "DARKGREEN"			, "#006400" );
		CLR_MAP.put( "DARKKHAKI"			, "#BDB76B" );
		CLR_MAP.put( "DARKMAGENTA"			, "#8B008B" );
		CLR_MAP.put( "DARKOLIVEGREEN"		, "#556B2F" );
		CLR_MAP.put( "DARKORANGE"			, "#FF8C00" );
		CLR_MAP.put( "DARKORCHID"			, "#9932CC" );
		CLR_MAP.put( "DARKRED"				, "#8B0000" );
		CLR_MAP.put( "DARKSALMON"			, "#E9967A" );
		CLR_MAP.put( "DARKSEAGREEN"			, "#8FBC8F" );
		CLR_MAP.put( "DARKSLATEBLUE"		, "#483D8B" );
		CLR_MAP.put( "DARKSLATEGRAY"		, "#2F4F4F" );
		CLR_MAP.put( "DARKTURQUOISE"		, "#00CED1" );
		CLR_MAP.put( "DARKVIOLET"			, "#9400D3" );
		CLR_MAP.put( "DEEPPINK"				, "#FF1493" );
		CLR_MAP.put( "DEEPSKYBLUE"			, "#00BFFF" );
		CLR_MAP.put( "DIMGRAY"				, "#696969" );
		CLR_MAP.put( "DODGERBLUE"			, "#1E90FF" );
		CLR_MAP.put( "FIREBRICK"			, "#B22222" );
		CLR_MAP.put( "FLORALWHITE"			, "#FFFAF0" );
		CLR_MAP.put( "FORESTGREEN"			, "#228B22" );
		CLR_MAP.put( "GAINSBORO"			, "#DCDCDC" );
		CLR_MAP.put( "GHOSTWHITE"			, "#F8F8FF" );
		CLR_MAP.put( "GOLD"					, "#FFD700" );
		CLR_MAP.put( "GOLDENROD"			, "#DAA520" );
		CLR_MAP.put( "GREENYELLOW"			, "#ADFF2F" );
		CLR_MAP.put( "HONEYDEW"				, "#F0FFF0" );
		CLR_MAP.put( "HOTPINK"				, "#FF69B4" );
		CLR_MAP.put( "INDIANRED"			, "#CD5C5C" );
		CLR_MAP.put( "INDIGO"				, "#4B0082" );
		CLR_MAP.put( "IVORY"				, "#FFFFF0" );
		CLR_MAP.put( "KHAKI"				, "#F0E68C" );
		CLR_MAP.put( "LAVENDER"				, "#E6E6FA" );
		CLR_MAP.put( "LAVENDERBLUSH"		, "#FFF0F5" );
		CLR_MAP.put( "LAWNGREEN"			, "#7CFC00" );
		CLR_MAP.put( "LEMONCHIFFON"			, "#FFFACD" );
		CLR_MAP.put( "LIGHTBLUE"			, "#ADD8E6" );
		CLR_MAP.put( "LIGHTCORAL"			, "#F08080" );
		CLR_MAP.put( "LIGHTCYAN"			, "#E0FFFF" );
		CLR_MAP.put( "LIGHTGOLDENRODYELLOW"	, "#FAFAD2" );
		CLR_MAP.put( "LIGHTGREEN"			, "#90EE90" );
		CLR_MAP.put( "LIGHTPINK"			, "#FFB6C1" );
		CLR_MAP.put( "LIGHTSALMON"			, "#FFA07A" );
		CLR_MAP.put( "LIGHTSEAGREEN"		, "#20B2AA" );
		CLR_MAP.put( "LIGHTSKYBLUE"			, "#87CEFA" );
		CLR_MAP.put( "LIGHTSLATEGRAY"		, "#778899" );
		CLR_MAP.put( "LIGHTSTEELBLUE"		, "#B0C4DE" );
		CLR_MAP.put( "LIGHTYELLOW"			, "#FFFFE0" );
		CLR_MAP.put( "LIMEGREEN"			, "#32CD32" );
		CLR_MAP.put( "LINEN"				, "#FAF0E6" );
		CLR_MAP.put( "MEDIUMAQUAMARINE"		, "#66CDAA" );
		CLR_MAP.put( "MEDIUMBLUE"			, "#0000CD" );
		CLR_MAP.put( "MEDIUMORCHID"			, "#BA55D3" );
		CLR_MAP.put( "MEDIUMPURPLE"			, "#9370DB" );
		CLR_MAP.put( "MEDIUMSEAGREEN"		, "#3CB371" );
		CLR_MAP.put( "MEDIUMSLATEBLUE"		, "#7B68EE" );
		CLR_MAP.put( "MEDIUMSPRINGGREEN"	, "#00FA9A" );
		CLR_MAP.put( "MEDIUMTURQUOISE"		, "#48D1CC" );
		CLR_MAP.put( "MEDIUMVIOLETRED"		, "#C71585" );
		CLR_MAP.put( "MIDNIGHTBLUE"			, "#191970" );
		CLR_MAP.put( "MINTCREAM"			, "#F5FFFA" );
		CLR_MAP.put( "MISTYROSE"			, "#FFE4E1" );
		CLR_MAP.put( "MOCCASIN"				, "#FFE4B5" );
		CLR_MAP.put( "NAVAJOWHITE"			, "#FFDEAD" );
		CLR_MAP.put( "OLDLACE"				, "#FDF5E6" );
		CLR_MAP.put( "OLIVEDRAB"			, "#6B8E23" );
		CLR_MAP.put( "ORANGERED"			, "#FF4500" );
		CLR_MAP.put( "ORCHID"				, "#DA70D6" );
		CLR_MAP.put( "PALEGOLDENROD"		, "#EEE8AA" );
		CLR_MAP.put( "PALEGREEN"			, "#98FB98" );
		CLR_MAP.put( "PALETURQUOISE"		, "#AFEEEE" );
		CLR_MAP.put( "PALEVIOLETRED"		, "#DB7093" );
		CLR_MAP.put( "PAPAYAWHIP"			, "#FFEFD5" );
		CLR_MAP.put( "PEACHPUFF"			, "#FFDAB9" );
		CLR_MAP.put( "PERU"					, "#CD853F" );
		CLR_MAP.put( "PLUM"					, "#DDA0DD" );
		CLR_MAP.put( "POWDERBLUE"			, "#B0E0E6" );
		CLR_MAP.put( "ROSYBROWN"			, "#BC8F8F" );
		CLR_MAP.put( "ROYALBLUE"			, "#4169E1" );
		CLR_MAP.put( "SADDLEBROWN"			, "#8B4513" );
		CLR_MAP.put( "SALMON"				, "#FA8072" );
		CLR_MAP.put( "SANDYBROWN"			, "#F4A460" );
		CLR_MAP.put( "SEAGREEN"				, "#2E8B57" );
		CLR_MAP.put( "SEASHELL"				, "#FFF5EE" );
		CLR_MAP.put( "SIENNA"				, "#A0522D" );
		CLR_MAP.put( "SKYBLUE"				, "#87CEEB" );
		CLR_MAP.put( "SLATEBLUE"			, "#6A5ACD" );
		CLR_MAP.put( "SLATEGRAY"			, "#708090" );
		CLR_MAP.put( "SNOW"					, "#FFFAFA" );
		CLR_MAP.put( "SPRINGGREEN"			, "#00FF7F" );
		CLR_MAP.put( "STEELBLUE"			, "#4682B4" );
		CLR_MAP.put( "TAN"					, "#D2B48C" );
		CLR_MAP.put( "THISTLE"				, "#D8BFD8" );
		CLR_MAP.put( "TOMATO"				, "#FF6347" );
		CLR_MAP.put( "TURQUOISE"			, "#40E0D0" );
		CLR_MAP.put( "VIOLET"				, "#EE82EE" );
		CLR_MAP.put( "WHEAT"				, "#F5DEB3" );
		CLR_MAP.put( "WHITESMOKE"			, "#F5F5F5" );
		CLR_MAP.put( "YELLOWGREEN"			, "#9ACD32" );

//		// 6.4.4.1 (2016/03/18) 数字に対応する
//		for( int i=0; i<CLR_ARY.length; i++ ) {
//			CLR_MAP.put( String.valueOf( i ) ,CLR_ARY[i] );			// 0:BLUE , 1:CYAN , 2:GRAY ･･･ を、Mapに登録しておきます。
//		}

//		// 6.9.9.2 (2018/09/18) 数字に対応する(配列からリストに変更の対応)
//		for( int i=0; i<CLR_LIST.size(); i++ ) {
//			CLR_MAP.put( String.valueOf( i ) ,CLR_LIST.get(i) );	// 0:BLUE , 1:CYAN , 2:GRAY ･･･ を、Mapに登録しておきます。
//		}

		// 7.0.1.1 (2018/10/22) 色番号も、Mapに登録しておきます。
		// ビビッドを、0～11 (V0～V11) , パステルを、11～23 (P0～P11) とします。
		int cno = 0;
		for( int i=0; i<VIVID_KEY.length; i++ ) {
			CLR_MAP.put( String.valueOf( cno++ ) , VIVID_KEY[i] );		// 0～11
			CLR_MAP.put( "V" + i				 , VIVID_KEY[i] );		// V0～V11
		}

		for( int i=0; i<PSTL_KEY.length; i++ ) {
			CLR_MAP.put( String.valueOf( cno++ ) , PSTL_KEY[i] );		// 11～23
			CLR_MAP.put( "P" + i				 , PSTL_KEY[i] );		// P0～P11
		}

		// 7.0.1.3 (2018/11/12) VIVIDとPASTELというｷｰﾜｰﾄﾞに、V0とP0を割り当てておきます。
		CLR_MAP.put( VIVID  , VIVID_KEY[0] );		// V0
		CLR_MAP.put( PASTEL , PSTL_KEY[0] );		// P0
	}

	/**
	 *	デフォルトコンストラクターをprivateにして、
	 *	オブジェクトの生成をさせないようにする。
	 *
	 */
	private ColorMap() {}

	/**
	 * カラーキーワードより、Colorオブジェクトを作成します。
	 *
	 * 指定文字列は、java.awt.Color クラスのstatic フィールド名で指定します。
	 * BLACK , BLUE , CYAN , DARK_GRAY , GRAY , GREEN , LIGHT_GRAY ,
	 * MAGENTA , ORANGE , PINK , RED , WHITE , YELLOW , PURPLE , TRANSPARENT(透明) が指定できます。
	 *
	 * 6.0.2.1 (2014/09/26) 以降では、CSS1/HTML3–4/VGA の色名称として、
	 * LIME , AQUA , SILVER , MAROON , OLIVE , TEAL , NAVY , FUCHSIA を追加。
	 * それと、X11の色名称 を追加しました。
	 *
	 * また、先頭に、# を付ける事で、#XXXXXX形式の16bitRGB表記 でも指定可能です。
	 * static フィールド名のMapを管理していますが、存在しない場合は、エラーになります。
	 *
	 * 色番号は、ビビッドを、0～11 (V0～V11) , パステルを、11～23 (P0～P11) とします。
	 *
	 * @og.rev 3.8.9.1 (2007/06/29) 新規作成
	 * @og.rev 4.1.1.0 (2008/02/04) CLR_MAP に存在しない場合はエラーにします。
	 * @og.rev 6.0.2.1 (2014/09/26) CSS1/HTML3–4/VGA の色名称 , X11の色名称 追加
	 * @og.rev 6.3.9.0 (2015/11/06) valueのnullチェックの仕方が間違っていた。
	 * @og.rev 6.4.6.1 (2016/06/03) 色番号の取得も対応しておきます。
	 * @og.rev 7.0.1.1 (2018/10/22) ビビッドとパステルの全24色を対象とします。
	 * @og.rev 7.0.6.4 (2019/11/29) 数値指定の色がMapにないとｴﾗｰになっていた個所を修正
	 *
	 * @param	value java.awt.Color フィールドを示す文字列または、#XXXXXX形式の16bitRGB表記
	 *
	 * @return	Colorオブジェクト
	 * @see		java.awt.Color#BLACK
	 */
	public static Color getColorInstance( final String value ) {
		// 6.3.9.0 (2015/11/06) valueのnullチェックの仕方が間違っていた。
		if( value == null || value.length() == 0 ) {
			final String errMsg = "色コードが指定されていません。" + CR			// 6.4.6.1 (2016/06/03)
						+ "ColorMap=" + CLR_MAP.keySet().toString();
			throw new OgRuntimeException( errMsg );
		}

		// 色コードの番号(#XXXXXX) Mapになければエラー
		final String colCode ;
		if( value.charAt(0) == '#' ) {
			colCode = value;
		}
		else {
			final String val = value.replaceAll( "_" , "" ).toUpperCase(Locale.JAPAN);
			colCode = CLR_MAP.get( val );

			if( colCode == null ) {
				// 7.0.6.4 (2019/11/29) 数値指定の色がMapにないとｴﾗｰになっていた個所を修正
				try {
					return getColorInstance( Integer.parseInt( val ) );
				}
				catch( final NumberFormatException ex ) {			// 数字ではない value をパースしているので、この、Exception は伝播不要。
					final String errMsg = "指定の色コードはカラーマップに存在しません。 Color=[" + value + "]" + CR
								+ "ColorMap=" + CLR_MAP.keySet().toString();
					throw new OgRuntimeException( errMsg,ex );
				}
			}
		}

		try {
			return Color.decode( colCode );
		}
		catch( final NumberFormatException ex ) {			// 10進、8進、および16進の整数値に変換できない場合
			final String errMsg = "指定の色コードのフォーマットが異なります(#XXXXXX形式の16bitRGB表記) Color=[" + value + "]" ;
			throw new OgRuntimeException( errMsg,ex );
		}

//		Color clr = null ;			// 6.4.6.1 (2016/06/03) 色番号で、再設定しているため、final をはずす。
//
//		if( value.charAt(0) == '#' ) {
//			try {
//				clr = Color.decode( value );
//			}
//			catch( final NumberFormatException ex ) {			// 10進、8進、および16進の整数値に変換できない場合
//				final String errMsg = "指定の色コードのフォーマットが異なります Color=[" + value + "]" + CR
//							+ "ColorMap=" + CLR_MAP.keySet().toString();
//				throw new OgRuntimeException( errMsg,ex );
//			}
//		}
//		else {
//			// "_"を削除するのは、LIGHT_GRAY などの表記を統一するため。
//			final String val = value.replaceAll( "_" , "" ).toUpperCase(Locale.JAPAN);
//			clr = CLR_MAP.get( val );
//			if( clr == null ) {
//				// 6.4.6.1 (2016/06/03) 色番号の取得も対応しておきます。
//				// 0 ～ 10 は登録済みなので、それ以降の数字の繰り返しが出来ない
//				try {
//					clr = getColorInstance( Integer.parseInt( value ) );
//				}
//				catch( final NumberFormatException ex ) {			// 数字ではない value をパースしているので、この、Exception は伝播不要。
//					final String errMsg = "指定の色コードは使用できません Color=[" + value + "]" + CR
//								+ "ColorMap=" + CLR_MAP.keySet().toString();
//					throw new OgRuntimeException( errMsg,ex );
//				}
//			}
//		}
//		return clr;
	}

	/**
	 * 指定の番号に対応した、色オブジェクトを返します。
	 *
	 * カラー番号は、ビビッドを、0～11 , パステルを、12～23 として登録しています。
	 * 従来のカラー番号と異なりますので、ご注意ください。
	 *
	 * 23を超えると、0 に戻ります。
	 *
	 * @og.rev 6.4.3.3 (2016/03/04) ViewForm_ImageTimeBar#uniqColor() メソッドから移動。
	 * @og.rev 6.9.9.2 (2018/09/18) 配列からリストに変更の対応
	 * @og.rev 7.0.1.1 (2018/10/22) ビビッドとパステルの全24色を対象とします。
	 *
	 * @param	no 内部で管理している色番号
	 * @return	色オブジェクト
	 */
	public static Color getColorInstance( final int no ) {
		try {
			return Color.decode( COLOR_KEY[no % COLOR_KEY.length] );
		}
		catch( final NumberFormatException ex ) {
			final String errMsg = "指定の色番号の内部フォーマットがエラーです。no=[" + no + "] , code=[" + COLOR_KEY[no % COLOR_KEY.length] + "]" ;
			throw new OgRuntimeException( errMsg,ex );
		}
	}

	/**
	 * 内部で管理している ColorMap のキー配列を返します。
	 *
	 * ここで返すのは、ビビッドを、0～11 , パステルを、12～23 として登録しています。
	 * この、#XXXXXX形式の16bitRGB表記 を返します。
	 *
	 * @og.rev 6.0.2.1 (2014/09/26) 新規追加
	 * @og.rev 7.0.1.1 (2018/10/22) ビビッドとパステルの全24色を対象とします。
	 *
	 * @param	no 内部で管理している色番号
	 * @return	キー配列
	 * @og.rtnNotNull
	 */
	public static String getColorKey( final int no ) {
		return COLOR_KEY[no % COLOR_KEY.length];
	}

	/**
	 * 内部で管理している ColorMap から、指定のキーに対する色コードを返します。
	 *
	 * 色名称でも返しますが、主に、色番号を想定しています。
	 * 引数が、文字列なのは、ビビッドを、0～11 (V0～V11) , パステルを、12～23 (P0～P11)
	 * として管理しているためです。
	 *
	 * 色コードは、、#XXXXXX形式の16bitRGB表記です。
	 * 色マップに値が存在しない場合は、null を返します。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) ビビッドとパステルの全24色を対象とします。
	 *
	 * @param	sno 内部で管理している色番号
	 * @return	色コード(対象の色がMapに存在しない場合は、nullを返します)
	 */
	public static String getColorKey( final String sno ) {
//		return CLR_MAP.get( sno );
		return getColorKey( sno , null );
	}

	/**
	 * 内部で管理している ColorMap から、指定のキーに対する色コードを返します。
	 *
	 * 色名称でも返しますが、主に、色番号を想定しています。
	 * 引数が、文字列なのは、ビビッドを、0～11 (V0～V11) , パステルを、12～23 (P0～P11)
	 * として管理しているためです。
	 *
	 * 色コードは、、#XXXXXX形式の16bitRGB表記です。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) ビビッドとパステルの全24色を対象とします。
	 *
	 * @param	sno 内部で管理している色番号,色記号
	 * @param	defCol 色マップに値が存在しない場合に返す色
	 * @return	色コード
	 */
	public static String getColorKey( final String sno , final String defCol ) {
		return CLR_MAP.getOrDefault( sno , defCol );
	}

	/**
	 * 内部で管理している ColorMap のキー配列を返します。
	 *
	 * ここで返すのは、ビビッドを、0～11 , パステルを、12～23 として登録しています。
	 * この、#XXXXXX形式の16bitRGB表記 を返します。
	 *
	 * @og.rev 6.0.2.1 (2014/09/26) 新規追加
	 * @og.rev 7.0.1.1 (2018/10/22) ビビッドとパステルの全24色を対象とします。
	 *
	 * @return	キー配列
	 * @og.rtnNotNull
	 */
	public static String[] getColorKeys() {
//		return CLR_MAP.keySet().toArray( new String[CLR_MAP.size()] );
		return COLOR_KEY.clone();
	}

	/**
	 * 引数に応じた内部で管理している ColorMap のキー配列を返します。
	 *
	 * colKey が null の場合は、#getColorKeys() と同じ、COLOR_KEYすべてを返します。
	 * colKey が、VIVID の場合は、#getVividKeys() を返します。
	 * colKey が、PASTEL の場合は、#getPastelKeys() を返します。
	 * それ以外は、CSVで分解して、各値を、#getColorKey(String,String) で求めた色コードの
	 * 配列を返します。
	 *
	 * @og.rev 7.0.1.3 (2018/11/12) 引数に応じたColorMap のキー配列を返す。
	 *
	 * @param	colKey 内部で管理している色コード(nullの場合は、COLOR_KEYすべて。)
	 * @return	キー配列
	 * @og.rtnNotNull
	 */
	public static String[] getColorKeys( final String colKey ) {
		final String[] rtnCols ;

		if( colKey == null || colKey.isEmpty() ) {     rtnCols = getColorKeys(); }
		else if( VIVID.equalsIgnoreCase(  colKey ) ) { rtnCols = getVividKeys(); }
		else if( PASTEL.equalsIgnoreCase( colKey ) ) { rtnCols = getPastelKeys(); }
		else {
			rtnCols = StringUtil.csv2Array( colKey );
			for( int i=0; i<rtnCols.length; i++ ) {
				rtnCols[i] = getColorKey( rtnCols[i] , rtnCols[i] );
			}
		}

		return rtnCols ;
	}

//	/**
//	 * 指定の番号に対応した、ビビッドカラーの色オブジェクトを返します。
//	 *
//	 * #getColorInstance(int) とほぼ同じ機能で、返すのは、パステルカラーの色オブジェクトです。
//	 * https://www.colordic.org/v/
//	 *
//	 * １２色のビビッドカラーを、指定の番号に応じて、返します。
//	 * 隣接する色は、できるだけ離れるようにしています。
//	 *
//	 * @og.rev 7.0.1.1 (2018/10/22) ビビッドとパステルの全24色を対象とします。
//	 * @og.rev 7.0.1.3 (2018/11/12) 廃止
//	 *
//	 * @param	no 内部で管理している色番号
//	 * @return	ビビッドカラーの色コード
//	 */
//	public static String getVividKey( final int no ) {
//		return VIVID_KEY[no % VIVID_KEY.length];
//	}

	/**
	 * 内部で管理している ビビッドカラー のキー配列を返します。
	 *
	 * @og.rev 7.0.1.1 (2018/10/22) ビビッドとパステルの全24色を対象とします。
	 *
	 * @return	パステルカラーのキー配列
	 */
	public static String[] getVividKeys() {
		return VIVID_KEY.clone();
	}

//	/**
//	 * 指定の番号に対応した、パステルカラーの色オブジェクトを返します。
//	 *
//	 * #getColorInstance(int) とほぼ同じ機能で、返すのは、パステルカラーの色オブジェクトです。
//	 * https://www.colordic.org/p/
//	 *
//	 * １２色のパステルカラーを、指定の番号に応じて、返します。
//	 * 隣接する色は、できるだけ離れるようにしています。
//	 *
//	 * @og.rev 6.9.9.2 (2018/09/18) パステルカラーの色オブジェクト
//	 * @og.rev 7.0.1.3 (2018/11/12) 廃止
//	 *
//	 * @param	no 内部で管理している色番号
//	 * @return	パステルカラーの色コード
//	 */
//	public static String getPastelKey( final int no ) {
//		return PSTL_KEY[no % PSTL_KEY.length];
//	}

	/**
	 * 内部で管理している パステルカラー のキー配列を返します。
	 *
	 * @og.rev 6.9.9.2 (2018/09/18) パステルカラーの色オブジェクト
	 *
	 * @return	パステルカラーのキー配列
	 */
	public static String[] getPastelKeys() {
		return PSTL_KEY.clone();
	}
}
