/*
 * Copyright (c) 2009 The openGion Project.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package org.opengion.fukurou.queue;

import javax.jms.QueueSession;

/**
 * キュー情報格納クラス キュー情報を格納するクラスです。
 * 
 * @og.rev 5.10.14.0 (2019/08/01) 新規作成
 *
 */
public class QueueInfo {
	// jmsServer
	private String jmsServer;
	// メッセージ
	private String message;
	// SQSFIFOグループID
	private String sqsFifoGroupId;
	// SQSFIFO重複排除ID
	private String sqsFifoDedupliId;
	// MQキュー名
	private String mqQueueName;
	// MQトランザクション処理判別
	private Boolean mqTransacted;
	// MQ確認応答モード
	private int mqAcknowledgeMode;;

	/**
	 * コントラクタ パラメータを初期化します。
	 */
	public QueueInfo() {
		jmsServer			= null;
		message				= null;
		sqsFifoGroupId		= null;
		sqsFifoDedupliId	= null;
		mqQueueName			= null;
		mqTransacted		= false;
		mqAcknowledgeMode	= QueueSession.AUTO_ACKNOWLEDGE;
	}

	/** Getter：Setter */
	/**
	 * メッセージを取得します。
	 * 
	 * @return メッセージ
	 */
	public String getMessage() {
		return message;
	}

	/**
	 * メッセージを設定します。
	 * 
	 * @param mess メッセージ
	 */
	public void setMessage(final String mess) {
		message = mess;
	}

	/**
	 * Jmsサーバを取得します。
	 * 
	 * @return JMSサーバ
	 */
	public String getJmsServer() {
		return jmsServer;
	}

	/**
	 * Jmsサーバーを設定します。
	 * 
	 * @param server Jmsサーバ(コンテキストに定義された名称)
	 */
	public void setJmsServer(final String server) {
		jmsServer = server;
	}

	/**
	 * SQSFIFO用のグループIDを取得します。
	 * 
	 * @return SQSFIFO用グループID
	 */
	public String getSqsFifoGroupId() {
		return sqsFifoGroupId;
	}

	/**
	 * SQSFIFO用のグループIDを設定します。
	 * 
	 * @param id SQSFIFO用グループID
	 */
	public void setSqsFifoGroupId(final String id) {
		sqsFifoGroupId = id;
	}

	/**
	 * SQSFIFO用の重複排除IDを取得します。
	 * 
	 * @return SQSFIFO用の重複排除ID
	 */
	public String getSqsFifoDedupliId() {
		return sqsFifoDedupliId;
	}

	/**
	 * SQSFIFO用の重複排除IDを設定します。
	 * 
	 * @param dedupliId SQSFIFO用の重複排除ID
	 */
	public void setSqsFifoDedupliId(final String dedupliId) {
		sqsFifoDedupliId = dedupliId;
	}

	/**
	 * MQ用のトランザクション処理指定フラグを取得します。
	 * 
	 * @return MQ用のトランザクション処理指定フラグ
	 */
	public Boolean isMqTransacted() {
		return mqTransacted;
	}
	
	/**
	 * MQ用のトランザクション処理指定フラグを設定します。
	 * 
	 * @param mqFlgTransacted MQ用のトランザクションン処理指定フラグ
	 */
	public void setMqTransacted(final Boolean mqFlgTransacted) {
		mqTransacted = mqFlgTransacted;
	}

	/**
	 * MQ用の確認応答モードを取得します。
	 * 
	 * @return MQ用の確認応答モード
	 */
	public int getMqAcknowledgeMode() {
		return mqAcknowledgeMode;
	}

	/**
	 * MQ用の確認応答モードを設定します。
	 * 
	 * @param ackMode MQ用の確認応答モード
	 */
	public void setMqAcknowledgeMode(final int ackMode) {
		mqAcknowledgeMode = ackMode;
	}
	
	/**
	 * MQ用のキュー名を取得します。
	 * 
	 * @return MQ用のキュー名
	 */
	public String getMqQueueName() {
		return mqQueueName;
	}

	/**
	 * MQ用のキュー名を設定します。
	 * 
	 * @param qName MQ用のキュー名
	 */
	public void setMqQueueName(final String qName) {
		mqQueueName = qName;
	}
}
