package monalipse.editors;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import monalipse.MonalipsePlugin;
import monalipse.server.BBSServerManager;
import monalipse.server.IBBSBoard;
import monalipse.server.ILinkedLineFragment;
import monalipse.server.INewResponseLineFragment;
import monalipse.server.IResponseEnumeration;
import monalipse.server.IResponseHeaderLine;
import monalipse.server.IThreadContentProvider;
import monalipse.utils.CancelableRunner;
import monalipse.widgets.ColoredText;
import monalipse.widgets.ColoredText.LinkTarget;
import monalipse.widgets.ColoredText.TextEvent;
import monalipse.widgets.ColoredText.ToolTipTarget;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.text.IFindReplaceTarget;
import org.eclipse.jface.text.ITextOperationTarget;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.program.Program;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.EditorPart;

public class ThreadViewerEditor extends EditorPart implements ColoredText.ToolTipProvider, ColoredText.LinkProvider, IFindReplaceTarget, IPartListener, IResourceChangeListener
{
	private static final Logger logger = MonalipsePlugin.getLogger(ThreadViewerEditor.class);

	private ThreadEditorInput input;
	private Display display;
	private ColoredText text;
	private int sequence;
	private int responseCount;
	private boolean writable;
	private boolean toolTipActive;
	private LinkSelectionProvider linkSelectionProvider;
	private boolean active = true;
	private boolean toBeUpdated;

	public ThreadViewerEditor()
	{
	}

	public void updateThread(final int scrollTo)
	{
		CancelableRunner cancelable = ((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).getCancelable();
		cancelable.run(this, new CancelableRunner.ICancelableRunnableWithProgress()
			{
				public void run(CancelableRunner.ICancelableProgressMonitor monitor)
				{
					updateThread(monitor, true, scrollTo);
				}
			});
	}
	
	private void updateThread(CancelableRunner.ICancelableProgressMonitor monitor, final boolean download, int scroll)
	{
		monitor.beginTask("getting thread...", 130);

		try
		{
			IThreadContentProvider thread = getContentProvider();
			if(thread == null)
				return;
			
			display.syncExec(new Runnable()
				{
					public void run()
					{
						setTitleImage(MonalipsePlugin.getDefault().getLabelImageOf(getContentProvider()));
						for(int i = 0; i < text.getLineCount(); i++)
							unmarkNewResponse(text.getLineAt(i));
						text.redraw();
					}
				});

			monitor.worked(10);
			IResponseEnumeration e;
			if(download)
				e = thread.updateResponses(monitor.getRunner().getSubProgressMonitor(monitor, 100), getSite().getWorkbenchWindow(), sequence, responseCount);
			else
				e = thread.getResponses(monitor.getRunner().getSubProgressMonitor(monitor, 100), sequence, responseCount);
			writable = e.isWritable();
			monitor.worked(10);
	
			if(e == null)
				return;

			try
			{
				if(!e.isPartialContent())
				{
					responseCount = 0;
					display.syncExec(new Runnable()
						{
							public void run()
							{
								text.clear();
							}
						});
				}

				monitor.worked(10);
				
				final ColoredText.TextPosition[] latestResponseLocation = new ColoredText.TextPosition[1];
				final ColoredText.TextPosition[] scrollTo = new ColoredText.TextPosition[1];
				if(download)
				{
					display.syncExec(new Runnable()
						{
							public void run()
							{
								if(responseCount != 0 && saveScrollPosition() == responseCount)
								{
									for(int i = 0; i < text.getLineCount(); i++)
									{
										ColoredText.Line line = text.getLineAt(i);
										if(line instanceof IResponseHeaderLine &&
											((IResponseHeaderLine)line).getReponseNumber() == responseCount)
										{
											latestResponseLocation[0] = new ColoredText.TextPosition(i, 0);
										}
									}
								}
							}
						});
				}
				
				if(scroll <= responseCount)
					scrollTo[0] = latestResponseLocation[0];

				final List lines = new ArrayList();
				long bulkWrite = System.currentTimeMillis();
				int writePeriod = responseCount + 10;
				
				Runnable applyLines = new Runnable()
					{
						public void run()
						{
							if(!download)
							{
								for(int i = 0; i < lines.size(); i++)
									unmarkNewResponse((ColoredText.Line)lines.get(i));
							}

							text.addLines(lines);
							lines.clear();
							
							if(scrollTo[0] != null)
							{
								Point pt = text.getPointFromIndex(scrollTo[0]);
								ScrollBar vert = text.getVerticalBar();
								if(vert.getSelection() + pt.y < vert.getMaximum() - vert.getThumb())
								{
									text.scrollTo(scrollTo[0], SWT.TOP);
									scrollTo[0] = null;
								}
							}
						}
					};

				try
				{
					while(!monitor.isCanceled() && e.hasNextResponse())
					{
						if(e.getNextResponse(lines))
						{
							if(responseCount == 0)
							{
								input.setTitle(e.getTitle());
								display.syncExec(new Runnable()
									{
										public void run()
										{
											setTitle(input.getTitle());
											((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateBookmarkDependentActions();
											ColoredText.Line line = new ColoredText.Line(0);
											line.addLineFragment(new ColoredText.LineFragment(input.getTitle(), ColoredText.Attributes.COLOR_RED, ColoredText.Attributes.FONT_NORMAL, false));
											text.addLine(line);
											text.addLine(new ColoredText.Line(0));
										}
									});
							}

							responseCount++;
							
							if(responseCount < 5 || responseCount == writePeriod || 300 < System.currentTimeMillis() - bulkWrite || scroll == responseCount)
							{
								bulkWrite = System.currentTimeMillis();
								display.syncExec(applyLines);
								
								if(scroll == responseCount)
								{
									display.syncExec(new Runnable()
										{
											public void run()
											{
												for(int i = text.getLineCount() - 1; 0 <= i; i--)
												{
													ColoredText.Line line = text.getLineAt(i);
													if(line instanceof IResponseHeaderLine)
													{
														scrollTo[0] = new ColoredText.TextPosition(i, 0);
														break;
													}
												}
											}
										});
								}
							}
							
							if(responseCount == 1)
							{
								String tooltipText = input.getTitle() + "\n" + input.getURL();
								input.setToolTipText(tooltipText);
								display.syncExec(new Runnable()
									{
										public void run()
										{
											setTitleToolTip(input.getToolTipText());
											setTitle(input.getTitle());
										}
									});
							}
						}
					}
				}
				catch (InterruptedException ex)
				{
				}
				finally
				{
					if(0 < lines.size())
						display.syncExec(applyLines);
				}

				sequence = e.getSequenceNumber();
				
				if(responseCount < scroll)
					scrollTo[0] = latestResponseLocation[0];

				if(scrollTo[0] != null)
				{
					display.syncExec(new Runnable()
						{
							public void run()
							{
								text.scrollTo(scrollTo[0], SWT.TOP);
							}
						});
				}
			}
			finally
			{
				e.close();
			}
		}
		finally
		{
			monitor.done();
		}
	}
	
	private void unmarkNewResponse(ColoredText.Line line)
	{
		for(int j = 0; j < line.getLineFragmentCount(); j++)
		{
			ColoredText.LineFragment f = line.getLineFragmentAt(j);
			if(f instanceof INewResponseLineFragment)
				((INewResponseLineFragment)f).unmark();
		}
	}
	
	public void disposeCache()
	{
		try
		{
			getContentProvider().getLogFile().delete(true, false, new NullProgressMonitor());
		}
		catch (CoreException e)
		{
		}
	}
	
	public void lockToolTip()
	{
		text.lockToolTip();
	}
	
	public ITextOperationTarget getTextOperationTarget()
	{
		return text;
	}

	public boolean scrollTo(final int responseNumber)
	{
		final boolean[] res = new boolean[1];
		display.asyncExec(new Runnable()
			{
				public void run()
				{
					for(int i = 0; i < text.getLineCount(); i++)
					{
						ColoredText.Line line = text.getLineAt(i);
						if(line instanceof IResponseHeaderLine)
						{
							if(((IResponseHeaderLine)line).getReponseNumber() == responseNumber)
							{
								ColoredText.TextPosition read = new ColoredText.TextPosition(i, 0);
								text.scrollTo(read, SWT.TOP);
								res[0] = true;
								break;
							}
						}
					}
				}
			});
		return res[0];
	}

	private int saveScrollPosition()
	{
		ColoredText.TextPosition bottomLine = text.getIndexFromPoint(new Point(0, text.getSize().y - 1), false);
		if(bottomLine.isValid())
		{
			for(int i = bottomLine.row; 0 <= i; i--)
			{
				ColoredText.Line line = text.getLineAt(i);
				if(line instanceof IResponseHeaderLine)
				{
					int read = ((IResponseHeaderLine)line).getReponseNumber();
					try
					{
						IThreadContentProvider cp = getContentProvider();
						cp.getLogFolder().setPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), cp.getID() + ".read"), String.valueOf(read));
						cp.getLogFolder().setPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), cp.getID() + ".readLine"), String.valueOf(bottomLine.row));
					}
					catch (CoreException e)
					{
					}
					return read;
				}
			}
		}
		return 0;
	}
	
	private void loadScrollPosition()
	{
		try
		{
			IThreadContentProvider cp = getContentProvider();
			ColoredText.TextPosition read = new ColoredText.TextPosition(Integer.parseInt(cp.getLogFolder().getPersistentProperty(new QualifiedName(ThreadViewerEditor.class.getName(), cp.getID() + ".readLine"))), 0);
			text.scrollTo(read, SWT.BOTTOM);
		}
		catch (CoreException e)
		{
		}
		catch (NumberFormatException e)
		{
		}
	}

	public IThreadContentProvider getContentProvider()
	{
		try
		{
			return BBSServerManager.getThreadContentProviderOf(new URL(input.getURL()));
		}
		catch (MalformedURLException e)
		{
			return null;
		}
	}

	public boolean isWritable()
	{
		return writable;
	}

	public void doSave(IProgressMonitor monitor)
	{
	}

	public void doSaveAs()
	{
	}

	public void gotoMarker(IMarker marker)
	{
	}

	public void init(IEditorSite site, IEditorInput input) throws PartInitException
	{
		setSite(site);
		setInput(input);
		
		if(input instanceof ThreadEditorInput)
			this.input = (ThreadEditorInput)input;
	}

	public boolean isDirty()
	{
		return false;
	}

	public boolean isSaveAsAllowed()
	{
		return false;
	}
	
	public static void activateEditor(IWorkbenchPartSite site, IThreadContentProvider thread)
	{
		logger.finest("begin");
		IWorkbenchPage page = site.getPage().getWorkbenchWindow().getActivePage();
		IEditorPart part = page.findEditor(new ThreadEditorInput(thread));
		if(part != null)
		{
			try
			{
				logger.finest("activate");
				page.openEditor(new ThreadEditorInput(thread), ThreadViewerEditor.class.getName(), false);
			}
			catch(PartInitException ex)
			{
				ex.printStackTrace();
			}
		}
		logger.finest("done");
	}

	public void createPartControl(Composite parent)
	{
		display = parent.getShell().getDisplay();
		
		text = new ColoredText(parent, SWT.V_SCROLL);
		text.setBackground(text.getAttributes().getColor(ColoredText.Attributes.COLOR_BACKGROUND_GRAD));
		text.setForeground(text.getAttributes().getColor(ColoredText.Attributes.COLOR_BLACK));
		text.setToolTipProvider(this);
		text.setLinkProvider(this);
		text.addTextTrackListener(new ColoredText.TextTrackListener()
			{
				public void textEnter(TextEvent e)
				{
					if(e.fragment instanceof ColoredText.ToolTipTarget)
						getContentProvider().prefetchToolTip((ColoredText.ToolTipTarget)e.fragment);
				}
	
				public void textExit(TextEvent e)
				{
				}
	
				public void textHover(TextEvent e)
				{
				}
				
				public void textSelectionEnter(TextEvent e)
				{
				}
				
				public void textSelectionExit(TextEvent e)
				{
				}
				
				public void textSelectionHover(TextEvent e)
				{
				}

			});
		text.addSelectionChangedListener(new ISelectionChangedListener()
			{
				public void selectionChanged(SelectionChangedEvent event)
				{
					((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateSelectionDependentActions();
				}
			});
		text.addKeyListener(new KeyListener()
			{
				public void keyPressed(KeyEvent e)
				{
					if(e.keyCode == SWT.ESC)
						text.setSelection(ColoredText.TextPosition.INVALID, ColoredText.TextPosition.INVALID, true);
					else if(e.keyCode == SWT.F5)
						updateThread(-1);
				}
	
				public void keyReleased(KeyEvent e)
				{
				}
			});
		
		hookContextMenu();
		
		ResourcesPlugin.getWorkspace().addResourceChangeListener(this);
		getSite().getPage().addPartListener(this);
		linkSelectionProvider = new LinkSelectionProvider();
		getSite().setSelectionProvider(linkSelectionProvider);

		CancelableRunner cancelable = ((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).getCancelable();
		cancelable.setDisplay(display);
		cancelable.runDirect(new CancelableRunner.ICancelableRunnableWithProgress()
			{
				public void run(CancelableRunner.ICancelableProgressMonitor monitor)
				{
					updateThread(monitor, false, -1);
				}
			});

		Thread thread = new Thread(new Runnable()
			{
				public void run()
				{
					display.asyncExec(new Runnable()
						{
							public void run()
							{
								loadScrollPosition();
							}
						});
				}
			});
		thread.start();
		try
		{
			thread.join();
		}
		catch (InterruptedException e)
		{
		}
	}
	
	public void resourceChanged(IResourceChangeEvent event)
	{
		IThreadContentProvider thread = getContentProvider();
		if(thread != null)
		{
			boolean threadChanged = thread.threadChanged(event);
			boolean boardChanged = thread.getBoard().threadListChanged(event);
			if(threadChanged || boardChanged)
			{
				setTitleImage(MonalipsePlugin.getDefault().getLabelImageOf(thread));
//				if(!threadChanged && active)
//					updateThread(-1);
			}
		}
	}
	
	public void setToBeUpdated(boolean updated)
	{
		toBeUpdated = updated;
	}

	private void hookContextMenu()
	{
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener()
			{
				public void menuAboutToShow(IMenuManager manager)
				{
					ThreadViewerEditorActionBarContributor cont = (ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor();
					cont.contributeToContextMenu(manager);
				}
			});
		Menu menu = menuMgr.createContextMenu(text);
		text.setMenu(menu);
	}
	
	
	public void linkClicked(ColoredText text, LinkTarget target)
	{
		if(target instanceof ILinkedLineFragment)
		{
			String href = ((ILinkedLineFragment)target).getURL();
			try
			{
				IThreadContentProvider thread = BBSServerManager.getThreadContentProviderOf(new URL(href));
				if(thread != null)
				{
					IWorkbenchPage page = getSite().getWorkbenchWindow().getActivePage();
					try
					{
						boolean update = page.findEditor(new ThreadEditorInput(thread)) == null;
						IEditorPart part = page.openEditor(new ThreadEditorInput(thread), ThreadViewerEditor.class.getName());
						if(part instanceof ThreadViewerEditor)
						{
							boolean scrolled = true;
							int pos = ((ILinkedLineFragment)target).getResponseFragment();
							if(0 < pos && ((ThreadViewerEditor)part).scrollTo(pos))
								pos = -1;
							if(update)
								((ThreadViewerEditor)part).updateThread(scrolled ? -1 : pos);
						}
					}
					catch(PartInitException ex)
					{
						ex.printStackTrace();
					}
					return;
				}

				IBBSBoard board = BBSServerManager.getBoardOf(new URL(href));
				if(board != null)
				{
					linkSelectionProvider.fireSelectionChangedEvent(board);
					return;
				}

				if(href.startsWith("http"))
				{
					Program.launch(href);
					return;
				}
			}
			catch (MalformedURLException ex)
			{
				ex.printStackTrace();
			}
		}
	}
	
	public Point fillToolTip(Composite parent, ColoredText text, int maxWidth, ToolTipTarget target)
	{
		return getContentProvider().fillToolTip(parent, text, maxWidth, target, getTitle());
	}
	
	public Point fillToolTip(Composite parent, ColoredText text, int maxWidth, String selection)
	{
		return getContentProvider().fillToolTip(parent, text, maxWidth, selection, getTitle());
	}
	
	public void toolTipActivated(ColoredText text)
	{
		toolTipActive = true;
		((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateToolTipDependentActions();
	}
	
	public void toolTipDeactivated(ColoredText text)
	{
		toolTipActive = false;
		((ThreadViewerEditorActionBarContributor)getEditorSite().getActionBarContributor()).updateToolTipDependentActions();
	}
	
	public boolean isToolTipActive()
	{
		return toolTipActive;
	}

	public void setFocus()
	{
		text.setFocus();
	}
	
	public void dispose()
	{
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(this);
		getSite().getPage().removePartListener(this);
		if(text != null)
			text.dispose();
		text = null;
		super.dispose();
	}
	
	public boolean canPerformFind()
	{
		return true;
	}

	public int findAndSelect(int offset, String findString, boolean searchForward, boolean caseSensitive, boolean wholeWord)
	{
		ColoredText.TextPosition off = new ColoredText.TextPosition(text.getLineCount() - 1, text.getLineAt(text.getLineCount() - 1).getTextLength());
		for(int i = 0; i < text.getLineCount(); i++)
		{
			int len = text.getLineAt(i).getTextLength();
			if(offset < len)
			{
				off = new ColoredText.TextPosition(i, Math.max(offset, 0));
				break;
			}
			offset -= len;
		}
		return text.textPositionToIntPosition(text.findAndSelect(off, findString, searchForward, caseSensitive, wholeWord));
	}

	public Point getSelection()
	{
		ColoredText.TextSelection sel = text.getTextSelection();
		int start = text.textPositionToIntPosition(sel.from);
		int end = text.textPositionToIntPosition(sel.to);
		return new Point(start, end - start);
	}
	
	public String getSelectionText()
	{
		ColoredText.TextSelection sel = text.getTextSelection();
		return text.getText(sel.from, sel.to, false);
	}

	public boolean isEditable()
	{
		return false;
	}

	public void replaceSelection(String text)
	{
	}
	
	public Object getAdapter(Class adapter)
	{
		if(adapter == IFindReplaceTarget.class)
			return this;
		else
			return null;
	}
	
	public void partActivated(IWorkbenchPart part)
	{
		if(part == this)
		{
			active = true;
			
//			IThreadContentProvider thread = getContentProvider();
//			if(thread != null)
//			{
//				if(toBeUpdated  || thread.hasNewResponses())
//				{
//					updateThread(-1);
//					toBeUpdated = false;
//				}
//			}
		}
	}

	public void partBroughtToTop(IWorkbenchPart part)
	{
	}

	public void partClosed(IWorkbenchPart part)
	{
	}

	public void partDeactivated(IWorkbenchPart part)
	{
		if(text != null)
			text.disposeToolTip();

		if(part == this && active)
		{
			saveScrollPosition();
			active = false;
		}
	}

	public void partOpened(IWorkbenchPart part)
	{
	}

	private class LinkSelectionProvider implements ISelectionProvider
	{
		private List selectionChangedListeners = new ArrayList();

		public void addSelectionChangedListener(ISelectionChangedListener listener)
		{
			selectionChangedListeners.add(listener);
		}
		
		public void removeSelectionChangedListener(ISelectionChangedListener listener)
		{
			selectionChangedListeners.remove(listener);
		}
	
		public void setSelection(ISelection selection)
		{
		}
		
		public ISelection getSelection()
		{
			return null;
		}
	
		private void fireSelectionChangedEvent(IBBSBoard board)
		{
			SelectionChangedEvent e = new SelectionChangedEvent(this, new StructuredSelection(board));
			for(int i = 0; i < selectionChangedListeners.size(); i++)
			{
				ISelectionChangedListener l = (ISelectionChangedListener)selectionChangedListeners.get(i);
				l.selectionChanged(e);
			}
		}
	}

}
