package com.snail.msglet;

import org.apache.commons.beanutils.BeanUtils;

import java.util.Map;


/**
 * Msgletフレームワーク.
 * <pre>
 * 全てのMsgletは、このクラスを継承します。
 * Msgletは、init(Map initParam), service(), destroy() を実装します。
 *
 * Msglet初期化時には、init(Map initParam) が呼ばれます。
 * initParam には、msg.xml に記載された初期化情報が格納されています
 *
 *   &lt;init-param&gt;
 *     &lt;param-name&gt;width&lt;/param-name&gt;
 *     &lt;param-value&gt;3&lt;/param-value&gt;
 *   &lt;/init-param&gt;
 *
 * と定義されていた場合、initParamには、
 * キー"width" に対して、値"3"が格納されています。
 *
 * Msglet終了時には、destroy() が呼ばれます。
 *
 * IPMsgクライアントからのメッセージを受信した場合には、service(req,res)
 * が呼ばれます。
 * reqには、受信情報が含まれます。
 * resには、返信情報を格納します。特に変更しなければ、resには、受信者に対して
 * 返信を行うように設定されています。(res.setMsg()で、返信メッセージを格納
 * するだけでよい)
 * </pre>
 * @author kagyuu
 */
public abstract class Msglet {
  /**
   * このMsgletの初期化情報 .
   */
  private Map<String, String> initParamMap = null;

  /**
   * msg.xml に記入された初期化情報を設定します .
   *
   * @param param このMsgletの初期化情報
   */
  final void setInitParamMap(final Map<String, String> param) {
    this.initParamMap = param;
  }

  /**
   * Msglet開始時(Msgletコンテナ開始時)に呼ばれます .
   *
   * @param initParam msg.xml に記入された初期化情報
   */
  public abstract void init(Map initParam);

  /**
   * IPMsgのメッセージ受信時に呼ばれます .
   * <pre>
   * 各Msgletで実装します。
   * </pre>
   * @param req 処理要求
   * @param res 処理結果
   */
  public abstract void service(Request req, Response res);

  /**
   * Msglet終了時(Msgletコンテナ終了時)に呼ばれます .
   */
  public abstract void destroy();

  /**
   * Msgletが初期化されるとき(Msgletコンテナ起動時)に呼ばれます .
   * <pre>
   * init(Map initParam) を呼び出します。
   * init(Map initParam) は、各Msgletで実装します。
   * </pre>
   */
  final void init() {
    init(initParamMap);
  }

  /**
   * Msgletを文字列化します .
   * @return Msgletの文字列表現
   */
  @Override
  public final String toString() {
    try {
      return BeanUtils.describe(this).toString();
    } catch (Exception ex) {
      return ex.toString();
    }
  }

  /**
   * あるMsgletから別のMsgletに処理を委譲するときに使われます .
   * @param req 処理要求
   * @param res 処理結果
   */
  public final void forward(final Request req, final Response res) {
    service(req, res);
  }
}
