#include <ruby.h>
#include <windows.h>

#define VERSION "0.2.0"
#define IDENTIFIER "WinTrayIcon"
#define TRAY_MESSAGE WM_USER
#define ID_TRAY  100

static char szClassName[] = "wintrayiconWindow";

typedef struct wintrayiconWindow {
	HINSTANCE hInst;
	HWND hWnd;
	HMENU hMenu;
	NOTIFYICONDATA ni;

	VALUE rvMenuItem;
	unsigned long ulMenuCount;
} wintrayiconWindow;

static void wintrayicon_Mark( wintrayiconWindow* tray );
static LRESULT CALLBACK wintrayicon_Proc( HWND hWnd, UINT msg, WPARAM wp, LPARAM lp );
static VALUE wintrayicon_Alloc( VALUE self );
static VALUE wintrayicon_Initialize( VALUE self );
static VALUE wintrayicon_Live( VALUE self );
static VALUE wintrayicon_Update( VALUE self );
static VALUE wintrayicon_Close( VALUE self );
static VALUE wintrayicon_AddMenu( int argc, VALUE *argv, VALUE self );
static VALUE wintrayicon_AddSeparater( VALUE self );
static VALUE wintrayicon_Enable( VALUE self, VALUE index );
static VALUE wintrayicon_Disable( VALUE self, VALUE index );
static VALUE wintrayicon_GetState( VALUE self, VALUE index );
static VALUE wintrayicon_SetState( VALUE self, VALUE index, VALUE flag );
static VALUE wintrayicon_LoadIcon( VALUE self, VALUE path, VALUE w, VALUE h );
static VALUE wintrayicon_GetIcon( VALUE self, VALUE id, VALUE w, VALUE h );
static VALUE wintrayicon_WindowHandle( VALUE self );
static VALUE wintrayicon_MenuHandle( VALUE self );

void Init_wintrayicon( void );

static void
wintrayicon_Mark( wintrayiconWindow* tray ) {
	rb_gc_mark( tray->rvMenuItem );
}

static LRESULT CALLBACK
wintrayicon_Proc( HWND hWnd, UINT msg, WPARAM wp, LPARAM lp ) {
	static VALUE self = Qnil;

	wintrayiconWindow *tray;
	POINT pt;
	VALUE proc;

	if( self != Qnil ) Data_Get_Struct( self, wintrayiconWindow, tray );

	switch( msg ) {
	case WM_CREATE:
		self = (VALUE)((LPCREATESTRUCT)lp)->lpCreateParams;
		break;

	case TRAY_MESSAGE:
		if( wp != ID_TRAY ) break;
		switch(lp) {
		case WM_RBUTTONDOWN:
		case WM_LBUTTONDOWN:
			GetCursorPos( &pt );
			SetForegroundWindow( tray->hWnd );
			TrackPopupMenu( tray->hMenu,
											TPM_BOTTOMALIGN,
											pt.x,
											pt.y,
											0,
											tray->hWnd,
											NULL );
			break;
		}
		break;

	case WM_COMMAND:
		proc = rb_ary_entry( tray->rvMenuItem, LOWORD(wp) );
		if( TYPE(proc) != T_NIL ) rb_funcall( proc, rb_intern("call"), 1, self );
		break;

	case WM_DESTROY:
		PostQuitMessage(0);
		break;

	default:
		return DefWindowProc( hWnd, msg, wp, lp );
	}
	return 0;
}

static VALUE
wintrayicon_Alloc( VALUE self ) {
	wintrayiconWindow *tray = ALLOC(wintrayiconWindow);
	return Data_Wrap_Struct( self, wintrayicon_Mark, -1, tray );
}

static VALUE
wintrayicon_Initialize( VALUE self ) {
	WNDCLASSEX wc;

	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );

	tray->hInst       = GetModuleHandle( NULL );
	tray->hWnd        = NULL;
	tray->hMenu       = NULL;
	tray->rvMenuItem  = rb_ary_new();
	tray->ulMenuCount = 0;

	wc.cbSize = sizeof(WNDCLASSEX);
	wc.style         = CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc   = (WNDPROC)wintrayicon_Proc;
	wc.cbClsExtra    = 0;
	wc.cbWndExtra    = 0;
	wc.hInstance     = tray->hInst;
	wc.hIcon         = NULL;
	wc.hCursor       = NULL;
	wc.hbrBackground = (HBRUSH)(COLOR_WINDOW+1);
	wc.lpszMenuName  = "MenuName";
	wc.lpszClassName = szClassName;
	wc.hIconSm       = NULL;
	if( !RegisterClassEx(&wc) )
		rb_raise( rb_eRuntimeError, "RegisterClass failed. %d", GetLastError() );

	tray->hWnd = CreateWindow( szClassName,
														 szClassName,
														 WS_OVERLAPPEDWINDOW,
														 1, 1, 100, 100,
														 NULL,
														 NULL,
														 0,
														 (LPVOID)self );

	if( !tray->hWnd )
		rb_raise( rb_eRuntimeError, "CreateWindow failed." );

	tray->hMenu = CreatePopupMenu();

	tray->ni.cbSize = sizeof(NOTIFYICONDATA);
	tray->ni.hWnd   = tray->hWnd;
	tray->ni.uID    = ID_TRAY;
	tray->ni.uFlags = NIF_ICON | NIF_MESSAGE | NIF_TIP;
	tray->ni.uCallbackMessage = TRAY_MESSAGE;
	tray->ni.hIcon = NULL;
	strncpy(tray->ni.szTip,"wintrayicon",8);
	Shell_NotifyIcon( NIM_ADD, &tray->ni );
}

static VALUE
wintrayicon_Live( VALUE self ) {
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );

	return tray->hWnd ? Qtrue : Qfalse;
}

static VALUE
wintrayicon_Update( VALUE self ) {
	MSG msg;
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );

	if( PeekMessage( &msg, tray->hWnd, 0, 0, PM_REMOVE ) ) {
		if( msg.message == WM_QUIT ) wintrayicon_Close(self);
		else {
			TranslateMessage( &msg );
			DispatchMessage( &msg );
		}
	}
	return Qnil;
}

static VALUE
wintrayicon_Close( VALUE self ) {
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );

	if( tray->hWnd ) {
		Shell_NotifyIcon( NIM_DELETE, &tray->ni );
		DestroyWindow( tray->hWnd );
		tray->hWnd = NULL;
	}
	return Qnil;
}

static VALUE
wintrayicon_AddMenu( int argc, VALUE *argv, VALUE self ) {
	MENUITEMINFO mii;
	wintrayiconWindow *tray;
	VALUE text,block;
	Data_Get_Struct( self, wintrayiconWindow, tray );

	rb_scan_args( argc, argv, "1&", &text, &block );
	rb_ary_push( tray->rvMenuItem, block );

	mii.cbSize     = sizeof( MENUITEMINFO );
	mii.fMask      = MIIM_TYPE | MIIM_SUBMENU | MIIM_ID;
	mii.fType      = MFT_STRING;
	mii.dwTypeData = TEXT(RSTRING(text)->ptr);
	mii.hSubMenu   = NULL;
	mii.wID        = tray->ulMenuCount;
	InsertMenuItem( tray->hMenu, tray->ulMenuCount++, TRUE, &mii );
	return Qnil;
}

static VALUE
wintrayicon_AddSeparater( VALUE self ) {
	MENUITEMINFO mii;
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );

	rb_ary_push( tray->rvMenuItem, Qnil );

	mii.cbSize     = sizeof( MENUITEMINFO );
	mii.fMask      = MIIM_TYPE | MIIM_SUBMENU;
	mii.fType      = MFT_SEPARATOR;
	mii.hSubMenu   = NULL;
	InsertMenuItem( tray->hMenu, tray->ulMenuCount++, TRUE, &mii );
	return Qnil;
}

static VALUE
wintrayicon_Enable( VALUE self, VALUE index ) {
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );
	EnableMenuItem( tray->hMenu, NUM2INT(index), MF_ENABLED | MF_BYCOMMAND );
	return Qnil;
}

static VALUE
wintrayicon_Disable( VALUE self, VALUE index ) {
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );
	EnableMenuItem( tray->hMenu, NUM2INT(index), MF_GRAYED | MF_BYCOMMAND );
	return Qnil;
}

static VALUE
wintrayicon_GetState( VALUE self, VALUE index ) {
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );
	if( GetMenuState( tray->hMenu, NUM2INT(index), MF_BYCOMMAND ) & MF_CHECKED )
		return Qtrue;
	else
		return Qfalse;
}

static VALUE
wintrayicon_SetState( VALUE self, VALUE index, VALUE flag ) {
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );
	if( TYPE(flag) == T_TRUE ) {
		CheckMenuItem( tray->hMenu, NUM2INT(index), MF_CHECKED | MF_BYCOMMAND );
		return Qtrue;
	} else {
		CheckMenuItem( tray->hMenu, NUM2INT(index), MF_UNCHECKED | MF_BYCOMMAND );
		return Qfalse;
	}
}

static VALUE
wintrayicon_LoadIcon( VALUE self, VALUE path, VALUE w, VALUE h ) {
	HANDLE old_icon;
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );

	if( !tray->hWnd ) return Qtrue;

	old_icon = tray->ni.hIcon;
	tray->ni.hIcon = LoadImage(
		tray->hInst,
		RSTRING(path)->ptr,
		IMAGE_ICON,
		NUM2INT(w),
		NUM2INT(h),
		LR_DEFAULTCOLOR | LR_DEFAULTSIZE | LR_LOADFROMFILE);

	if( tray->ni.hIcon ) {
		Shell_NotifyIcon( NIM_MODIFY, &tray->ni );
		if( old_icon ) DestroyIcon( old_icon );
		return Qtrue;
	}
	return Qfalse;
}

static VALUE
wintrayicon_GetIcon( VALUE self, VALUE id, VALUE w, VALUE h ) {
	HANDLE old_icon;
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );

	if( !tray->hWnd ) return Qtrue;

	old_icon = tray->ni.hIcon;

	if( TYPE(id) == T_FIXNUM ) {
		tray->ni.hIcon = LoadImage(
			tray->hInst,
			MAKEINTRESOURCE(NUM2INT(id)),
			IMAGE_ICON,
			NUM2INT(w),
			NUM2INT(h),
			LR_DEFAULTCOLOR | LR_DEFAULTSIZE );
	} else {
		tray->ni.hIcon = LoadImage(
			tray->hInst,
			RSTRING(rb_any_to_s(id))->ptr,
			IMAGE_ICON,
			NUM2INT(w),
			NUM2INT(h),
			LR_DEFAULTCOLOR | LR_DEFAULTSIZE );
	}

	if( tray->ni.hIcon ) {
		Shell_NotifyIcon( NIM_MODIFY, &tray->ni );
		if( old_icon ) DestroyIcon( old_icon );
		return Qtrue;
	}
	return Qfalse;
}

static VALUE
wintrayicon_GetTip( VALUE self ) {
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );

	return rb_str_new2(tray->ni.szTip);
}

static VALUE
wintrayicon_SetTip( VALUE self, VALUE str ) {
	int i=0;
	char *tmp;
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );

	tmp = RSTRING(str)->ptr;
	while( i<63 && tmp ) {
		tray->ni.szTip[i++] = *tmp++;
	}
	tray->ni.szTip[i] = 0;
	Shell_NotifyIcon( NIM_MODIFY, &tray->ni );
	return Qnil;
}

static VALUE
wintrayicon_WindowHandle( VALUE self ) {
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );
	return LONG2NUM((unsigned long)tray->hWnd);
}

static VALUE
wintrayicon_MenuHandle( VALUE self ) {
	wintrayiconWindow *tray;
	Data_Get_Struct( self, wintrayiconWindow, tray );
	return LONG2NUM((unsigned long)tray->hMenu);
}

void Init_wintrayicon( void ) {
	VALUE rb_cwintrayicon;
	rb_cwintrayicon = rb_define_class("WinTrayIcon",rb_cObject );
	rb_define_alloc_func( rb_cwintrayicon, wintrayicon_Alloc );
	rb_define_private_method( rb_cwintrayicon, "initialize", wintrayicon_Initialize, 0 );

	rb_define_const( rb_cwintrayicon, "VERSION", rb_str_new2(VERSION) );
	rb_define_const( rb_cwintrayicon, "IDENTIFIER", rb_str_new2(IDENTIFIER) );

	rb_define_method( rb_cwintrayicon, "live?",  wintrayicon_Live,   0 );
	rb_define_method( rb_cwintrayicon, "update", wintrayicon_Update, 0 );
	rb_define_method( rb_cwintrayicon, "close",  wintrayicon_Close,  0 );

	rb_define_method( rb_cwintrayicon, "add_menu",      wintrayicon_AddMenu,      -1 );
	rb_define_method( rb_cwintrayicon, "add_separater", wintrayicon_AddSeparater,  0 );
	rb_define_method( rb_cwintrayicon, "enable",        wintrayicon_Enable,        1 );
	rb_define_method( rb_cwintrayicon, "disable",       wintrayicon_Disable,       1 );
	rb_define_method( rb_cwintrayicon, "get_state",     wintrayicon_GetState,      1 );
	rb_define_method( rb_cwintrayicon, "set_state",     wintrayicon_SetState,      2 );
	rb_define_alias( rb_cwintrayicon, "[]",  "get_state" );
	rb_define_alias( rb_cwintrayicon, "[]=", "set_state" );
	
	rb_define_method( rb_cwintrayicon, "load_icon", wintrayicon_LoadIcon, 3 );
	rb_define_method( rb_cwintrayicon, "get_icon",  wintrayicon_GetIcon,  3 );

	rb_define_method( rb_cwintrayicon, "tip",  wintrayicon_GetTip, 0 );
	rb_define_method( rb_cwintrayicon, "tip=", wintrayicon_SetTip, 1 );

	rb_define_method( rb_cwintrayicon, "window_handle", wintrayicon_WindowHandle, 0 );
	rb_define_method( rb_cwintrayicon, "menu_handle",   wintrayicon_MenuHandle,   0 );
}
