/**
 * Copyright  2007 matsu@zuena.org.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, 
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.zuena.guiceex.jpa.transaction;

import java.util.HashMap;
import java.util.Map;

import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.zuena.guiceex.transaction.NoTransactionException;
import org.zuena.guiceex.transaction.TransactionActiveException;
import org.zuena.guiceex.transaction.TransactionAttributeType;
import org.zuena.guiceex.transaction.annotation.Transactional;

import static org.zuena.guiceex.jpa.EntityManagerManager.*;
import static org.zuena.guiceex.transaction.TransactionAttributeType.*;

/**
 * {@link TransactionAttributeType}ɉgUNVs܂B
 * ̃NX̃CX^X́A{@link TransactionAttributeType}ɉ
 * č쐬A{@link TransactionInterceptor}Ńvg^CvƂėpA
 * sx쐬ꂽs܂B<br/>
 * guiceex-0.9.1ł́AResource-LocalgUNV̂݃T|[g܂B<br/>
 * ȉ{@link TransactionAttributeType}̐U镑܂B<br/>
 * <blockquote>
 *<table border="1">
 *<tr>
 *<th>TransactionAttributeType</th>
 *<th>EntityManager쐬ς݂<br/>gUNVJnς</th>
 *    <th>EntityManager쐬A<br/>gUNVJn</th>
 *</tr>
 *<tr>
 *<td >NOT_SUPPORTED 
 *</td>
 *<td >EntityManagerVK쐬B<br/>gUNV͊JnȂB</td>
 *    <td>EntityManager쐬̏ꍇ͍쐬B<br/>gUNV͊JnȂB</td>
 *</tr>
 *<tr>
 *<td >SUPPORTS 
 *</td>
 *<td >̃gUNVgpB </td>
 *    <td>EntityManager쐬̏ꍇ͍쐬B<br/>gUNV͊JnȂB </td>
 *</tr>
 *<tr>
 *<td >REQUIRED</td>
 *<td >̃gUNVgpB</td>
 *    <td>EntityManager쐬̏ꍇ͍쐬B<br/>gUNVJnB </td>
 *</tr>
 *<tr>
 *    <td >REQUIRES_NEW 
 *</td>
 *<td >EntityManagerVK쐬B<br/>gUNVJnB </td>
 *    <td>EntityManagerVK쐬B<br/>gUNVJnB</td>
 *</tr>
 *<tr>
 *<td >MANDATORY 
 *</td>
 *<td >̃gUNVgpB</td>
 *    <td>NoTransactionExceptionX[B </td>
 *</tr>
 *<tr>
 *<td >NEVER 
 *</td>
 *<td >{@link TransactionActiveException}X[B</td>
 *<td>EntityManagerꍇ͐VK쐬B<br/>gUNV͊JnȂB</td>
 *  </tr>
 *</table>
 *</blockquote>
 * @author matsu@zuena.org.
 * @since 0.9.1
 */
public abstract class TransactionPropagator implements Cloneable{

	protected static final Log log = LogFactory.getLog(TransactionPropagator.class);
	protected static final Map<TransactionAttributeType, TransactionPropagator> propagators 
	= new HashMap<TransactionAttributeType, TransactionPropagator>();

	protected Class<? extends Throwable>[] rollbackFor = null;
	protected TransactionAttributeType type = null;
	protected PersistenceContext pc;

	static{
		propagators.put(NOT_SUPPORTED, new TransactionPropagator(NOT_SUPPORTED){
			EntityManager oldEm = null;
			boolean entityManagerCreated = false;
			@Override
			protected void prepare() {
				oldEm = getCurrentEntityManager(pc.unitName());

				// EntityManagerA쐬ς݁gUNVJnς݂̏ꍇ
				// VKEntityManager쐬B
				// gUNV͊JnȂB
				if (oldEm == null 
						||(oldEm != null && oldEm.getTransaction().isActive())){	
					EntityManager em = createNewEntityManager(pc);
					setCurrentEntityManager(pc.unitName(), em);
					entityManagerCreated = true;
				}
			}
			@Override
			protected void commitProcess() {
			}
			@Override
			protected void rollbackProcess() {
			}
			@Override
			protected void closeProcess() {
				if (entityManagerCreated){
					closeCurrentEntityManager(pc.unitName());
					setCurrentEntityManager(pc.unitName(), oldEm);
				}
			}
		});


		propagators.put(SUPPORTS, new TransactionPropagator(SUPPORTS){
			boolean entityManagerCreated = false;
			@Override
			protected void prepare() {
				EntityManager em = getCurrentEntityManager(pc.unitName());

				// EntityManager쐬̏ꍇ͐VK쐬B
				// gUNV͊JnȂB
				if (em == null){
					em = createNewEntityManager(pc);
					setCurrentEntityManager(pc.unitName(), em);
					entityManagerCreated = true;
				}

			}
			@Override
			protected void commitProcess() {
			}
			@Override
			protected void rollbackProcess() {
			}
			@Override
			protected void closeProcess() {
				if (entityManagerCreated){
					closeCurrentEntityManager(pc.unitName());
					setCurrentEntityManager(pc.unitName(), null);
				}
			}
		});

		propagators.put(REQUIRED, new TransactionPropagator(REQUIRED){

			private boolean entityManagerCreated = false;
			private boolean transactionActivated = false;

			@Override
			protected void prepare() {
				EntityManager em = getCurrentEntityManager(pc.unitName());
				if (em == null){
					em = createNewEntityManager(pc);
					setCurrentEntityManager(pc.unitName(), em);
					entityManagerCreated = true;
				}
				if (!em.getTransaction().isActive()){
					beginTransaction(pc.unitName());
					transactionActivated = true;
				}
			}
			@Override
			protected void commitProcess() {
				if (!entityManagerCreated && !transactionActivated){
					return;
				}
				if (log.isDebugEnabled()){
					log.debug(REQUIRED + ":#commitProcess()");
				}
				commitCurrentEm(pc.unitName());
			}
			@Override
			protected void rollbackProcess() {
				if (!entityManagerCreated && !transactionActivated){
					return;
				}
				if (log.isDebugEnabled()){
					log.debug(REQUIRED + ":#rollbackProcess()");
				}
				rollbackCurrentEntityManager(pc.unitName());
			}
			@Override
			protected void closeProcess() {
				if (entityManagerCreated){
					closeCurrentEntityManager(pc.unitName());
				}
			}
		});

		propagators.put(REQUIRES_NEW, new TransactionPropagator(REQUIRES_NEW){
			EntityManager oldEm = null;
			@Override
			protected void prepare() {
				oldEm =  getCurrentEntityManager(pc.unitName());
				EntityManager newEm = createNewEntityManager(pc);
				setCurrentEntityManager(pc.unitName(), newEm);
				beginTransaction(pc.unitName());
			}
			@Override
			protected void commitProcess() {
				commitCurrentEm(pc.unitName());
			}
			@Override
			protected void rollbackProcess() {
				rollbackCurrentEntityManager(pc.unitName());
			}
			@Override
			protected void closeProcess() {
				closeCurrentEntityManager(pc.unitName());
				setCurrentEntityManager(pc.unitName(), oldEm);
			}
		});

		propagators.put(MANDATORY, new TransactionPropagator(MANDATORY){
			@Override
			protected void prepare() {
				EntityManager em = getCurrentEntityManager(pc.unitName());
				if (em == null || !em.getTransaction().isActive()){
					throw new NoTransactionException();
				}
			}
			@Override
			protected void commitProcess() {
			}
			@Override
			protected void rollbackProcess() {
			}
			@Override
			protected void closeProcess() {
			}
		});

		propagators.put(NEVER, new TransactionPropagator(NEVER){
			boolean entityManagerCreated = false;
			@Override
			protected void prepare() {
				EntityManager em = getCurrentEntityManager(pc.unitName());
				if (em != null && em.getTransaction().isActive()){
					throw new TransactionActiveException();
				}

				if (em == null){
					em = createNewEntityManager(pc);
					setCurrentEntityManager(pc.unitName(), em);
					entityManagerCreated = true;
				}
			}
			@Override
			protected void commitProcess() {
			}
			@Override
			protected void rollbackProcess() {
			}
			@Override
			protected void closeProcess() {
				if (entityManagerCreated){
					closeCurrentEntityManager(pc.unitName());
				}
			}
		});
	}


	public TransactionPropagator(TransactionAttributeType type) {
		this.type = type;
	}

	public void setPersistenceContext(PersistenceContext pc) {
		this.pc = pc;
	}
	/**
	 * {@link TransactionAttributeType}ɉ{@link TransactionPropagator}CX^X擾܂B
	 */
	public static TransactionPropagator getTransactionPropagator(TransactionAttributeType type){
		return propagators.get(type);
	}

	/**
	 * gUNVsA{@link Process#execute()}s܂B
	 */
	public Object propagate(Process process) throws Throwable{
		try{
			if (log.isDebugEnabled()){
				log.debug(type + ":#prepare()");
			}
			prepare();
			Object ret = process.execute(getCurrentEntityManager(pc.unitName()));
			if (log.isDebugEnabled()){
				log.debug(type + ":#commitProcess()");
			}
			EntityManager em = getCurrentEntityManager(pc.unitName());
			assert em != null;
			if (em.getTransaction().isActive() && em.getTransaction().getRollbackOnly()){
				rollbackProcess();
			}else{
				commitProcess();
			}
			return ret;
		}catch(Throwable t){
			if (log.isDebugEnabled()){
				log.debug(type + ":#exceptionProcess()",t);
			}
			exceptionProcess(t);
			throw t;
		}finally{
			if (log.isDebugEnabled()){
				log.debug(type + ":#closeTransaction()");
			}
			closeProcess();
		}
	}

	/**
	 * gUNV[obNO^CvZbg܂B<br/>
	 * O^Cv{@code @}{@link Transactional#rollbackFor()}Ŏw肵O^CvłB
	 */
	public void setRollbackFor(Class<? extends Throwable>[] rollbackFor) {
		this.rollbackFor = rollbackFor;
	}

	/**
	 * vg^CvCX^X蕡쐬郁\bhłB
	 */
	public TransactionPropagator clone(){
		try {
			return (TransactionPropagator)super.clone();
		} catch (CloneNotSupportedException e) {
			return null;
		}
	}
	/**
	 * ÕR~bg/[obNs܂B
	 */
	protected void exceptionProcess(Throwable t) throws Throwable{

		for (Class<? extends Throwable> throwable : rollbackFor) {
			if (throwable.isAssignableFrom(t.getClass())){
				rollbackProcess();
				return;
			}
		}

		commitProcess();
	}

	/**
	 * {@link Process#execute(EntityManager)}ĂяoÓAgUNVJnȂǂ̎Os܂B
	 */
	protected abstract void prepare();
	
	/**
	 * {@link Process#execute(EntityManager)}ĂяoɁAgUNVR~bgۂɌĂ΂܂
	 */
	protected abstract void commitProcess();
	
	/**
	 * {@link Process#execute(EntityManager)}ĂяoɁAgUNV[obNۂɌĂ΂܂
	 */
	protected abstract void rollbackProcess();
	
	/**
	 * {@link EntityManager#close()}̌ĂяoȂǂs܂B
	 */
	protected abstract void closeProcess();

	/**
	 * gUNVŎs鏈LqNXɎC^[tFCXłB<br/>
	 * ʏA{@link TransactionInterceptor}ɂăC^[ZvgĂ郁\bȟĂяos܂B
	 * @author matsu@zuena.org.
	 */
	public interface Process{
		Object execute(EntityManager manager) throws Throwable;
	}
}