require 'umigame/arrayoptparser'
require 'test/unit/assertions'
require 'ftools'
require 'umigame/rubyunit'
require 'umigame/pyunit'
require 'umigame/junit'
require 'umigame/cppunitx'
require 'umigame/cppunit'

module Umigame
  class Core
    include Test::Unit::Assertions
    
    RUBY = 0
    PYTHON = 1
    CPP = 2
    CPPX = 3
    C = 4
    PERL = 5
    JAVA = 6
    
    $default = RUBY
    
    attr_reader :parser
    
    def initialize
      @parser = Umigame::ArrayOptParser.new
      @parser.set_option(
      ['--add', 'a', ArrayOptParser::REQUIRED_ARGUMENT],
      ['--cout', ArrayOptParser::NO_ARGUMENT],
      ['--force', 'f', ArrayOptParser::NO_ARGUMENT],
      ['--help', 'h', ArrayOptParser::NO_ARGUMENT],
      ['--lang', 'l', ArrayOptParser::REQUIRED_ARGUMENT],
      ['--main', 'm', ArrayOptParser::REQUIRED_ARGUMENT],
      ['--reference', 'r', ArrayOptParser::REQUIRED_ARGUMENT],
      ['--skeleton', 's', ArrayOptParser::REQUIRED_ARGUMENT],
      ['--version', 'v', ArrayOptParser::NO_ARGUMENT]
      )
      @lang_map = {"ruby"=>RUBY, "cpp"=>CPP, "cppx"=>CPPX,
       "c"=>C,"python"=>PYTHON,"perl"=>PERL,"java"=>JAVA}
      @fileext_map = {"rb"=>RUBY, "cpp"=>CPP, "c"=>C,
       "py"=>PYTHON, "pl"=>PERL, "java"=>JAVA, "cppx"=>CPPX}
      @xunit_map = {RUBY=>RubyUnit, PYTHON=>PythonUnit, CPPX=>CppUnitX, CPP=>CppUnit,
        JAVA=>JUnit};
      @creator_map = {RUBY=>RubyUnitCreator, PYTHON=>PythonUnitCreator, JAVA=>JUnitCreator,
        CPPX=>CppUnitXCreator, CPP=>CppUnitCreator}
      @adder_map = {RUBY=>RubyUnitAdder, PYTHON=>PythonUnitAdder, JAVA=>JUnitAdder,
        CPPX=>CppUnitXAdder, CPP=>CppUnitAdder}
    end
    
    def eval_argv(argv)
      begin
        @parser.parse(argv)
        if @parser['help']
          usage
          exit(0)
        elsif @parser['version']
          version
          exit(0)
        end
        if @parser['reference']
          xunit = @xunit_map[@lang_map[@parser['reference']]]
          assert_not_nil(xunit, "-r: bad language option")
          xunit.reference
          exit(0)
        end
        @lang = language(@parser['lang'], getfile)
        if @parser['add']
          add
        elsif @parser['skeleton'] or @parser['main']
          create
        end
      rescue  Test::Unit::AssertionFailedError
        print $!.to_s.sub(/Expected .*$/, "")
        exit(1)
      end
    end
    
    def add
      adder = @adder_map[@lang].new
      STDERR.print @parser['add'] + "...\n"
      adder.set_argv(@parser.other)
      output = adder.output
      output.each do |filename, content|
        if @parser['cout']
          print content
        else
          backup(filename)
          file = open(filename, "w")
          file.print content
          file.close
        end
        STDERR.print "update"
      end
    end
    
    def backup(file)
      if File.exist?(file)
        File.unlink(file + ".bak") if File.exist?(file + ".bak")
        File.copy(file, file + ".bak")
      end
    end
    
    def create
      assert_not_nil(@creator_map[@lang], 'language error:lang=#{@lang}')
      creator = @creator_map[@lang].new
      creator.set_argv(@parser.other)
      output = creator.output
      output.each do |filename, content|
        if @parser['cout']
          print content
        else
          STDERR.print filename + "..."
          return if file_exist?(filename)
          backup(filename)
          file = open(filename, "w")
          file.print content
          file.close
          STDERR.print "create\n"
        end
      end
    end
    
    def getfile
      if @parser['skeleton']
        file = @parser['skeleton']
        @parser.other += ['-s', file.sub(/\..+$/, "")]
      end
      if @parser['main']
        file = @parser['main']
        @parser.other += ['-m', file.sub(/\..+$/, "")]
      end
      if @parser['add']
        file = @parser['add']
        @parser.other += ['-a', file]
      end
      return file
    end
    
    def language(lang, file="")
      if lang
        ret = @lang_map[lang]
        assert_not_nil(ret, "bad language: #{file}")
        return ret
      elsif file =~ /.*\.(.*)/
        ret = @fileext_map[$1]
        assert_not_nil(ret, "bad file ext: #{file}")
        return ret
      else
        return $default
      end
    end
    
    def version
      puts "umigame version 2.0-2002/07/02"
      print "  for "
      @xunit_map.each_value do |lang|
        print lang.version + " "
      end
      print "\n\n"
    end
    
    def usage
      version
      print <<"EOB"
usage:
  skeleton: umigame -s Class [-l lang] [-Pf] [--cout] [-i include] [methods]
  main:     umigame -m MainFile [-l lang] [-f] [--cout]
  add:      umigame -a File [-l lang] [--cout] [-i include] [methods]
  
mode_option:
  --help,      -h        : display help
  --lang,      -l <Name> : output language
  --reference, -r <Name> : assertion quick reference for <Name> language
  --version,   -v        : print version
  --main,      -m <Name> : create test launcher
  --skeleton,  -s <Name> : create skeleton class <Name>
  --add,       -a <Name> : add testmethod to existing TestCase file

common_options:
  --cin                  : input from standard-in
  --cout                 : output from standard-out
  --force,     -f        : force creation

output_options:
  --include,   -i <Name> : include 'Name'
  --noregist,  -N        : disable registration
  --prefix,    -P        : add prefix 'test'
EOB
      @xunit_map.each_value do |lang|
        lang.usage
      end
      exit(0)
    end
    
    def file_exist?(path)
      if File.exist?(path) && !@parser['force']
        STDERR.print "#{path} already exists (skip)\n"
        true
      elsif (File.exist?(path) && @parser['force']) || !File.exist?(path)
        false
      end
    end
  end
  
  def Umigame.main
    umigame = Umigame::Core.new
    if ARGV.include? '--dos'
      args = File.readlines(ARGV[1])
      args.shift
      args.collect! do |arg|
        arg.split.first
      end
      umigame.eval_argv(args)
    elsif ARGV.include? '--cin'
      STDIN.each_line do |line|
        if line[0].chr == "-"
          arg = line.split(' ')
          umigame.eval_argv(arg)
        else
          puts line
        end
      end
    elsif ARGV.empty?
      umigame.eval_argv(['-v'])
    else
      umigame.eval_argv(ARGV)
    end
  end
end
