require 'umigame/xunit'
require 'umigame/creator'
require 'umigame/adder'
require 'umigame/sourcecode'
require 'umigame/method'

module Umigame

  class CppUnitX < XUnit
    def CppUnitX.version
      "CppUnit-x-20020331"
    end
    
    def CppUnitX.usage
      print <<"EOB"

#{CppUnitX.version}(--lang cppx)
  support option: -a, -i, -m, -N, -P, -r, -s
EOB
    end
    
    def CppUnitX.reference
      print <<"EOB"
Assertion Quick Reference
  for #{CppUnitX.version}

TEST_ASSERT(cond)
TEST_ASSERT_MESSAGE(cond, msg)
TEST_ASSERT_EQUALS(expected, actual)
TEST_ASSERT_EQUALS_DOUBLE(expected, actual, delta)
TEST_FAIL(msg)
TEST_ERROR(msg)
EOB
    end
  end
  
  class CppUnitXMethod < Method
    def initialize(class_name, method_name, statement)
      super(class_name, method_name, "//", statement)
    end
    
    def decl
     ret = []
      ret.push "void #{name}() {"
      ret.push @statement
      ret.push "}"
      return prefix + ret.join("\n" + prefix) + "\n"
    end
    
    def suite
      return "#{prefix}  suite->addTest(new TEST_CALLER(#{@classname}, #{name}));\n"
    end
  end
  
  class CppUnitXSourceCode < SourceCode
    def add_method(methodname, prefix=false, statement=nil)
      if statement
        statement =~ /^(.*)\}/m
        statement = $1
      else
        statement = "  TEST_FAIL(\"no implementation\");"
      end
      method = CppUnitXMethod.new(@class_name, methodname, statement)
      super(method, prefix)
    end
    
    def print
      include = ""
      decl = ""
      suite = <<"EOB"
  static Test* suite() {
    TestSuite* suite = new TestSuite;
//CUPPA:suite=+
EOB
      @include.each do |file|
        include +="\ninclude \"#{file}\""
      end
      @method.each do |method|
        decl += "\n" + method.decl
        suite += method.suite
      end
      suite += "//CUPPA:suite=-\n    return suite;\n  }"
      class_end = @class_end.join
      return <<"EOB"
#include <cppunit/TestCaller.h>
#include <cppunit/TestSuite.h>
#include <cppunit/TestCase.h>
USING_NAMESPACE_CPPUNIT
//CUPPA:include=+#{include}
//CUPPA:include=-
#{@class_begin.join}//CUPPA:decl=+#{decl}
//CUPPA:decl=-

#{suite}
#{class_end}
EOB
    end
  end
  
  class CppUnitXAdder < Adder
    def initialize
      super("//", CppUnitXSourceCode.new)
    end
    
    def add_method(method_name)
      @code.add_method(method_name)
    end
    
    protected
    
    def parse_include
      do_parse_include(/#include "(.+)"/)
    end
    
    def parse_class_begin
      do_parse_class_begin(/class (.+) : public TestCase/)
    end
    
    def parse_method
      do_parse_method(/^(..)void (.+)\(\) \{/)
    end
    
    def parse_suite
      do_parse_suite(/static Test\* suite\(\)/, 
        /(\/\/|\s\s)\s*suite->addTest\(new TEST_CALLER\(#{@code.class_name},\s*(.+)\)\);/)
    end
  end
  
  class CppUnitXCreator < Creator
    
    attr_reader :code
    
    def filename(name)
      name + ".cpp"
    end
    
    def eval_argv
    end
    
    def write_skeleton
      code = CppUnitXSourceCode.new
      code.class_name = @skeleton
      @include.each do |includefile|
        code.add_include includefile
      end
      code.class_begin.push <<"EOB"

extern void add_test(Test* test);

class #{@skeleton} : public TestCase {
  private:
  // your stuff...
  
  public:
  explicit #{@skeleton}(const char* name) : TestCase(name) {}
  
  virtual void setUp() {
    // initialize
  }
  
  virtual void tearDown() {
    // terminate
  }

EOB
      code.has_suite = !@prefix

      @testmethod.each do | testcase |
        code.add_new_method testcase
      end

      unless @noregist
        code.class_end.push <<"EOB"
  class Registrer {
    public: Registrer() {
      add_test(#{@skeleton}::suite());
    }
  };
};
static #{@skeleton}::Registrer registrer;
EOB
      else
        code.class_end.push "};"
      end
      return code.print
    end
    
    def write_main
     return <<"EOF"
#include <cppunit/SimpleList.h>
#include <cppunit/TestSuite.h>
#include <cppunit/TestRunner.h>

USING_NAMESPACE_CPPUNIT

SimpleList& registry() {
  static SimpleList registry_;
  return registry_;
}

void add_test(Test* test) {
  registry().push_back(test);
}

TestSuite* get_test() {
  SimpleList::iterator i;
  TestSuite* suite = new TestSuite;
  for(i=registry().begin();i!=registry().end();++i) {
    suite->addTest(static_cast<Test*>(*i));
    *i = 0;
  }
  return suite;
}

int main(int argc, char** argv) {
  TestRunner runner;
  runner.addTest(get_test());
  return runner.run(argc, argv);
}
EOF
    end
  end
end
