/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.property.css2;

import junit.framework.TestCase;

import org.trix.cuery.property.CascadableProperty;
import org.trix.cuery.property.SimpleProperty;
import org.trix.cuery.value.CSSOperator;
import org.trix.cuery.value.CSSPercentage;
import org.trix.cuery.value.CSSString;
import org.trix.cuery.value.CSSValue;
import org.trix.cuery.value.Identifier;
import org.trix.cuery.value.PixelLength;
import org.trix.cuery.value.ValueUtil;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: FontDefinitionTest.java,v 1.01 2005/09/28 11:06:46 Teletha Exp $
 */
public class FontDefinitionTest extends TestCase {

    /*
     * Test method for 'org.trix.cuery.property.ShorthandPropertyDefinition.parse(CSSValue)'
     */
    public void testParse1() {
        CSSValue value = new PixelLength(12, null);
        CSSValue value2 = new CSSOperator("/", value);
        CSSValue value3 = new PixelLength(14, value2);
        new Identifier("sans-serif", value3);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("font", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(property, CascadableProperty.ORIGIN_AGENT, 1, 1);

        assertEquals("12", cascadable.getValueAsString("font-size"));
        assertEquals("14", cascadable.getValueAsString("line-height"));

        assertEquals("12px/14px sans-serif", ValueUtil.toString(cascadable.getValue("font")));
    }

    /*
     * Test method for 'org.trix.cuery.property.ShorthandPropertyDefinition.parse(CSSValue)'
     */
    public void testParse2() {
        CSSValue value = new CSSPercentage(80, null);
        new Identifier("sans-serif", value);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("font", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(property, CascadableProperty.ORIGIN_AGENT, 1, 1);

        assertEquals("9.6", cascadable.getValueAsString("font-size"));
        assertEquals("11.5", cascadable.getValueAsString("line-height"));

        assertEquals("80% sans-serif", ValueUtil.toString(cascadable.getValue("font")));
    }

    /*
     * Test method for 'org.trix.cuery.property.ShorthandPropertyDefinition.parse(CSSValue)'
     */
    public void testParse3() {
        CSSValue value = new Identifier("x-large", null);
        CSSValue value2 = new CSSOperator("/", value);
        CSSValue value3 = new CSSPercentage(110, value2);
        CSSValue value4 = new CSSString("new century schoolbook", value3);
        CSSValue value5 = new CSSOperator(",", value4);
        new Identifier("serif", value5);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("font", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(property, CascadableProperty.ORIGIN_AGENT, 1, 1);

        assertEquals("17.3", cascadable.getValueAsString("font-size"));
        assertEquals("19", cascadable.getValueAsString("line-height"));

        assertEquals("x-large/110% \"new century schoolbook\", serif", ValueUtil.toString(cascadable.getValue("font")));
    }

    /*
     * Test method for 'org.trix.cuery.property.ShorthandPropertyDefinition.parse(CSSValue)'
     */
    public void testParse4() {
        CSSValue value = new Identifier("bold");
        CSSValue value2 = new Identifier("italic", value);
        CSSValue value3 = new Identifier("large", value2);
        CSSValue value4 = new Identifier("Palatino", value3);
        CSSValue value5 = new CSSOperator(",", value4);
        new Identifier("serif", value5);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("font", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(property, CascadableProperty.ORIGIN_AGENT, 1, 1);

        assertEquals("bold", cascadable.getValueAsString("font-weight"));
        assertEquals("italic", cascadable.getValueAsString("font-style"));
        assertEquals("normal", cascadable.getValueAsString("font-variant"));
        assertEquals("14.4", cascadable.getValueAsString("font-size"));
        assertEquals("17.3", cascadable.getValueAsString("line-height"));

        assertEquals("bold italic large Palatino, serif", ValueUtil.toString(cascadable.getValue("font")));
    }

    /*
     * Test method for 'org.trix.cuery.property.ShorthandPropertyDefinition.parse(CSSValue)'
     */
    public void testParse5() {
        CSSValue value = new Identifier("normal");
        CSSValue value2 = new Identifier("small-caps", value);
        CSSValue value3 = new CSSPercentage(120, value2);
        CSSValue value4 = new CSSOperator("/", value3);
        CSSValue value5 = new CSSPercentage(120, value4);
        new Identifier("fantasy", value5);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("font", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(property, CascadableProperty.ORIGIN_AGENT, 1, 1);

        assertEquals("normal", cascadable.getValueAsString("font-weight"));
        assertEquals("normal", cascadable.getValueAsString("font-style"));
        assertEquals("small-caps", cascadable.getValueAsString("font-variant"));
        assertEquals("14.4", cascadable.getValueAsString("font-size"));
        assertEquals("17.3", cascadable.getValueAsString("line-height"));
        assertEquals("fantasy", cascadable.getValueAsString("font-family"));

        assertEquals("normal small-caps 120%/120% fantasy", ValueUtil.toString(cascadable.getValue("font")));
    }

    /*
     * Test method for 'org.trix.cuery.property.ShorthandPropertyDefinition.parse(CSSValue)'
     */
    public void testParse6() {
        CSSValue value = new Identifier("oblique");
        CSSValue value2 = new PixelLength(12, value);
        CSSValue value3 = new CSSString("Helvetica Nue", value2);
        CSSValue value4 = new CSSOperator(",", value3);
        new Identifier("serif", value4);

        SimpleProperty property = new SimpleProperty();
        property.setProperty("font", value, false);

        CascadableProperty cascadable = new CascadableProperty();
        cascadable.addProperty(property, CascadableProperty.ORIGIN_AGENT, 1, 1);

        assertEquals("normal", cascadable.getValueAsString("font-weight"));
        assertEquals("oblique", cascadable.getValueAsString("font-style"));
        assertEquals("normal", cascadable.getValueAsString("font-variant"));
        assertEquals("12", cascadable.getValueAsString("font-size"));
        assertEquals("14.4", cascadable.getValueAsString("line-height"));
        assertEquals("\"Helvetica Nue\", serif", ValueUtil.toString(cascadable.getValue("font-family")));

        assertEquals("oblique 12px \"Helvetica Nue\", serif", ValueUtil.toString(cascadable.getValue("font")));
    }
}
