/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.filter;

import org.w3c.css.sac.CombinatorCondition;
import org.w3c.css.sac.Condition;
import org.w3c.css.sac.ConditionalSelector;
import org.w3c.css.sac.Selector;
import org.w3c.css.sac.SimpleSelector;
import org.w3c.dom.Element;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: MultipleFilter.java,v 1.0 2005/07/26 1:55:44 Teletha Exp $
 */
public class MultipleFilter implements Filter, CombinatorCondition, ConditionalSelector {

    /** The first filter. */
    private Filter first;

    /** The second filter. */
    private Filter second;

    /**
     * Create MultipleFilter instance.
     * 
     * @param first A first filter.
     * @param second A second filter.
     */
    public MultipleFilter(Filter first, Filter second) {
        this.first = first;
        this.second = second;
    }

    /**
     * @see org.trix.cuery.filter.Filter#accept(org.w3c.dom.Element)
     */
    public boolean accept(Element target) {
        return first.accept(target) && second.accept(target);
    }

    /**
     * @see org.trix.cuery.filter.Filter#getSpecificity()
     */
    public int getSpecificity() {
        return first.getSpecificity() + second.getSpecificity();
    }

    /**
     * @see org.trix.cuery.filter.Filter#getState()
     */
    public int getState() {
        return first.getState() | second.getState();
    }

    /**
     * @see org.w3c.css.sac.Condition#getConditionType()
     */
    public short getConditionType() {
        return Condition.SAC_AND_CONDITION;
    }

    /**
     * @see org.w3c.css.sac.Selector#getSelectorType()
     */
    public short getSelectorType() {
        return SAC_CONDITIONAL_SELECTOR;
    }

    /**
     * @see org.w3c.css.sac.CombinatorCondition#getFirstCondition()
     */
    public Condition getFirstCondition() {
        return first;
    }

    /**
     * @see org.w3c.css.sac.CombinatorCondition#getSecondCondition()
     */
    public Condition getSecondCondition() {
        return second;
    }

    /**
     * @see org.w3c.css.sac.ConditionalSelector#getCondition()
     */
    public Condition getCondition() {
        return second;
    }

    /**
     * @see org.w3c.css.sac.ConditionalSelector#getSimpleSelector()
     */
    public SimpleSelector getSimpleSelector() {
        if (first instanceof SimpleSelector) {
            return (SimpleSelector) first;
        } else {
            return new SimpleSelectorWrapper(first);
        }
    }

    /**
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return first.toString() + second.toString();
    }

    /**
     * DOCUMENT.
     * 
     * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
     * @version $ Id: SimpleSelectorWrapper.java,v 1.0 2005/07/29 12:08:50 Teletha Exp $
     */
    private static final class SimpleSelectorWrapper implements SimpleSelector {

        /** The actual selector. */
        private Selector selector;

        /**
         * Create SimpleSelectorWrapper instance.
         * 
         * @param selector An actual selector.
         */
        public SimpleSelectorWrapper(Selector selector) {
            this.selector = selector;
        }

        /**
         * @see org.w3c.css.sac.Selector#getSelectorType()
         */
        public short getSelectorType() {
            return selector.getSelectorType();
        }

        /**
         * @see java.lang.Object#toString()
         */
        public String toString() {
            return selector.toString();
        }
    }
}
