/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.css;

import org.trix.cuery.sac.SACMediaListImpl;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.LexicalUnit;
import org.w3c.css.sac.SACMediaList;
import org.w3c.css.sac.SelectorList;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: MediaAwarePipe.java,v 1.0 2005/08/05 17:49:47 Teletha Exp $
 */
public class MediaAwarePipe extends AbstractCSSPipe {

    /** The awarable media. */
    private SACMediaListImpl awareList = new SACMediaListImpl();

    /** The current mode. */
    private boolean aware = true;

    /**
     * Set awarable media types.
     * 
     * @param media a media type.
     */
    public void setAwareMedia(String media) {
        awareList.appendMedium(media);
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#startDocument(org.w3c.css.sac.InputSource)
     */
    public void startDocument(InputSource inputSource) throws CSSException {
        super.startDocument(inputSource);
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#endDocument(org.w3c.css.sac.InputSource)
     */
    public void endDocument(InputSource inputSource) throws CSSException {
        super.endDocument(inputSource);
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#comment(java.lang.String)
     */
    public void comment(String comment) throws CSSException {
        if (aware) {
            super.comment(comment);
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#ignorableAtRule(java.lang.String)
     */
    public void ignorableAtRule(String atRule) throws CSSException {
        if (aware) {
            super.ignorableAtRule(atRule);
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#namespaceDeclaration(java.lang.String, java.lang.String)
     */
    public void namespaceDeclaration(String prefix, String namespaceURI) throws CSSException {
        if (aware) {
            super.namespaceDeclaration(prefix, namespaceURI);
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#importStyle(java.lang.String,
     *      org.w3c.css.sac.SACMediaList, java.lang.String)
     */
    public void importStyle(String uri, SACMediaList mediaList, String defaultNamespaceURI) throws CSSException {
        if (aware && isAware(mediaList)) {
            super.importStyle(uri, mediaList, defaultNamespaceURI);
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#startMedia(org.w3c.css.sac.SACMediaList)
     */
    public void startMedia(SACMediaList mediaList) throws CSSException {
        aware = isAware(mediaList);

        if (aware) {
            super.startMedia(mediaList);
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#endMedia(org.w3c.css.sac.SACMediaList)
     */
    public void endMedia(SACMediaList mediaList) throws CSSException {
        if (aware) {
            super.endMedia(mediaList);
        }
        aware = true;
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#startPage(java.lang.String, java.lang.String)
     */
    public void startPage(String name, String pseudoPage) throws CSSException {
        if (aware) {
            super.startPage(name, pseudoPage);
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#endPage(java.lang.String, java.lang.String)
     */
    public void endPage(String name, String pseudoPage) throws CSSException {
        if (aware) {
            super.endPage(name, pseudoPage);
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#startFontFace()
     */
    public void startFontFace() throws CSSException {
        if (aware) {
            super.startFontFace();
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#endFontFace()
     */
    public void endFontFace() throws CSSException {
        if (aware) {
            super.endFontFace();
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#startSelector(org.w3c.css.sac.SelectorList)
     */
    public void startSelector(SelectorList selectors) throws CSSException {
        if (aware) {
            super.startSelector(selectors);
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#endSelector(org.w3c.css.sac.SelectorList)
     */
    public void endSelector(SelectorList selectors) throws CSSException {
        if (aware) {
            super.endSelector(selectors);
        }
    }

    /**
     * @see org.w3c.css.sac.DocumentHandler#property(java.lang.String, org.w3c.css.sac.LexicalUnit,
     *      boolean)
     */
    public void property(String name, LexicalUnit value, boolean important) throws CSSException {
        if (aware) {
            super.property(name, value, important);
        }
    }

    /**
     * Check media type.
     * 
     * @param mediaList A media list.
     * @return Whether this media is aware or not.
     */
    private boolean isAware(SACMediaList mediaList) {
        // check null
        if (mediaList == null) {
            return false;
        }

        for (int i = 0; i < mediaList.getLength(); i++) {
            String mediaType = mediaList.item(i);

            // check all
            if (mediaType.equalsIgnoreCase("all")) {
                return true;
            }

            for (int j = 0; j < awareList.getLength(); j++) {
                if (mediaType.equalsIgnoreCase(awareList.item(j))) {
                    return true;
                }
            }
        }
        return false;
    }
}
