/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.parser;

import java.io.FileReader;
import java.util.ArrayList;
import java.util.List;

import junit.framework.TestCase;

import org.trix.cuery.css.AbstractCSSConsumer;
import org.trix.cuery.css.CSSConsumer;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.CSSParseException;
import org.w3c.css.sac.InputSource;
import org.w3c.css.sac.LexicalUnit;
import org.w3c.css.sac.SACMediaList;
import org.w3c.css.sac.Selector;
import org.w3c.css.sac.SelectorList;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: CueryParserTest.java,v 1.0 2005/07/28 8:20:09 Teletha Exp $
 */
public class CueryParserTest extends TestCase {

    /** The test parser. */
    private static CueryParser parser;

    // initialize
    static {
        parser = new CueryParser();

        CSSConsumer consumer = new CountHandler();
        parser.setDocumentHandler(consumer);
        parser.setErrorHandler(consumer);
    }

    /*
     * Test method for 'org.trix.cuery.parser.CueryParser.parseStyleSheet(String)'
     */
    public final void testParseStyleSheetString() {
        try {
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/normal.css");
        } catch (Exception e) {
            System.out.println(e);
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test method for 'org.trix.cuery.parser.CueryParser.parseStyleSheet(InputSource)'
     */
    public final void testParseStyleSheetInputSource() {
        try {
            FileReader reader = new FileReader("src/test/org/trix/cuery/parser/normal.css");
            InputSource is = new InputSource(reader);

            parser.parseStyleSheet(is);
        } catch (Exception e) {
            System.out.println(e);
            fail(e.getLocalizedMessage());
        }

    }

    /*
     * Test method for 'org.trix.cuery.parser.CueryParser.parseStyleSheet(String)'
     */
    public final void testParseStyleSheetCharset() {
        try {
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/charset-Shift_JIS.css");
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/charset-JIS.css");
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/charset-EUC-JP.css");
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/charset-UTF-8.css");
        } catch (Exception e) {
            System.out.println(e);
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test method for 'org.trix.cuery.parser.CueryParser.parseStyleDeclaration(InputSource)'
     */
    public final void testParseStyleDeclaration() {

    }

    /*
     * Test method for 'org.trix.cuery.parser.CueryParser.parseRule(InputSource)'
     */
    public final void testParseRule() {

    }

    /*
     * Test method for 'org.trix.cuery.parser.CueryParser.parseSelector(String)'
     */
    public final void testParseSelector() {
        Selector selector1 = parser.parseSelector("one>two three");
        assertEquals("one > two three", selector1.toString());

        Selector selector2 = parser.parseSelector("one > two three");
        assertEquals(false, selector1.equals(selector2));
        assertEquals(true, selector1.toString().equals(selector2.toString()));
    }

    /*
     * Test method for 'org.trix.cuery.parser.CueryParser.parseSelectors(InputSource)'
     */
    public final void testParseSelectors() {

    }

    /*
     * Test method for 'org.trix.cuery.parser.CueryParser.parsePropertyValue(InputSource)'
     */
    public final void testParsePropertyValue() {

    }

    /*
     * Test method for 'org.trix.cuery.parser.CueryParser.parsePriority(InputSource)'
     */
    public final void testParsePriority() {

    }

    /*
     * Test comment.
     */
    public final void testParse0() {
        try {
            CountHandler handler = new CountHandler();
            parser.setDocumentHandler(handler);
            parser.setErrorHandler(handler);
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/parsing0.css");

            assertEquals(11, handler.getSelecterCount());
            assertEquals(1, handler.getWarningCounter());
        } catch (Exception e) {
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test comment.
     */
    public final void testParse1() {
        try {
            CountHandler handler = new CountHandler();
            parser.setDocumentHandler(handler);
            parser.setErrorHandler(handler);
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/parsing1.css");

            assertEquals(5, handler.getSelecterCount());
            assertEquals(1, handler.getWarningCounter());
        } catch (Exception e) {
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test comment.
     */
    public final void testParse2() {
        try {
            CountHandler handler = new CountHandler();
            parser.setDocumentHandler(handler);
            parser.setErrorHandler(handler);
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/parsing2.css");

            assertEquals(13, handler.getSelecterCount());
            assertEquals(3, handler.getWarningCounter());
        } catch (Exception e) {
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test comment.
     */
    public final void testParse3() {
        try {
            CountHandler handler = new CountHandler();
            parser.setDocumentHandler(handler);
            parser.setErrorHandler(handler);
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/parsing3.css");

            assertEquals(3, handler.getSelecterCount());
            assertEquals(1, handler.getWarningCounter());
        } catch (Exception e) {
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test comment.
     */
    public final void testParse4() {
        try {
            CountHandler handler = new CountHandler();
            parser.setDocumentHandler(handler);
            parser.setErrorHandler(handler);
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/parsing4.css");

            assertEquals(1, handler.getSelecterCount());
            assertEquals(1, handler.getWarningCounter());
        } catch (Exception e) {
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test comment.
     */
    public final void testParse5() {
        try {
            CountHandler handler = new CountHandler();
            parser.setDocumentHandler(handler);
            parser.setErrorHandler(handler);
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/parsing5.css");

            assertEquals(2, handler.getPropertyCount());
            assertEquals(1, handler.getWarningCounter());
        } catch (Exception e) {
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test comment.
     */
    public final void testComment() {
        try {
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/comment.css");
        } catch (Exception e) {
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test color.
     */
    public final void testColor() {
        try {
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/color.css");
        } catch (Exception e) {
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test import.
     */
    public final void testImport() {
        try {
            final List<String> list = new ArrayList<String>();
            CSSConsumer consumer = new AbstractCSSConsumer() {

                public void importStyle(String uri, SACMediaList mediaList, String defaultNamespaceURI)
                        throws CSSException {
                    list.add(uri);
                }
            };

            parser.setDocumentHandler(consumer);
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/import.css");

            assertEquals(3, list.size());
            assertEquals("imported.css", list.get(0));
        } catch (Exception e) {
            fail(e.getLocalizedMessage());
        }
    }

    /*
     * Test function.
     */
    public final void testFunction() {
        try {
            CountHandler handler = new CountHandler();
            parser.setDocumentHandler(handler);
            parser.parseStyleSheet("src/test/org/trix/cuery/parser/function.css");
        } catch (Exception e) {
            fail(e.getLocalizedMessage());
        }
    }

    /**
     * DOCUMENT.
     * 
     * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
     * @version $ Id: CountHandler.java,v 1.0 2005/07/28 8:27:55 Teletha Exp $
     */
    public static final class CountHandler extends AbstractCSSConsumer {

        /** The selector counter. */
        private int selectorCounter = 0;

        /** The property counter. */
        private int propertyCounter = 0;

        /** The warning counter. */
        private int warningCounter = 0;

        /**
         * @see org.w3c.css.sac.DocumentHandler#endSelector(org.w3c.css.sac.SelectorList)
         */
        public void endSelector(SelectorList list) throws CSSException {
            selectorCounter++;
        }

        /**
         * @see org.w3c.css.sac.DocumentHandler#property(java.lang.String,
         *      org.w3c.css.sac.LexicalUnit, boolean)
         */
        public void property(String property, LexicalUnit unit, boolean important) throws CSSException {
            propertyCounter++;
            LexicalUnit next = unit.getNextLexicalUnit();

            while (next != null) {
                next = next.getNextLexicalUnit();
            }
        }

        /**
         * @see org.w3c.css.sac.ErrorHandler#warning(org.w3c.css.sac.CSSParseException)
         */
        public void warning(CSSParseException exception) throws CSSException {
            warningCounter++;
        }

        /**
         * @return A number of selector.
         */
        public int getSelecterCount() {
            return selectorCounter;
        }

        /**
         * @return A number of property.
         */
        public int getPropertyCount() {
            return propertyCounter;
        }

        /**
         * @return A number of warning.
         */
        public int getWarningCounter() {
            return warningCounter;
        }
    }

}
