/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.css;

import java.util.HashMap;
import java.util.Map;

import org.trix.cuery.value.CSSColor;
import org.trix.cuery.value.CSSLength;
import org.trix.cuery.value.CSSNumber;
import org.trix.cuery.value.CSSPercentage;
import org.trix.cuery.value.CSSString;
import org.trix.cuery.value.CSSURI;
import org.trix.cuery.value.Value;

import org.w3c.css.sac.CSSException;
import org.w3c.css.sac.LexicalUnit;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: ValidatableCSSPipe.java,v 1.0 2005/08/10 1:59:41 Teletha Exp $
 */
public class ValidatableCSSPipe extends AbstractCSSPipe {

    /** The property validation map. */
    private static final Map<String, Object[]> VALIDATION = new HashMap<String, Object[]>();

    /**
     * @see org.w3c.css.sac.DocumentHandler#property(java.lang.String, org.w3c.css.sac.LexicalUnit,
     *      boolean)
     */
    @Override
    public void property(String name, LexicalUnit value, boolean important) throws CSSException {
        Object[] acceptables = VALIDATION.get(name);

        if (acceptables == null) {
            consumer.property(name, value, important);
            return;
        }

        for (Object acceptable : acceptables) {
            if (acceptable instanceof Class) {
                Class clazz = (Class) acceptable;

                if (clazz.isAssignableFrom(value.getClass())) {
                    consumer.property(name, value, important);
                    return;
                }
            } else if (acceptable instanceof String) {
                String string = (String) acceptable;

                if (string.equalsIgnoreCase(value.getStringValue())) {
                    consumer.property(name, value, important);
                    return;
                }
            }
        }
    }

    // initialize
    static {
        VALIDATION.put("background-attachment", new Object[] {"scroll", "fixed", Value.INHERIT});
        VALIDATION.put("background-color", new Object[] {CSSColor.class, "transparent", Value.INHERIT});
        VALIDATION.put("background-image", new Object[] {CSSURI.class, "none", Value.INHERIT});
        VALIDATION.put("background-position", new Object[] {CSSColor.class, "transparent", Value.INHERIT});
        VALIDATION
                .put("background-repeat", new Object[] {"repeat", "repeat-x", "repeat-y", "no-repeat", Value.INHERIT});
        VALIDATION.put("color", new Object[] {CSSColor.class, Value.INHERIT});
        VALIDATION.put("font-size'", new Object[] {CSSLength.class, CSSPercentage.class, "xx-small", "x-small",
                "small", "medium", "large", "x-large", "xx-large", "larger", "smaller", Value.INHERIT});
        VALIDATION.put("font-size-adjust'", new Object[] {"none", CSSNumber.class, Value.INHERIT});
        VALIDATION.put("font-stretch", new Object[] {"normal", "wider", "narrower", "ultra-condensed",
                "extra-condensed", "condensed", "semi-condensed", "semi-expanded", "expanded", "extra-expanded",
                "ultra-expanded", Value.INHERIT});
        VALIDATION.put("font-style'", new Object[] {"normal", "italic", "oblique", Value.INHERIT});
        VALIDATION.put("font-variant'", new Object[] {"normal", "small-caps", Value.INHERIT});
        VALIDATION.put("font-weight", new Object[] {"normal", "bold", "bolder", "lighter", "100", "200", "300", "400",
                "500", "600", "700", "800", "900", Value.INHERIT});
        VALIDATION.put("letter-spaceing'", new Object[] {"normal", CSSLength.class, Value.INHERIT});
        VALIDATION.put("text-align'", new Object[] {"left", "right", "center", "justify", CSSString.class,
                Value.INHERIT});
        VALIDATION.put("text-decoration'", new Object[] {"none", "underline", "overline", "line-through", "blink",
                Value.INHERIT});
        VALIDATION.put("text-indet'", new Object[] {CSSLength.class, CSSPercentage.class, Value.INHERIT});
        VALIDATION.put("text-transform'", new Object[] {"none", "capitalize", "uppercase", "lowercase", Value.INHERIT});
        VALIDATION.put("white-space'", new Object[] {"normal", "pre", "nowrap", Value.INHERIT});
        VALIDATION.put("word-spaceing'", new Object[] {"normal", CSSLength.class, Value.INHERIT});
    }
    // background-position, background, font-family, font, text-shadow

}
