/*
 *  Breeze  --  An application launcher with command-line style
 *  Copyright (C) 2005, 2006 Hironao Komatsu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <glib.h>

#include "breeze.h"
#include "user_commands.h"
#include "config.h"

#define USER_COMMANDS_FILE ".breeze_commands"

GHashTable *user_commands = NULL;
gboolean user_commands_changed = FALSE;

static const char *user_commands_file = NULL;

static const char *get_user_commands_file(void)
{
  char str[256];

  snprintf(str, sizeof(str), "%s/%s", getenv("HOME"), USER_COMMANDS_FILE);

  return strdup(str);
}

static int create_user_commands_file(void)
{
  if (creat(user_commands_file, 0600) == -1) {
    fprintf(stderr, "Cannot create commands file.\n");
    return -1;
  }

  return 0;
}

/*
 *  specification of commands file:
 *
 *    lines of odd line number -- hash key (keyword)
 *    lines of even line number -- hash value (command)
 */
int load_user_commands(void)
{
  FILE *fp;
  char buf[256];
  gchar *keyword = NULL, *command = NULL;
  gboolean odd = FALSE;

  if (!user_commands_file)
    user_commands_file = get_user_commands_file();

 retry:
  if ((fp = fopen(user_commands_file, "r")) == NULL) {
    fprintf(stderr, "Your commands file does not appear. Creating new.\n");

    if (create_user_commands_file() == -1)
      return -1;

    goto retry;
  }

  free_user_commands();

  user_commands = g_hash_table_new_full(g_str_hash, g_str_equal,
					g_free, g_free);

  while (fgets(buf, sizeof(buf), fp) != NULL) {
    gchar *buf2;
    gboolean is_empty;

    buf2 = strip(buf, &is_empty);

    if (is_empty)
      fprintf(stderr, "warning: empty command.\n");

    odd = !odd;

    if (odd)
      keyword = g_strdup(buf2);
    else {
      command = g_strdup(buf2);
      g_hash_table_insert(user_commands, keyword, command);
    }
  }

  if (odd) {
    fprintf(stderr, "warning: unmatched commands; ignored.\n");
    g_free(keyword);
  }

  fclose(fp);

  user_commands_changed = FALSE;

  return 0;
}

void user_commands_table_foreach(GHFunc func, gpointer user_data)
{
  g_hash_table_foreach(user_commands, func, user_data);
}

int insert_user_commands(gchar *keyword, gchar *command)
{
  g_hash_table_insert(user_commands, keyword, command);
  user_commands_changed = TRUE;

  return 0;
}

gconstpointer lookup_user_commands(const gchar *keyword)
{
  return g_hash_table_lookup(user_commands, keyword);
}

static void write_commands_func(gpointer keyword, gpointer command,
				gpointer user_data)
{
  FILE *fp = (FILE*)user_data;

  fputs(keyword, fp);
  fputc('\n', fp);
  fputs(command, fp);
  fputc('\n', fp);
}

int save_user_commands(void)
{
  FILE *fp;

  if (!user_commands_changed)
    return 0;

  if ((fp = fopen(user_commands_file, "w")) == NULL) {
    fprintf(stderr, "Failed to open commands file.\n");
    return -1;
  }

  g_hash_table_foreach(user_commands, write_commands_func, fp);

  fclose(fp);

  user_commands_changed = FALSE;

  return 0;
}

void free_user_commands(void)
{
  if (!user_commands)
    ;   /* commands already freed or not loaded */
  else {
    if (user_commands_changed)
      save_user_commands();

    g_hash_table_destroy(user_commands);
    user_commands = NULL;
  }
}

