/*
 *  Breeze  --  An application launcher with command-line style
 *  Copyright (C) 2005, 2006 Hironao Komatsu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <sys/wait.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#include <gtk/gtk.h>

#include "breeze.h"
#include "config.h"

#include "getopt.h"

/*
 *  Edit here to grab another key.
 *
 *  Default is "Windows key + space" (PC/AT keyboard)
 */
#define GRAB_KEY XK_space
#define GRAB_KEY_MASK Mod4Mask

volatile gboolean continue_loop_flag = TRUE;

static int key_to_grab;
static unsigned int modifiers_to_grab;
static unsigned int NumLockMask, CapsLockMask, ScrollLockMask;

static Display *dpy = NULL;
static Window root;

static void finalize(void);

static void signal_handler(int sig)
{
  switch (sig) {
  case SIGTERM:
  case SIGINT:
  case SIGHUP:
    finalize();
    exit(0);
  case SIGSEGV:
    finalize();
    exit(1);
  case SIGCHLD:
    wait(NULL);
    break;
  default:
    ;
  }
}

/*
 *  Functions init_keyboard and grab_key are originally from keylaunch.c;
 *  written by Ken Lynch and Stefan Pfetzing.
 */
static void init_keyboard()
{
  XModifierKeymap *xmk = NULL;
  KeyCode *map;
  int m, k;

  xmk = XGetModifierMapping(dpy);
  if (xmk) {
    map = xmk->modifiermap;
    for (m = 0; m < 8; m++)
      for (k = 0; k < xmk->max_keypermod; k++, map++) {
        if (*map == XKeysymToKeycode(dpy, XK_Num_Lock))
          NumLockMask = (1 << m);
        if (*map == XKeysymToKeycode(dpy, XK_Caps_Lock))
          CapsLockMask = (1 << m);
        if (*map == XKeysymToKeycode(dpy, XK_Scroll_Lock))
          ScrollLockMask = (1 << m);
      }
    XFreeModifiermap(xmk);
  }
}

static void grab_key(int keycode, unsigned int modifiers, Window w)
{
  if(keycode) {
    XGrabKey(dpy, keycode, modifiers, w, False, GrabModeAsync, GrabModeAsync);
    XGrabKey(dpy, keycode, modifiers | NumLockMask,
	     w, False, GrabModeAsync, GrabModeAsync);
    XGrabKey(dpy, keycode, modifiers | CapsLockMask,
	     w, False, GrabModeAsync, GrabModeAsync);
    XGrabKey(dpy, keycode, modifiers | ScrollLockMask,
	     w, False, GrabModeAsync, GrabModeAsync);
    XGrabKey(dpy, keycode, modifiers | NumLockMask | CapsLockMask,
	     w, False, GrabModeAsync, GrabModeAsync);
    XGrabKey(dpy, keycode, modifiers | NumLockMask | ScrollLockMask,
	     w, False, GrabModeAsync, GrabModeAsync);
    XGrabKey(dpy, keycode, modifiers | NumLockMask | CapsLockMask | 
	     ScrollLockMask, w, False, GrabModeAsync, GrabModeAsync);
  }
}

static void help(void)
{
  printf("Breeze " VERSION "\n"
	 "  An application launcher with command-line style\n"
	 "\n"
	 "Usage: breeze [options...]\n"
	 "\n"
	 "Options:\n"
	 "  -h or --help    show this message\n"
	 "  -v or --version same as above\n");
}

static void initialize(int argc, char **argv)
{
  int opt;
  const struct option opts[] = {
    { "help", no_argument, NULL, 'h' },
    { "version", no_argument, NULL, 'v' },
    { 0, 0, 0, 0 }
  };
  struct sigaction sa;

  while ((opt = getopt_long(argc, argv, "hv", opts, NULL)) != -1) {
    switch (opt) {
    case 'h':
    case 'v':
      help();
      exit(0);
    case '?':
      exit(0);
    }
  }

  sa.sa_handler = signal_handler;
  sa.sa_flags = 0;

  sigaction(SIGTERM, &sa, NULL);
  sigaction(SIGINT, &sa, NULL);
  sigaction(SIGSEGV, &sa, NULL);
  sigaction(SIGHUP, &sa, NULL);
  sigaction(SIGCHLD, &sa, NULL);

  if ((dpy = XOpenDisplay(NULL)) == NULL) {
    fprintf(stderr, "Can't open display.\n");
    exit(1);
  }

  if (initialize_commands()) {
    fprintf(stderr, "Can't initialize commands table.\n");
    XCloseDisplay(dpy);
    exit(1);
  }

  root = XDefaultRootWindow(dpy);

  init_keyboard();

  key_to_grab = XKeysymToKeycode(dpy, GRAB_KEY);
  modifiers_to_grab = GRAB_KEY_MASK;

  grab_key(key_to_grab, modifiers_to_grab, root);

  gtk_init(&argc, &argv);
}

static void finalize(void)
{
  finalize_commands();

  XUngrabKey(dpy, AnyKey, AnyModifier, root);
  XCloseDisplay(dpy);
}

int main(int argc, char **argv)
{
  initialize(argc, argv);

  while (continue_loop_flag) {
    while (XPending(dpy)) {
      XEvent ev;
      XNextEvent(dpy, &ev);

      if (ev.type == KeyPress &&
	  ev.xkey.keycode == key_to_grab &&
	  ev.xkey.state & modifiers_to_grab)
	popup_mainwin();
    }
    usleep(250);
  }

  finalize();

  return 0;
}
