/*
 * blanco Framework
 * Copyright (C) 2004-2006 IGA Tosiki
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.dbmetadata;

import java.io.ByteArrayInputStream;
import java.io.CharArrayReader;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;

import blanco.dbmetadata.valueobject.BlancoDbMetaDataColumnStructure;

/**
 * f[^x[XSQL𔭍sA^擾郆[eBeBB
 * 
 * @author IGA Tosiki
 */
public class BlancoDbMetaDataSql {
    /**
     * f[^x[XSQL𔭍sA^擾܂B
     * 
     * iSQL`FbN͎s܂B܂ł^擾݂̂{܂B
     * 
     * @throws SQLException
     *             SQLɕsꍇȂǂɔ܂B
     */
    public static List<BlancoDbMetaDataColumnStructure> getResultSetMetaData(
            final Connection conn, final String argSql) throws SQLException {
        // SQL Server 2005 ł́Astmt.executeQuery()
        // sȂꍇɂ́ASQL̓p[^̃ZbgsvłB
        // āAstmt.executeQuery() sȂƂ^͎擾ł܂B

        return getResultSetMetaData(conn, argSql, null);
    }

    /**
     * f[^x[XSQL𔭍sA^擾܂B
     * 
     * SQL𔭍sČʃZbg擾܂BargSqlInParameterɒlZbgĂꍇɂ́ASQL̎łs܂B
     * 
     * @param conn
     *            f[^x[XڑB
     * @param argSql
     *            SQLB
     * @param argSqlInParameter
     *            SQL̓p[^Bnull^ƁAiSQL`FbN̓XLbvꂽs܂B
     * @throws SQLException
     *             SQLɕsꍇȂǂɔ܂B
     */
    public static List<BlancoDbMetaDataColumnStructure> getResultSetMetaData(
            final Connection conn, final String argSql,
            final List<BlancoDbMetaDataColumnStructure> argSqlInParameter)
            throws SQLException {
        final List<BlancoDbMetaDataColumnStructure> result = new ArrayList<BlancoDbMetaDataColumnStructure>();

        final PreparedStatement stmt = conn.prepareStatement(argSql);

        ResultSet resultSet = null;
        try {
            if (argSqlInParameter != null) {
                // System.out.println("SQLSQL̓p[^ZbgŎۂɎsĂ݂܂:["
                // + argSql + "]");

                bindSqlInParameter(stmt, argSqlInParameter);

                resultSet = stmt.executeQuery();
                // SQLsÓÂ܂܃X[܂B
            }

            ResultSetMetaData resultSetMetaData = null;
            if (resultSet != null) {
                // ʃZbĝł΁A𗘗pB
                // ȂȂJDBChCoɂẮAʃZbg擾Ȃstmt擾pXpłȂłB
                resultSetMetaData = resultSet.getMetaData();
            } else {
                resultSetMetaData = stmt.getMetaData();
            }

            if (resultSetMetaData == null) {
                throw new IllegalArgumentException(
                        "^擾: ʃZbg^̎擾Ɏs܂BSQLsIvViteratorɕύXĂB");
            }

            final int columnCount = resultSetMetaData.getColumnCount();
            for (int indexColumn = 1; indexColumn <= columnCount; indexColumn++) {
                final BlancoDbMetaDataColumnStructure columnStructure = parseResultSetMetaData(
                        resultSetMetaData, indexColumn);
                result.add(columnStructure);
            }
        } finally {
            if (resultSet != null) {
                // ʃZbg^f[^擾ĂN[Y悤ɂ܂B
                // Ȃ PostgreSQL 8.1ł̓^f[^擾o܂łB
                resultSet.close();
                resultSet = null;
            }
            stmt.close();
        }

        return result;
    }

    /**
     * ResultSetMetaData\̂ɓWJ܂B
     * 
     * @param resultSetMetaData
     * @param indexColumn
     * @return
     * @throws SQLException
     */
    private static BlancoDbMetaDataColumnStructure parseResultSetMetaData(
            final ResultSetMetaData resultSetMetaData, final int indexColumn)
            throws SQLException {
        final BlancoDbMetaDataColumnStructure columnStructure = new BlancoDbMetaDataColumnStructure();

        columnStructure.setName(resultSetMetaData.getColumnName(indexColumn));

        columnStructure.setDataType(resultSetMetaData
                .getColumnType(indexColumn));
        if (columnStructure.getDataType() == Types.OTHER) {
            BlancoDbMetaDataUtil.mapTypeName2DataType(columnStructure);
        }

        columnStructure.setDataTypeDisplayName(BlancoDbMetaDataUtil
                .convertJdbcDataTypeToString(columnStructure.getDataType()));
        columnStructure.setTypeName(resultSetMetaData
                .getColumnTypeName(indexColumn));
        columnStructure.setColumnSize(resultSetMetaData
                .getPrecision(indexColumn));
        columnStructure.setDecimalDigits(resultSetMetaData
                .getScale(indexColumn));
        columnStructure.setNullable(resultSetMetaData.isNullable(indexColumn));
        columnStructure.setNullableDisplayName(BlancoDbMetaDataUtil
                .convertJdbcNullableToString(columnStructure.getNullable()));

        // SQLsɂ̂ݎ擾\ȍځB
        columnStructure.setWritable(resultSetMetaData.isWritable(indexColumn));

        return columnStructure;
    }

    /**
     * SQL̓p[^oCh܂B
     * 
     * @param stmt
     * @param argSqlInParameter
     * @throws SQLException
     * @throws IllegalArgumentException
     *             SQL̓p[^sȏꍇȂǂɔ܂B
     */
    private static void bindSqlInParameter(final PreparedStatement stmt,
            final List<BlancoDbMetaDataColumnStructure> argSqlInParameter)
            throws SQLException {
        if (argSqlInParameter == null) {
            // p[^null̏ꍇɂ́AoChׂSQL̓p[^̂Ɣf܂B
            return;
        }

        // SQL̓p[^Zbg܂B
        for (int indexInParameter = 0; indexInParameter < argSqlInParameter
                .size(); indexInParameter++) {
            final BlancoDbMetaDataColumnStructure columnStructure = argSqlInParameter
                    .get(indexInParameter);

            switch (columnStructure.getDataType()) {
            case Types.BIT:
            case Types.BOOLEAN:
                stmt.setBoolean(indexInParameter + 1, false);
                break;
            case Types.TINYINT:
                stmt.setByte(indexInParameter + 1, (byte) 0x00);
                break;
            case Types.SMALLINT:
                stmt.setShort(indexInParameter + 1, (short) 0);
                break;
            case Types.INTEGER:
                stmt.setInt(indexInParameter + 1, 0);
                break;
            case Types.BIGINT:
                stmt.setLong(indexInParameter + 1, 0);
                break;
            case Types.REAL:
                stmt.setFloat(indexInParameter + 1, 0);
                break;
            case Types.FLOAT:
            case Types.DOUBLE:
                stmt.setDouble(indexInParameter + 1, 0);
                break;
            case Types.NUMERIC:
            case Types.DECIMAL:
                // "0"Ƃė^Ă܂BJDK1.5΍łB
                stmt.setBigDecimal(indexInParameter + 1, new BigDecimal("0"));
                break;
            case Types.CHAR:
            case Types.VARCHAR:
            case Types.LONGVARCHAR:
                // SQL Server 2005uniqueidentifierł́Anull^̂œKłB
                stmt.setString(indexInParameter + 1, null);
                break;
            case Types.DATE:
                //  TIMESTAMPƓĂ܂B
            case Types.TIME:
                //  TIMESTAMPƓĂ܂B
            case Types.TIMESTAMP:
                stmt.setTimestamp(indexInParameter + 1, new Timestamp(0));
                break;
            case Types.BINARY:
            case Types.VARBINARY:
            case Types.LONGVARBINARY:
            case Types.BLOB: {
                byte[] dummy = "0".getBytes();
                stmt.setBinaryStream(indexInParameter + 1,
                        new ByteArrayInputStream(dummy), dummy.length);
            }
                break;
            case Types.JAVA_OBJECT:
                stmt.setObject(indexInParameter + 1, new Object());
                break;
            case Types.CLOB: {
                char[] dummy = new char[1];
                dummy[0] = '0';
                stmt.setCharacterStream(indexInParameter + 1,
                        new CharArrayReader(dummy), dummy.length);
            }
                break;
            case Types.DISTINCT:
            case Types.STRUCT:
            case Types.ARRAY:
            case Types.NULL:
            case Types.OTHER:
            case Types.REF:
            case Types.DATALINK:
            default:
                throw new IllegalArgumentException(
                        "BlancoDbMetaDataSql: SQL̓p[^["
                                + columnStructure.getName()
                                + "]("
                                + BlancoDbMetaDataUtil
                                        .convertJdbcDataTypeToString(columnStructure
                                                .getDataType())
                                + ")̃oCh: łȂSQL̓p[^̌^("
                                + columnStructure.getDataType()
                                + "/"
                                + BlancoDbMetaDataUtil
                                        .convertJdbcDataTypeToString(columnStructure
                                                .getDataType()) + ")w肳܂B");
            }
        }
    }
}
