/*
 * blancoDb
 * Copyright (C) 2004-2005 Yasuo Nakanishi
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package blanco.db.collector;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Iterator;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.dom.DOMResult;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import blanco.commons.util.BlancoBigDecimalUtil;
import blanco.commons.util.BlancoStringUtil;
import blanco.commons.util.BlancoXmlUtil;
import blanco.db.conf.AbstractQuery;
import blanco.db.conf.BlancoDbMetadata;
import blanco.db.conf.CallQuery;
import blanco.db.conf.ExecuteQuery;
import blanco.db.conf.SelectQuery;
import blanco.db.definition.BlancoDbDefinition;
import blanco.db.definition.FieldFactory;
import blanco.db.definition.QueryCaller;
import blanco.db.definition.QueryInvoker;
import blanco.db.definition.QueryIterator;
import blanco.db.mapping.BlancoDbMappingUtil;
import blanco.db.resourcebundle.BlancoDbResourceBundle;
import blanco.db.util.BlancoDbQueryParserUtil;
import blanco.ig.expander.Type;
import blanco.ig.expander.Value;

/**
 * @author Yasuo Nakanishi
 */
public class BlancoDbCollector {
    /**
     * fobO[hƂē삳邩ǂBʏfalseŉ^p܂B
     */
    private static final boolean IS_DEBUG = false;

    /**
     * blancoDb\[Xohւ̃ANZTB
     */
    private final BlancoDbResourceBundle bundle = new BlancoDbResourceBundle();

    /**
     * ʏ񂪒~ĂGgB
     */
    private static final String ELEMENT_COMMON = "blancodb-common";

    /**
     * SQL̓p[^~ĂGgB
     */
    private static final String ELEMENT_INPARAMETERS = "blancodb-inparameters";

    /**
     * SQLo̓p[^~ĂGgB
     */
    private static final String ELEMENT_OUTPARAMETERS = "blancodb-outparameters";

    /**
     * SQL~ĂGgB
     */
    private static final String ELEMENT_QUERY = "blancodb-query";

    /**
     * SQL`̒XML͂ɁASQL`W܂B
     * 
     * @param metadata
     *            XMLt@CB
     * @param conn
     *            blancof[^x[XڑIuWFNgB
     * @return blancoDb`B
     * @throws SQLException
     *             SQLOꍇB
     * @throws SAXException
     *             SAXOꍇB
     * @throws IOException
     *             o͗OꍇB
     * @throws ParserConfigurationException
     *             p[TRtBO[VOꍇB
     * @throws TransformerException
     *             XMLϊOꍇB
     */
    public BlancoDbDefinition collect(final BlancoDbMetadata metadata,
            final BlancoDbDatabaseConnection conn) throws SQLException,
            SAXException, IOException, ParserConfigurationException,
            TransformerException {
        // WblancoDb`ɂďs܂B
        final BlancoDbDefinition blancoDbDef = new BlancoDbDefinition("NoName");
        blancoDbDef.setIteratorList(new ArrayList());
        blancoDbDef.setInvokerList(new ArrayList());
        blancoDbDef.setCallerList(new ArrayList());

        // XMLt@CDOMƂăp[X܂B
        InputStream inStream = null;
        final DOMResult result;
        try {
            inStream = new BufferedInputStream(new FileInputStream(metadata
                    .getXml()));
            result = BlancoXmlUtil.transformStream2Dom(inStream);
        } finally {
            inStream.close();
        }

        // [gm[h擾܂B
        final Node nodeRootNode = result.getNode();
        if (nodeRootNode == null) {
            if (IS_DEBUG) {
                System.out.println("TRACE: XMLt@Cł͂܂B");
            }
            return blancoDbDef;
        }

        final Element eleWorkbook = BlancoXmlUtil.getElement(nodeRootNode,
                "workbook");
        if (eleWorkbook == null) {
            return blancoDbDef;
        }

        // V[gGgWJ܂B
        final NodeList listSheet = eleWorkbook.getElementsByTagName("sheet");
        if (listSheet == null) {
            if (IS_DEBUG) {
                System.out.println("TRACE: V[g܂B");
            }
            return blancoDbDef;
        }

        // ̂̂̃V[gGgɂďs܂B
        final int nodeLength = listSheet.getLength();
        for (int index = 0; index < nodeLength; index++) {
            final Node nodeSheet = listSheet.item(index);
            if (nodeSheet instanceof Element == false) {
                continue;
            }
            final Element eleSheet = (Element) nodeSheet;
            expandSheet(eleSheet, conn, blancoDbDef);
        }

        return blancoDbDef;
    }

    /**
     * ^ꂽV[gGgWJ܂B
     * 
     * @param eleSheet
     *            V[gGgB
     * @param conn
     *            blancof[^x[XڑIuWFNgB
     * @param blancoDbDef
     *            blancoDb`B
     */
    private void expandSheet(final Element eleSheet,
            final BlancoDbDatabaseConnection conn,
            final BlancoDbDefinition blancoDbDef) {
        // ŏɋʏWJ܂B
        final AbstractQuery abstractQuery = expandCommon(eleSheet);
        if (abstractQuery == null) {
            // SQL`ƂĂӂ킵Ȃ̂ŁAXLbv܂B
            if (IS_DEBUG) {
                System.out.println("TRACE: SQL`ƂĂ͂ӂ킵ȂƔf܂B");
            }
            return;
        }

        // SQL̓p[^WJ܂B
        expandInParameter(eleSheet, abstractQuery);

        // SQLo̓p[^WJ܂B
        expandOutParameter(eleSheet, abstractQuery);

        // SQLWJ܂B
        expandQuery(eleSheet, abstractQuery);

        // ЂƂƂ̏낦ŁAW̑Ó`FbNȂ܂B
        if (abstractQuery.getQuery() == null
                || abstractQuery.getQuery().length() == 0) {
            // SQL擾łȂ̂̓G[܂B
            throw new IllegalArgumentException(bundle
                    .getXml2javaclassErr001(abstractQuery.getName()));
        }

        // W Iterator, Invoker, CallerƂNX̃IuWFNgւƋlߑւ܂B
        if (abstractQuery instanceof SelectQuery) {
            try {
                // QueryIteratoȑꍇɂ́Af[^x[XɐڑďW܂B
                getQueryFields(conn.getConnection(),
                        (SelectQuery) abstractQuery);
            } catch (SQLException e) {
                throw new IllegalArgumentException(bundle
                        .getXml2javaclassErr002(abstractQuery.getName(), e
                                .getSQLState(), BlancoBigDecimalUtil
                                .toBigDecimal(e.getErrorCode()), e.toString()));
            }
            blancoDbDef.addQueryIterator(new QueryIterator(
                    (SelectQuery) abstractQuery));
        } else if (abstractQuery instanceof ExecuteQuery) {
            blancoDbDef.addQueryInvoker(new QueryInvoker(
                    (ExecuteQuery) abstractQuery));
        } else if (abstractQuery instanceof CallQuery) {
            blancoDbDef.addQueryCaller(new QueryCaller(
                    (CallQuery) abstractQuery));
        } else {
            throw new IllegalArgumentException(bundle.getXml2javaclassErr003(
                    abstractQuery.toString(), abstractQuery.getClass()
                            .toString()));
        }

        // ŌblancoDb`ɑ΂ĎW̏ǉ܂B
        if (abstractQuery instanceof SelectQuery) {
            blancoDbDef.addQueryIterator(new QueryIterator(
                    (SelectQuery) abstractQuery));
        } else if (abstractQuery instanceof ExecuteQuery) {
            blancoDbDef.addQueryInvoker(new QueryInvoker(
                    (ExecuteQuery) abstractQuery));
        } else if (abstractQuery instanceof CallQuery) {
            blancoDbDef.addQueryCaller(new QueryCaller(
                    (CallQuery) abstractQuery));
        }
    }

    /**
     * ^ꂽʃGg͂ďWJ܂B
     * 
     * @param elementSheet
     *            V[gIuWFNgB
     * @return ۃNGIuWFNgB
     */
    private AbstractQuery expandCommon(final Element eleSheet) {
        final Element elementCommon = BlancoXmlUtil.getElement(eleSheet,
                ELEMENT_COMMON);
        if (elementCommon == null) {
            // ELEMENT_COMMONȂꍇɂ́ÃV[gXLbv܂B
            if (IS_DEBUG) {
                System.out.println("TRACE: " + ELEMENT_COMMON + "܂B");
            }

            return null;
        }

        final String name = BlancoStringUtil.null2Blank(BlancoXmlUtil
                .getTextContent(elementCommon, "name"));
        if (name.length() == 0) {
            // ̒`͏̕Kv܂BȂƂƂ܂B
            // ]query-typew̏ꍇɂXLbvĂ܂A݂̓G[Ƃ܂B
            if (IS_DEBUG) {
                System.out.println("TRACE: " + ELEMENT_COMMON
                        + "name܂B");
            }
            return null;
        }

        final String description = BlancoStringUtil.null2Blank(BlancoXmlUtil
                .getTextContent(elementCommon, "description"));
        final String queryType = BlancoStringUtil.null2Blank(BlancoXmlUtil
                .getTextContent(elementCommon, "query-type"));
        final boolean isSingle = BlancoStringUtil.null2Blank(
                BlancoXmlUtil.getTextContent(elementCommon, "single")).equals(
                "true");
        final String scroll = BlancoStringUtil.null2Blank(BlancoXmlUtil
                .getTextContent(elementCommon, "scroll"));
        final boolean updatable = BlancoStringUtil.null2Blank(
                BlancoXmlUtil.getTextContent(elementCommon, "updatable"))
                .equals("true");

        // NG^ɂ킹ăIuWFNgVK쐬܂B
        AbstractQuery abstractQuery = null;
        if (queryType.equals("iterator")) {
            abstractQuery = new SelectQuery(name);
            ((SelectQuery) abstractQuery).setScrollInterface(scroll);
            ((SelectQuery) abstractQuery)
                    .setEnableUpdatableInterface(updatable);
        } else if (queryType.equals("invoker")) {
            abstractQuery = new ExecuteQuery(name);
        } else if (queryType.equals("caller")) {
            abstractQuery = new CallQuery(name);
        } else {
            // ł܂Bf܂B
            throw new IllegalArgumentException("T|[gȂNG^Cv[" + queryType
                    + "]^܂Bf܂B");
        }

        // NG^ʂċʂ̏ݒ肵܂B
        abstractQuery.setDescription(description);
        abstractQuery.setSingle(isSingle);

        return abstractQuery;
    }

    /**
     * ^ꂽV[g͂ďWJ܂B
     * 
     * @param elementSheet
     *            V[gIuWFNgB
     * @param abstractQuery
     *            ۃNGIuWFNgB
     */
    private void expandInParameter(final Element elementSheet,
            final AbstractQuery abstractQuery) {
        final Element elementBlancoDbInparameters = BlancoXmlUtil.getElement(
                elementSheet, ELEMENT_INPARAMETERS);
        if (elementBlancoDbInparameters == null) {
            return;
        }

        final NodeList nodeList = elementBlancoDbInparameters
                .getElementsByTagName("inparameter");
        if (nodeList == null) {
            if (IS_DEBUG) {
                System.out.println("TRACE: SQL̓p[^͂܂B");
            }
            return;
        }
        final int nodeLength = nodeList.getLength();
        for (int index = 0; index < nodeLength; index++) {
            final Node nodeLook = nodeList.item(index);
            if (nodeLook instanceof Element == false) {
                continue;
            }
            final String no = BlancoXmlUtil.getTextContent((Element) nodeLook,
                    "no");
            final String name = BlancoXmlUtil.getTextContent(
                    (Element) nodeLook, "name");
            final String type = BlancoXmlUtil.getTextContent(
                    (Element) nodeLook, "type");
            // ݁Adescription͊i[悪܂B
            // final String description = BlancoXmlUtil.getTextContent(
            // (Element) nodeLook, "description");

            final String paramNoString = (no == null ? "" : " No.[" + no + "] ");
            if (name == null || name.length() == 0) {
                throw new IllegalArgumentException(bundle
                        .getXml2javaclassErr004(abstractQuery.getName(),
                                paramNoString, type));
            }
            if (type == null || type.length() == 0) {
                throw new IllegalArgumentException(bundle
                        .getXml2javaclassErr005(abstractQuery.getName(),
                                paramNoString, name));
            }

            final Value value = new Value(new Type(type), name);
            abstractQuery.addInParameter(value);
        }
    }

    /**
     * ^ꂽV[g͂ďWJ܂B
     * 
     * @param elementSheet
     *            V[gIuWFNgB
     * @param abstractQuery
     *            ۃNGIuWFNgB
     */
    private void expandOutParameter(final Element elementSheet,
            final AbstractQuery abstractQuery) {
        final Element elementBlancoDbOutparameters = BlancoXmlUtil.getElement(
                elementSheet, ELEMENT_OUTPARAMETERS);
        if (elementBlancoDbOutparameters == null) {
            if (IS_DEBUG) {
                System.out.println("TRACE: SQLo̓p[^͂܂B");
            }
            return;
        }

        final NodeList nodeList = elementBlancoDbOutparameters
                .getElementsByTagName("outparameter");
        if (nodeList == null) {
            return;
        }
        final int nodeLength = nodeList.getLength();
        for (int index = 0; index < nodeLength; index++) {
            final Node nodeLook = nodeList.item(index);
            if (nodeLook instanceof Element == false) {
                continue;
            }
            final String no = BlancoXmlUtil.getTextContent((Element) nodeLook,
                    "no");
            final String name = BlancoXmlUtil.getTextContent(
                    (Element) nodeLook, "name");
            final String type = BlancoXmlUtil.getTextContent(
                    (Element) nodeLook, "type");
            // ݁Adescription͊i[悪܂B
            // final String description = BlancoXmlUtil.getTextContent(
            // (Element) nodeLook, "description");

            final String paramNoString = (no == null ? "" : " No.[" + no + "] ");
            if (name == null || name.length() == 0) {
                throw new IllegalArgumentException(bundle
                        .getXml2javaclassErr006(abstractQuery.getName(),
                                paramNoString, type));
            }
            if (type == null || type.length() == 0) {
                throw new IllegalArgumentException(bundle
                        .getXml2javaclassErr007(abstractQuery.getName(),
                                paramNoString, name));
            }
            if (abstractQuery instanceof CallQuery == false) {
                throw new IllegalArgumentException(bundle
                        .getXml2javaclassErr008(abstractQuery.getName(),
                                paramNoString, name));
            }

            final Value value = new Value(new Type(type), name);
            ((CallQuery) abstractQuery).addOutParameter(value);
        }
    }

    /**
     * ^ꂽV[g͂ďWJ܂B
     * 
     * @param elementSheet
     *            V[gIuWFNgB
     * @param abstractQuery
     *            ۃNGIuWFNgB
     */
    private void expandQuery(final Element elementSheet,
            final AbstractQuery abstractQuery) {
        final Element elementBlancoDbInparameters = BlancoXmlUtil.getElement(
                elementSheet, ELEMENT_QUERY);
        if (elementBlancoDbInparameters == null) {
            return;
        }

        final NodeList nodeList = elementBlancoDbInparameters
                .getElementsByTagName("query-line");
        if (nodeList == null) {
            return;
        }
        final int nodeLength = nodeList.getLength();
        for (int index = 0; index < nodeLength; index++) {
            final String queryLine = BlancoStringUtil.null2Blank(BlancoXmlUtil
                    .getTextContent(nodeList.item(index)));

            String query = abstractQuery.getQuery();
            if (query == null || query.length() == 0) {
                query = "";
            } else {
                // 񂪂łɑ݂Ăꍇɂ̂݉st^܂B
                query = query + "\n";
            }
            abstractQuery.setQuery(query + queryLine);
        }
    }

    /**
     * SQLsʂ̌ʗꗗ擾āAL܂B
     * 
     * JDBC̋@\𗘗pSQLłAʂ̗ꗗ擾ƂAblancoDb̊jƂ@\̎ӏłB
     * 
     * @param conn
     *            blancof[^x[XڑIuWFNgB
     * @param selectQuery
     *            QueryIterator̂߂̏B
     * @throws SQLException
     *             SQLOꍇB
     */
    private final void getQueryFields(final Connection conn,
            final SelectQuery selectQuery) throws SQLException {
        // SQLp[X邽߂̃[eBeBB
        final BlancoDbQueryParserUtil parserUtil = new BlancoDbQueryParserUtil(
                selectQuery.getQuery());

        PreparedStatement statement = null;
        ResultSet resultSet = null;

        final FieldFactory fieldFactory = new FieldFactory();
        try {
            statement = conn.prepareStatement(parserUtil.getNaturalSqlString());

            // SQL̓p[^ւ̃oCh{܂B
            // 2005.05.20 t.iga
            for (Iterator iteParameter = selectQuery.getInParameterIterator(); iteParameter
                    .hasNext();) {
                final Value val = (Value) iteParameter.next();
                int[] listCol = parserUtil.getSqlParameters(val.getName());
                if (listCol == null) {
                    throw new IllegalArgumentException(bundle
                            .getXml2javaclassErr009(selectQuery.getName(), val
                                    .getName()));
                }
                for (int iteSame = 0; iteSame < listCol.length; iteSame++) {
                    final int index = listCol[iteSame];
                    BlancoDbMappingUtil.processPreparedStatementWithSomeValue(
                            val.getType().getName(), statement, index);
                }
            }

            resultSet = statement.executeQuery();

            final ResultSetMetaData meta = resultSet.getMetaData();
            for (int i = 0; i < meta.getColumnCount(); i++) {
                selectQuery
                        .addField(fieldFactory.createQueryField(i + 1, meta));
            }
        } finally {
            if (resultSet != null) {
                resultSet.close();
            }
            if (statement != null) {
                statement.close();
            }
        }
    }
}