%feature("docstring") OT::VertexValueFunction
"Function mapping a field to a field.

Parameters
----------
g : :class:`~openturns.Function`
    Function :math:`g: \\Rset^n \\times \\Rset^d \\rightarrow \\Rset^{d'}`.
mesh : :class:`~openturns.Mesh`
    Mesh on which the function is defined.

Notes
-----
A VertexValueFunction acts on fields to produce fields:

.. math::

    f: \\left| \\begin{array}{rcl}
               \\cM_N \\times (\\Rset^d)^N & \\rightarrow & \\cM_N \\times (\\Rset^{d'})^{N} \\\\
                F & \\mapsto & F'
              \\end{array} \\right.

with :math:`F = (\\vect{t}_i, \\vect{v}_i)_{1 \\leq i \\leq N}`, :math:`F' = (\\vect{t}_i, \\vect{v}'_i)_{1 \\leq i \\leq N}` and :math:`\\cM_{N}` a mesh of :math:`\\cD \\subset \\Rset^{n}`.

The VertexValueFunction keeps the mesh unchanged: the input mesh is also the output mesh.

The field :math:`F'` is defined by the function :math:`g: \\Rset^n \\times \\Rset^d \\rightarrow \\Rset^{d'}`:

.. math::

   \\forall 1 \\leq i \\leq N, \\quad \\vect{v}'_i = g(\\vect{t}_i, \\vect{v}_i)

When :math:`g` is not specified, the constructor builds an object which evaluation operator is not defined (it throws a *NotYetImplementedException*).
The instantiation of such an object is used to extract an actual :class:`~openturns.VertexValueFunction` from a :class:`~openturns.Study`.

See also
--------
FieldFunction, ValueFunction, VertexValueFunction

Examples
--------
Create a function :math:`g : \\Rset \\times \\Rset \\rightarrow \\Rset` such as:

.. math::

    g: \\left|\\begin{array}{rcl}
                \\Rset \\times \\Rset & \\rightarrow & \\Rset \\\\
                (t, x) & \\mapsto & (x + t^2)
            \\end{array}\\right.

>>> import openturns as ot
>>> g = ot.SymbolicFunction(['t', 'x'], ['x + t^2'])

Convert :math:`g` into a vertex value function with :math:`n=1`:

>>> n = 1
>>> grid = ot.RegularGrid(0.0, 0.2, 6)
>>> myVertexValueFunction = ot.VertexValueFunction(g, grid)
>>> # Create a TimeSeries
>>> data = ot.Sample(grid.getN(), g.getInputDimension()-1)
>>> for i in range(data.getSize()):
...     for j in range(data.getDimension()):
...         data[i, j] = i * data.getDimension() + j
>>> ts = ot.TimeSeries(grid, data)
>>> print(ts)
    [ t   v0  ]
0 : [ 0   0   ]
1 : [ 0.2 1   ]
2 : [ 0.4 2   ]
3 : [ 0.6 3   ]
4 : [ 0.8 4   ]
5 : [ 1   5   ]
>>> print(myVertexValueFunction(ts))
    [ y0   ]
0 : [ 0    ]
1 : [ 1.04 ]
2 : [ 2.16 ]
3 : [ 3.36 ]
4 : [ 4.64 ]
5 : [ 6    ]"

// ---------------------------------------------------------------------

%feature("docstring") OT::VertexValueFunction::getFunction
"Get the function of :math:`g`.

Returns
-------
l : :class:`~openturns.Function`
   Function :math:`g: \\Rset^n \\times \\Rset^d \\rightarrow \\Rset^{d'}`.

Examples
--------
>>> import openturns as ot
>>> g = ot.SymbolicFunction(['t', 'x'], ['x + t^2'])
>>> n = 1
>>> mesh = ot.Mesh(n)
>>> myVertexValueFunction = ot.ValueFunction(g, mesh)
>>> print(myVertexValueFunction.getFunction())
[t,x]->[x + t^2]"

// ---------------------------------------------------------------------

%feature("docstring") OT::VertexValueFunction::getMarginal
"Get the marginal(s) at given indice(s).

Parameters
----------
i : int or list of ints, :math:`0 \\leq i < d'`
    Indice(s) of the marginal(s) to be extracted.

Returns
-------
function : :class:`~openturns.VertexValueFunction`
    The initial function restricted to the concerned marginal(s) at the indice(s)
    :math:`i`."


// ---------------------------------------------------------------------

%feature("docstring") OT::VertexValueFunction::getOutputMesh
"Get the mesh associated to the output domain.

Returns
-------
outputMesh : :class:`~openturns.Mesh`
    The output mesh which is also equal to the input one: :math:`\\cM_{N}`."





