%feature("docstring") OT::Normal
"Normal distribution.

Its probability density function is defined as:

.. math::

    f_{\\vect{X}}(\\vect{x}) = \\frac{1}
             {\\displaystyle (2\\pi)^{\\frac{n}{2}}
                            (\\mathrm{det}\\mat{\\Sigma})^{\\frac{1}{2}}}
             \\displaystyle exp \\left(-\\frac{1}{2}\\Tr{(\\vect{x} - \\vect{\\mu})}
                                     \\mat{\\Sigma}^{-1}(\\vect{x} - \\vect{\\mu})\\right)

with :math:`\\mat{\\Sigma} = \\diag(\\vect{\\sigma}) \\mat{R} \\diag(\\vect{\\sigma})`.
Both :math:`\\mat{R}` and :math:`\\mat{\\Sigma}` are symmetric positive definite.

The first moments of the distribution are:

.. math::
    :nowrap:

    \\begin{eqnarray*}
        \\Expect{\\vect{X}} & = & \\vect{\\mu} \\\\
        \\Var{\\vect{X}} & = & \\mat{\\Sigma}
    \\end{eqnarray*}

Available constructors:
    Normal(*d*)

    Normal(*mu, sigma*)

    Normal(*mu, sigma, R*)

    Normal(*mu, Sigma*)

Parameters
----------
d : int, :math:`d \\geq 1`
    Dimension.
    
    Default value is 1.

mu : float :math:`\\mu`, sequence of float, :math:`\\vect{\\mu} \\in \\Rset^d`
    Mean.

    Default value is 0.0.

sigma : float :math:`\\sigma`, sequence of float, :math:`\\vect{\\sigma} \\in \\Rset^{*d}_+`
    Scale parameter.

    Default value is 1.0.

R : :class:`~openturns.CorrelationMatrix` :math:`\\mat{R} \\in  \\cM_{d \\times d}(\\Rset)`
    Correlation matrix.

    Default value is the identity matrix.

Sigma : :class:`~openturns.CovarianceMatrix` :math:`\\mat{\\Sigma} \\in  \\cM_{d \\times d}(\\Rset)`
    Covariance matrix.

Notes
-----
In the first usage, a scalar centered and unit-variance normal distribution is built.

The CDF is computed using several algorithms, depending on the dimension of the distribution:

- In dimension 1, it is done using the implementation of the complementary
  error function developed by Steven G. Johnson, Massachusetts Institute of
  Technology;
- In any dimension, it is done as the product of 1D marginal CDFs if the
  components are independent;
- In dimension 2 and 3, it is done using an integration in dimension 1 or 2 of
  a dedicated kernel;
- In dimension 4 to the value given by the entry *Normal-SmallDimension* of
  :class:`~openturns.ResourceMap`, it is done using the adaptive integration
  algorithm :class:`~openturns.IteratedQuadrature`.
- In dimension greater than the entry *Normal-SmallDimension* of
  :class:`~openturns.ResourceMap`, it is done using a Monte Carlo estimation
  controlled by the *Normal-MinimumCDFEpsilon*, *Normal-MinimumNumberOfPoints*
  and *Normal-MaximumNumberOfPoints* entries of :class:`~openturns.ResourceMap`.


Examples
--------
Create a distribution:

>>> import openturns as ot
>>> distribution = ot.Normal(0.0, 1.0)

Draw a sample:

>>> sample = distribution.getSample(5)

Create a multivariate normal distribution with constant :math:`\\sigma` and identity correlation matrix:

>>> dimension = 5
>>> mu = ot.Point(dimension)
>>> sigma = ot.Point(dimension, 0.1)
>>> R = ot.CorrelationMatrix(dimension)
>>> distribution = ot.Normal(mu, sigma, R)
>>> print('distribution=', distribution)
distribution= Normal(mu = [0,0,0,0,0], sigma = [0.1,0.1,0.1,0.1,0.1], R = 5x5
[[ 1 0 0 0 0 ]
 [ 0 1 0 0 0 ]
 [ 0 0 1 0 0 ]
 [ 0 0 0 1 0 ]
 [ 0 0 0 0 1 ]])

Create a multivariate normal distribution from a covariance matrix:

>>> Sigma = ot.CovarianceMatrix(dimension)
>>> Sigma[1, 1] = 3.0
>>> Sigma[0, 2] = -0.5
>>> Sigma[3, 4] = -0.2
>>> distribution = ot.Normal(mu, Sigma)
>>> print('distribution=', distribution)
distribution= Normal(mu = [0,0,0,0,0], sigma = [1,1.73205,1,1,1], R = 5x5
[[  1    0   -0.5  0    0   ]
 [  0    1    0    0    0   ]
 [ -0.5  0    1    0    0   ]
 [  0    0    0    1   -0.2 ]
 [  0    0    0   -0.2  1   ]])"
